@file:Suppress("NOTHING_TO_INLINE")

package dev.kikugie.commons.serialization

import kotlinx.serialization.KSerializer
import kotlinx.serialization.descriptors.PrimitiveKind
import kotlinx.serialization.descriptors.PrimitiveSerialDescriptor
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder

/**
 * Serializes [IntRange] in their closer representation: `6..9 // -> "6..9"`.
 * @sample dev.kikugie.commons_samples.Serialization.rangeSerialization
 */
public object IntRangeSerializer : KSerializer<IntRange> {
    override val descriptor: SerialDescriptor = PrimitiveSerialDescriptor("kotlin.ranges.IntRange", PrimitiveKind.STRING)

    override fun serialize(encoder: Encoder, value: IntRange) {
        encoder.encodeString("${value.first}..${value.last}")
    }

    override fun deserialize(decoder: Decoder): IntRange {
        val parts = decoder.decodeString().split("..", limit = 2)
        check(parts.size == 2) { "Missing range delimiter" }
        val (start, end) = parts
        return if (end.startsWith('<')) start.toInt()..<end.drop(1).toInt()
        else start.toInt()..end.toInt()
    }
}

/**
 * Serializes [LongRange] in their closer representation: `6L..9L // -> "6..9"`.
 * @sample dev.kikugie.commons_samples.Serialization.rangeSerialization
 */
public object LongRangeSerializer : KSerializer<LongRange> {
    override val descriptor: SerialDescriptor = PrimitiveSerialDescriptor("kotlin.ranges.LongRange", PrimitiveKind.STRING)
    override fun serialize(encoder: Encoder, value: LongRange) {
        encoder.encodeString("${value.first}..${value.last}")
    }
    override fun deserialize(decoder: Decoder): LongRange {
        val parts = decoder.decodeString().split("..", limit = 2)
        check(parts.size == 2) { "Missing range delimiter" }
        val (start, end) = parts
        return if (end.startsWith('<')) start.toLong()..<end.drop(1).toLong()
        else start.toLong()..end.toLong()
    }
}