package dev.kikugie.commons.text

/**
 * Gets the character at the given [index],
 * or returns the [default] value if it's out of bounds.
 * The [default] is `' '` (single whitespace) when unspecified.
 *
 * *This can be used in favour of [getOrNull] in when-cases,
 * since the Kotlin compiler turns nullable ones into if-else chains
 * instead of a switch table.*
 *
 * @sample dev.kikugie.commons_samples.Strings.getOrDefault
 */
public fun CharSequence.getOrDefault(index: Int, default: Char = ' '): Char =
    getOrElse(index) { default }

/**
 * Counts continuous characters in [this] string that are present in the [chars] array.
 * @sample dev.kikugie.commons_samples.Strings.countMatching
 */
@JvmName("countMatchingFull")
public fun CharSequence.countMatching(vararg chars: Char): Int =
    countWhile { it in chars }

/**
 * Counts continuous characters in [this] string that are present in the [chars] array.
 * The count goes withing the given [range], which is coerced to the string boundaries.
 * @sample dev.kikugie.commons_samples.Strings.countMatching
 */
public fun CharSequence.countMatching(range: IntRange, vararg chars: Char): Int =
    countWhile(range) { it in chars }

/**
 * Counts continuous characters in [this] string that are present in the [chars] array.
 * The count goes from [start] to [end] (exclusive),
 * which cover the full string by default and coerce to its boundaries.
 * @sample dev.kikugie.commons_samples.Strings.countMatching
 */
public fun CharSequence.countMatching(start: Int = 0, end: Int = length, vararg chars: Char): Int =
    countWhile(start, end) { it in chars }

/**
 * Counts continuous characters in [this] string that match the given [predicate].
 * @sample dev.kikugie.commons_samples.Strings.countWhile
 */
@JvmName("countWhileFull")
public inline fun CharSequence.countWhile(predicate: (Char) -> Boolean): Int =
    countWhile(0, length, predicate)

/**
 * Counts continuous characters in [this] string that match the given [predicate].
 * The count goes withing the given [range], which is coerced to the string boundaries.
 * @sample dev.kikugie.commons_samples.Strings.countWhile
 */
public inline fun CharSequence.countWhile(range: IntRange, predicate: (Char) -> Boolean): Int =
    countWhile(range.first, range.last + 1, predicate)

/**
 * Counts continuous characters in [this] string that match the given [predicate].
 * The count goes from [start] to [end] (exclusive),
 * which cover the full string by default and coerce to its boundaries.
 * @sample dev.kikugie.commons_samples.Strings.countWhile
 */
public inline fun CharSequence.countWhile(start: Int = 0, end: Int = length, predicate: (Char) -> Boolean): Int {
    var count = 0
    for (i in start.coerceAtLeast(0) ..< end.coerceAtMost(length))
        if (predicate(this[i])) count++
        else break
    return count
}

/**
 * Creates a view for [this] string with all characters in the reverse order.
 * Any changes in the original string will be reflected in the view.
 *
 * **Note**: calling [subSequence] will create an immutable substring
 * of the reversed segment.
 * @sample dev.kikugie.commons_samples.Strings.reverseView
 */
public fun CharSequence.reverseView(): CharSequence = object : CharSequence {
    override val length: Int get() = this@reverseView.length
    override fun get(index: Int): Char = this@reverseView[length - index - 1]
    override fun subSequence(startIndex: Int, endIndex: Int): CharSequence =
        this@reverseView.substring(length - endIndex, length - startIndex).reversed()
    override fun toString(): String = this@reverseView.toString().reversed()
}
