package dev.kikugie.commons.collections

import java.util.Queue

/**
 * A strict queue with the given [capacity]. The differences from [java.util.ArrayDeque] are:
 * - The underlying array is not resizeable.
 * - Removal of arbitrary elements is not supported. Elements must be consumed one by one.
 * - Reduced operation overhead for frequent insertions and removals.
 * @sample dev.kikugie.commons_samples.Collections.fixedQueue
 */
@Suppress("UNCHECKED_CAST")
public class FixedQueue<T>(public val capacity: Int) : Queue<T> {
    private val array: Array<Any?> = arrayOfNulls(capacity)
    private var head = 0
    override var size: Int = 0
        private set

    public operator fun get(index: Int): T =
        if (index < 0 || index >= size) throw IndexOutOfBoundsException("Index $index out of bounds for length $size")
        else array[(head + index) % capacity] as T

    override fun toString(): String = "FixedQueue(size=$size, capacity=$capacity, values=${array.contentToString()})"
    override fun isEmpty(): Boolean = size == 0
    override fun contains(element: T?): Boolean = array.contains(element)
    override fun containsAll(elements: Collection<T?>): Boolean = elements.all(::contains)
    override fun clear() {
        array.fill(null)
        head = 0
        size = 0
    }

    override fun add(e: T?): Boolean = when {
        size >= capacity -> throw IndexOutOfBoundsException("Queue is full")
        else -> true.also {
            array[(head + size++) % capacity] = e
        }
    }

    override fun offer(e: T?): Boolean = when {
        size >= capacity -> false
        else -> true.also {
            array[(head + size++) % capacity] = e
        }
    }

    override fun remove(): T = when {
        isEmpty() -> throw NoSuchElementException("Queue is empty")
        else -> (array[head] as T).also {
            array[head] = null
            head = (head + 1) % capacity
            --size
        }
    }

    override fun poll(): T? = when {
        isEmpty() -> null
        else -> (array[head] as T).also {
            array[head] = null
            head = (head + 1) % capacity
            --size
        }
    }

    override fun element(): T? = when {
        isEmpty() -> throw NoSuchElementException("Queue is empty")
        else -> array[head] as T
    }

    override fun peek(): T? = when {
        isEmpty() -> null
        else -> array[head] as T
    }

    override fun addAll(elements: Collection<T?>): Boolean = true.also {
        if (size + elements.size > capacity) throw IndexOutOfBoundsException("Not enough space in queue")
        elements.forEach(::add)
    }

    override fun iterator(): FixedQueueIterator = FixedQueueIterator()

    @Deprecated("Unsupported operation", level = DeprecationLevel.ERROR)
    override fun remove(element: T?): Boolean {
        throw UnsupportedOperationException("Removal of arbitrary elements is not supported")
    }

    @Deprecated("Unsupported operation", level = DeprecationLevel.ERROR)
    override fun removeAll(elements: Collection<T?>): Boolean {
        throw UnsupportedOperationException("Removal of arbitrary elements is not supported")
    }

    @Deprecated("Unsupported operation", level = DeprecationLevel.ERROR)
    override fun retainAll(elements: Collection<T?>): Boolean {
        throw UnsupportedOperationException("Removal of arbitrary elements is not supported")
    }

    public inner class FixedQueueIterator : MutableIterator<T> {
        private var index = 0
        override fun hasNext(): Boolean = isNotEmpty() && index < size
        override fun next(): T =
            if (hasNext()) array[(head + index++) % capacity] as T
            else throw NoSuchElementException("Finished iterating queue")

        @Deprecated("Unsupported operation", level = DeprecationLevel.ERROR)
        override fun remove() {
            throw UnsupportedOperationException("Removal of arbitrary elements is not supported")
        }
    }
}