@file:Suppress("NOTHING_TO_INLINE", "UNCHECKED_CAST")
package dev.kikugie.commons.collections

import dev.kikugie.commons.applyIf
/**Returns an element at the given [index] or the [default] if the [index] is out of bounds.*/
public inline fun <T> List<T>.getOrDefault(index: Int, default: T): T =
    getOrElse(index) { default }

/**Returns an element at the given [index] or throws [IndexOutOfBoundsException] with the [message] if the [index] is out of bounds.*/
public inline fun <T> List<T>.getOrThrow(index: Int, message: List<T>.(Int) -> String): T =
    getOrElse(index) { throw IndexOutOfBoundsException(message(index)) }

/**Returns the [Result] of [List.get] call.*/
public inline fun <T> List<T>.getResult(index: Int): Result<T> =
    runCatching { get(index) }

/**Returns the [Result] of [List.getOrThrow] call.*/
public inline fun <T> List<T>.getResult(index: Int, message: List<T>.(Int) -> String): Result<T> =
    runCatching { getOrThrow(index, message) }

/**Returns the last non-null element produced by [transform] function or `null` otherwise.
* The element is found by iterating the collection in reverse order.*/
public inline fun <T, R : Any> List<T>.lastNotNullOfOrNull(transform: (T) -> R?): R? {
    for (index in size - 1 downTo 0) {
        val element = transform(get(index))
        if (element != null) return element
    }
    return null
}

/**Returns the last non-null element produced by [transform] function or throws [NoSuchElementException] otherwise.*/
public inline fun <T, R : Any> List<T>.lastNotNullOf(transform: (T) -> R?): R =
    notNullElement(lastNotNullOfOrNull(transform)) { "No element was transformed to a non-null value." }

/**Returns the last non-null element or `null` otherwise.*/
public fun <T : Any> List<T>.lastNotNullOrNull(): T? =
    lastNotNullOfOrNull { it }

/**Returns the last non-null element or throws [NoSuchElementException] otherwise.*/
public fun <T : Any> List<T>.lastNotNull(): T =
    notNullElement(lastNotNullOrNull()) { "No non-null element was found in the list" }

