@file:Suppress("NOTHING_TO_INLINE", "UNCHECKED_CAST")
package dev.kikugie.commons.collections

import dev.kikugie.commons.applyIf
import dev.kikugie.commons.takeAs
import kotlin.reflect.KClass
import kotlin.reflect.cast

@PublishedApi
internal const val MISSING_KEY: String = "Key is not present in the map"

/**Applies [action] to the receiver if it's not empty.*/
public inline fun <K, V, M : Map<K, V>> M.ifNotEmpty(action: M.() -> Unit): M =
    applyIf(isNotEmpty(), action)

/**Returns an element at the given [key] or the [default] if the [key] is not present.*/
public inline fun <K, V> Map<K, V>.getOrDefault(key: K, default: V): V =
    getOrElse(key) { default }

/**Returns an element at the given [key] or throws [NoSuchElementException] with the [message] if the [key] is not present.*/
public inline fun <K, V> Map<K, V>.getOrThrow(key: K, message: Map<K, V>.(K) -> String = { MISSING_KEY }): V =
    getOrElse(key) { throw NoSuchElementException(message(key)) }

/**Returns the [Result] of [Map.getOrThrow] call.*/
public inline fun <K, V> Map<K, V>.getResult(key: K, message: Map<K, V>.(K) -> String = { MISSING_KEY }): Result<V> =
    runCatching { getOrThrow(key, message) }

/**Retrieves the value associated with the specified [key] in the map and casts it to the provided [type].
 * If the key is not present, throws a [NoSuchElementException] with the given [message].*/
public inline fun <K, V> Map<K, *>.getAs(key: K, type: Class<out V>, message: Map<K, *>.(K) -> String = { MISSING_KEY }): V =
    getOrThrow(key, message).let(type::cast)

/**Retrieves the value associated with the specified [key] in the map and casts it to the provided [type].
 * If the key is not present, throws a [NoSuchElementException] with the given [message].*/
public inline fun <K, V> Map<K, *>.getAs(key: K, type: KClass<out V & Any>, message: Map<K, *>.(K) -> String = { MISSING_KEY }): V =
    getOrThrow(key, message).let(type::cast)

/**Retrieves the value associated with the specified [key] in the map and casts it as [V].
 * If the key is not present, throws a [NoSuchElementException] with the given [message].*/
public inline fun <K, reified V> Map<K, *>.getAs(key: K, message: Map<K, *>.(K) -> String = { MISSING_KEY }): V =
    getOrThrow(key, message) as V

/**Retrieves the value associated with the specified [key] in the map, casts it to the provided [type],
 * and wraps the result in a [Result] object. If the key is not present or the cast fails, the exception
 * is encapsulated within the [Result].*/
public inline fun <K, V> Map<K, *>.getAsResult(key: K, type: Class<out V>, message: Map<K, *>.(K) -> String = { MISSING_KEY }): Result<V> =
    runCatching { getAs(key, type, message) }

/**Retrieves the value associated with the specified [key] in the map, casts it to the provided [type],
 * and wraps the result in a [Result] object. If the key is not present or the cast fails, the exception
 * is encapsulated within the [Result].*/
public inline fun <K, V> Map<K, *>.getAsResult(key: K, type: KClass<out V & Any>, message: Map<K, *>.(K) -> String = { MISSING_KEY }): Result<V> =
    runCatching { getAs(key, type, message) }

/**Retrieves the value associated with the specified [key] in the map, casts it as [V],
 * and wraps the result in a [Result] object. If the key is not present or the cast fails, the exception
 * is encapsulated within the [Result].*/
public inline fun <K, reified V> Map<K, *>.getAsResult(key: K, message: Map<K, *>.(K) -> String = { MISSING_KEY }): Result<V> =
    runCatching { getAs(key, message) }