
@file:Suppress("NOTHING_TO_INLINE", "UNCHECKED_CAST")
package dev.kikugie.commons.ranges

import kotlin.math.max
import kotlin.math.min

/**Creates a range starting at [this], spanning the provided [length]. The result is unchecked.*/
public inline infix fun Long.extend(length: Long): LongRange = this..(this + length)

/**Creates a range starting at [this], spanning the provided [length]. The result is unchecked.*/
public inline infix fun Long.extend(length: Int): LongRange = this..(this + length)

/**Shifts [this] range [n] steps to the left. The result is unchecked.*/
public inline infix fun LongRange.shl(n: Long): LongRange = (first - n)..(last - n)

/**Shifts [this] range [n] steps to the left. The result is unchecked.*/
public inline infix fun LongRange.shl(n: Int): LongRange = (first - n)..(last - n)

/**Shifts [this] range [n] steps to the right. The result is unchecked.*/
public inline infix fun LongRange.shr(n: Long): LongRange = (first + n)..(last + n)

/**Shifts [this] range [n] steps to the right. The result is unchecked.*/
public inline infix fun LongRange.shr(n: Int): LongRange = (first + n)..(last + n)

/**Creates a union of [this] and the [other] range,
 * **covering all values between them**.*/
public infix fun LongRange.extend(other: LongRange): LongRange =
    min(first, other.first)..max(last, other.last)

/**Creates a union of [this] and the [other] range **if they overlap**,
 * otherwise returns [LongRange.EMPTY].*/
public infix fun LongRange.merge(other: LongRange): LongRange =
    if (this.overlaps(other)) extend(other)
    else LongRange.EMPTY

/**Creates an intersection between [this] and the [other] range,
 * returning [LongRange.EMPTY] if there's no overlap.*/
public infix fun LongRange.cross(other: LongRange): LongRange =
    if (this.overlaps(other)) max(first, other.first)..min(last, other.last)
    else LongRange.EMPTY

