@file:OptIn(ExperimentalContracts::class)
@file:Suppress("UNCHECKED_CAST", "NOTHING_TO_INLINE")

package dev.kikugie.commons.result

import kotlin.contracts.ExperimentalContracts
import kotlin.contracts.InvocationKind
import kotlin.contracts.contract

/**
 * Converts the given [value] to a [Result],
 * that stores a [NullPointerException] when the [value] is `null`.
 * @sample dev.kikugie.commons_samples.Results.notNullResult
 */
public inline fun <T : Any> notNullResult(value: T?): Result<T> {
    return if (value != null) Result.success(value)
    else Result.failure(NullPointerException())
}

/**
 * Converts the given [value] to a [Result],
 * that stores a [NullPointerException] when the [value] is `null`.
 * @sample dev.kikugie.commons_samples.Results.notNullResult
 */
public inline fun <T : Any> notNullResult(value: T?, message: () -> String): Result<T> {
    contract { callsInPlace(message, InvocationKind.AT_MOST_ONCE) }
    return if (value != null) Result.success(value)
    else Result.failure(NullPointerException(message()))
}

/**
 * Transforms the result exception if it's a failure, otherwise returns itself.
 * @sample dev.kikugie.commons_samples.Results.mapException
 */
public inline fun <T> Result<T>.mapException(transform: (Throwable) -> Throwable): Result<T> {
    contract { callsInPlace(transform, InvocationKind.AT_MOST_ONCE) }
    return exceptionOrNull()?.let { Result.failure(transform(it)) } ?: this
}

/**
 * If the result is successful, replaces it with the value returned by the [transform] operation.
 * @sample dev.kikugie.commons_samples.Results.mapResult
 */
public inline fun <T, R> Result<T>.mapResult(transform: (T) -> Result<R>): Result<R> {
    contract { callsInPlace(transform, InvocationKind.AT_MOST_ONCE) }
    return exceptionOrNull()?.let { this as Result<R> } ?: transform(getOrThrow())
}

public fun <T : Throwable> T.inherit(other: Throwable): T = apply {
    stackTrace = other.stackTrace
    runCatching { initCause(other.cause) }
    other.suppressed.forEach(::addSuppressed)
}