/*
 * Copyright 2021-2024 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package dk.cloudcreate.essentials.components.foundation.postgresql;

import org.jdbi.v3.core.*;
import org.postgresql.*;
import org.slf4j.*;
import reactor.core.publisher.Flux;
import reactor.core.scheduler.Schedulers;

import java.sql.*;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;

import static dk.cloudcreate.essentials.shared.FailFast.*;
import static dk.cloudcreate.essentials.shared.MessageFormatter.NamedArgumentBinding.arg;
import static dk.cloudcreate.essentials.shared.MessageFormatter.*;

/**
 * Helper class for setting up the NOTIFY part of the classical Postgresql LISTEN/NOTIFY concept.<br>
 * See https://jdbc.postgresql.org/documentation/81/listennotify.html and<br>
 * https://www.postgresql.org/docs/current/sql-notify.html
 *
 * @see MultiTableChangeListener
 */
public final class ListenNotify {
    private static final Logger log           = LoggerFactory.getLogger(ListenNotify.class);
    public static final  String TABLE_NAME    = "table_name";
    public static final  String SQL_OPERATION = "sql_operation";

    public enum SqlOperation {INSERT, UPDATE, DELETE, TRUNCATE}

    /**
     * Resolve the Postgresql LISTEN/NOTIFY channel name used to communicate each notification related
     * to the changes that we NOTIFY, based on the {@link #addChangeNotificationTriggerToTable(Handle, String, List, String...)}
     * setup.
     *
     * @param tableName the name of the table
     * @return the channel name
     */
    public static String resolveTableChangeChannelName(String tableName) {
        return requireNonNull(tableName, "No tableName provided");
    }

    /**
     * Create (or replace an existing) Table change notification <code>FUNCTION</code> and an <b>AFTER</b> <code>TRIGGER</code> to the given Table in order to support the classical Postgresql LISTEN/NOTIFY concept.<br>
     * The NOTIFY/LISTEN <b>channel</b> is defined by the {@link #resolveTableChangeChannelName(String)} and the payload is a JSON formatted
     * string containing a {@value #TABLE_NAME} property as well as any additional columns provided in the <code>includeAdditionalTableColumnsInNotificationPayload</code>
     * parameter.<br>
     * <p>
     * You can use {@link #listen(Jdbi, String, Duration)} to setup asynchronous change notification
     * <p>
     * <b>Important information from https://www.postgresql.org/docs/current/sql-notify.html:</b>
     * <blockquote><i>
     * NOTIFY interacts with SQL transactions in some important ways. Firstly, if a NOTIFY is executed inside a transaction, the notify events are not delivered until and unless the transaction is committed.
     * This is appropriate, since if the transaction is aborted, all the commands within it have had no effect, including NOTIFY. But it can be disconcerting if one is expecting the notification events to be delivered immediately.
     * Secondly, if a listening session receives a notification signal while it is within a transaction, the notification event will not be delivered to its connected client until just after the transaction is completed (either committed or aborted).
     * Again, the reasoning is that if a notification were delivered within a transaction that was later aborted, one would want the notification to be undone somehow — but the server cannot “take back” a notification once it has sent it to the client.
     * So notification events are only delivered between transactions. The upshot of this is that applications using NOTIFY for real-time signaling should try to keep their transactions short.<br>
     * <br>
     * If the same channel name is signaled multiple times with identical payload strings within the same transaction, only one instance of the notification event is delivered to listeners. On the other hand, notifications with distinct
     * payload strings will always be delivered as distinct notifications.
     * </i></blockquote>
     * Example:<br>
     * <blockquote>
     * <b>Given</b> this table definition:
     * <pre>CREATE TABLE test_table (id bigint GENERATED BY DEFAULT AS IDENTITY PRIMARY KEY, column1 TEXT, column2 TEXT)</pre>
     * <b>When</b> adding notification trigger:
     * <pre>{@code
     * jdbi.useTransaction(handle -> {
     *             ListenNotify.addChangeNotificationTriggerToTable(handle, "test_table", List.of(ListenNotify.TriggerOnEvent.INSERT), "column1", "column2");
     *         });
     * }</pre>
     * <b>Then</b> the listener subscription:
     * <pre>{@code
     * subscription = ListenNotify.listen(jdbi,
     *                                    "test_table",
     *                                    Duration.ofMillis(200))
     *                            .subscribe(notificationPayload -> {
     *                                // Handle the String based notification payload received
     *                            });
     * }</pre>
     * will for an inserted row:
     * <pre>{@code
     * jdbi.useTransaction(handle -> {
     *    handle.execute("INSERT INTO test_table (column1, column2) VALUES ('Column1Value', 'Column2Value')");
     * });
     * }</pre>
     * receive this payload:<br>
     * <pre>{"table_name":"test_table","column1":"Column1Value","column2":"Column2Value"}</pre>
     * which you can parse into an Object using e.g. Jacksons ObjectMapper.<br>
     * If you extend {@link TableChangeNotification} you automatically inherit the mapping for
     * the table-name and the {@link SqlOperation} properties:
     * <pre>{@code
     * class TestTableNotification extends TableChangeNotification {
     *     @JsonProperty("column1")
     *     private String column1;
     *     @JsonProperty("column2")
     *     private String column2;
     * }
     *
     *
     * var notification = objectMapper.readValue(notificationPayload, TestTableNotification.class);
     * }</pre>
     * </blockquote>
     *
     * @param handle                                             the jdbi handle
     * @param tableName                                          the name of the table for which we want to setup a Table change notification FUNCTION and TRIGGER that can be used for LISTEN/NOTIFY
     * @param triggerOnSqlOperations                             what type of {@link SqlOperation should result in a notification
     * @param includeAdditionalTableColumnsInNotificationPayload any additional column names that should be included in the NOTIFY JSON formatted String payload
     */
    public static void addChangeNotificationTriggerToTable(Handle handle,
                                                           String tableName,
                                                           List<SqlOperation> triggerOnSqlOperations,
                                                           String... includeAdditionalTableColumnsInNotificationPayload) {
        requireNonNull(handle, "No handle provided");
        requireNonBlank(tableName, "No tableName provided");
        requireNonEmpty(triggerOnSqlOperations, "No triggerOnSqlOperations entries provided");

        var additionalColumnsPayLoadStatement = TABLE_NAME + ", " + SQL_OPERATION;
        var additionalColumnsSelectStatement  = "TG_TABLE_NAME, TG_OP";
        if (includeAdditionalTableColumnsInNotificationPayload != null && includeAdditionalTableColumnsInNotificationPayload.length > 0) {
            additionalColumnsPayLoadStatement += ", " + Arrays.stream(includeAdditionalTableColumnsInNotificationPayload).reduce((result, column) -> result + ", " + column).get();
            additionalColumnsSelectStatement += ", " + Arrays.stream(includeAdditionalTableColumnsInNotificationPayload).map(column -> "NEW." + column).reduce((result, column) -> result + ", " + column).get() + "\n";
        }

        var notifyFunctionSql = bind("CREATE OR REPLACE FUNCTION notify_{:tableName}_change()\n" +
                                             "       RETURNS trigger AS $$\n" +
                                             "       BEGIN\n" +
                                             "         PERFORM (\n" +
                                             "            WITH payload({:additionalColumnsPayLoadStatement}) AS (\n" +
                                             "              SELECT {:additionalColumnsSelectStatement}" +
                                             "            )\n" +
                                             "            SELECT pg_notify('{:channelName}', row_to_json(payload)::text) FROM payload);\n" +
                                             "         RETURN NULL;\n" +
                                             "       END;\n" +
                                             "       $$ LANGUAGE PLPGSQL;",
                                     arg("tableName", tableName),
                                     arg("channelName", resolveTableChangeChannelName(tableName)),
                                     arg("additionalColumnsPayLoadStatement", additionalColumnsPayLoadStatement),
                                     arg("additionalColumnsSelectStatement", additionalColumnsSelectStatement));

        log.debug("Notify Function for changes on '{}' SQL:\n{}", tableName, notifyFunctionSql);
        var update = handle.createUpdate(notifyFunctionSql);
        update.execute();
        log.info("Added Notification FUNCTION 'notify_{}_change' for table '{}'",
                 tableName,
                 tableName);

        // See https://www.postgresql.org/docs/current/plpgsql-trigger.html#PLPGSQL-DML-TRIGGER for more information
        var isReplaceTriggerSupported = PostgresqlUtil.getServiceMajorVersion(handle) >= 14;
        if (!isReplaceTriggerSupported) {
            var dropTriggerSql = bind("DROP TRIGGER IF EXISTS notify_on_{:tableName}_changes ON {:tableName} CASCADE",
                                  arg("tableName", tableName));
            log.debug("Trigger for '{}' changes SQL:\n{}", tableName, dropTriggerSql);
            update = handle.createUpdate(dropTriggerSql);
            update.execute();
            log.info("Dropped Notification TRIGGER 'notify_on_{}_changes' for table '{}'",
                     tableName,
                     tableName);
        }

        var triggerSql = bind("CREATE {:optionalReplace}TRIGGER notify_on_{:tableName}_changes\n" +
                                      "      {:when} {:on} ON {:tableName}\n" +
                                      "      FOR EACH ROW\n" +
                                      "         EXECUTE FUNCTION notify_{:tableName}_change();",
                              arg("tableName", tableName),
                              arg("optionalReplace", isReplaceTriggerSupported ? "OR REPLACE " : ""),
                              arg("when", "AFTER"),
                              arg("on", triggerOnSqlOperations.stream().map(Enum::name).reduce((result, on) -> result + " OR " + on).get())
                             );
        log.debug("Trigger for '{}' changes SQL:\n{}", tableName, triggerSql);
        update = handle.createUpdate(triggerSql);
        update.execute();
        log.info("Added Notification TRIGGER 'notify_on_{}_changes' for table '{}'",
                 tableName,
                 tableName);
    }

    /**
     * Remove Table change notification <code>FUNCTION</code> and an <b>AFTER</b> <code>TRIGGER</code> to the given Table in order to support the classical Postgresql LISTEN/NOTIFY concept.<br>
     *
     * @param handle    the jdbi handle
     * @param tableName the name of the table for which we want to remove the Table change notification FUNCTION and TRIGGER used for LISTEN/NOTIFY
     */
    public static void removeChangeNotificationTriggerFromTable(Handle handle,
                                                                String tableName) {
        requireNonNull(handle, "No handle provided");
        requireNonBlank(tableName, "No tableName provided");
        var notifyFunctionSql = bind("DROP FUNCTION IF EXISTS notify_{:tableName}_change() CASCADE",
                                     arg("tableName", tableName));

        var update = handle.createUpdate(notifyFunctionSql);
        update.execute();
        log.info("Removed Notification FUNCTION 'notify_{}_change' for table '{}'",
                 tableName,
                 tableName);

        var triggerSql = bind("DROP TRIGGER IF EXISTS notify_on_{:tableName}_changes " +
                                      "ON {:tableName} CASCADE",
                              arg("tableName", tableName));
        update = handle.createUpdate(triggerSql);
        update.execute();
        log.info("Removed Notification TRIGGER 'notify_on_{}_changes' for table '{}'",
                 tableName,
                 tableName);
    }


    /**
     * Listen to notifications related to the given table that was setup using {@link #addChangeNotificationTriggerToTable(Handle, String, List, String...)}<br>
     * As the default Postgresql driver doesn't support asynchronous notification, we're using a polling mechanism.<br>
     * <b>Note: If you need to listen to notification from multiple tables, then you MUST use {@link MultiTableChangeListener} instead.</b>
     * <blockquote>
     * Example:
     * <pre>{@code
     * subscription = ListenNotify.listen(jdbi,
     *                                    "test_table",
     *                                    Duration.ofMillis(200))
     *                            .subscribe(notificationPayload -> {
     *                                // Handle the String based notification payload received
     *                            });
     * }</pre>
     * </blockquote>
     *
     * @param jdbi            the jdbi instance
     * @param tableName       the name of the table for which we want to setup a Table change notification FUNCTION and TRIGGER that can be used for LISTEN/NOTIFY
     * @param pollingInterval the interval between each polling attempt
     * @return a {@link Flux} that contains the notification payload as a String.
     * @see MultiTableChangeListener
     */
    public static Flux<String> listen(Jdbi jdbi,
                                      String tableName,
                                      Duration pollingInterval) {
        requireNonNull(jdbi, "No jdbi provided");
        requireNonBlank(tableName, "No tableName provided");
        requireNonNull(pollingInterval, "No pollingInterval provided");

        var handleReference = new AtomicReference<Handle>();
        return Flux.defer(() -> {
                       var          handle = handleReference.get();
                       PGConnection connection;
                       try {
                           if (handle == null) {
                               handleReference.set(handle = jdbi.open());
                               handle.getConnection().setAutoCommit(true);
                               handle.getConnection().setReadOnly(true);
                               handle.getConnection().setTransactionIsolation(Connection.TRANSACTION_READ_COMMITTED);

                               log.info("Setting up LISTENER for table '{}'", tableName);
                               handle.execute("LISTEN " + resolveTableChangeChannelName(tableName));
                           }
                           connection = handle.getConnection().unwrap(PGConnection.class);
                           var notifications = connection.getNotifications();
                           if (notifications.length > 0) {
                               log.debug("Received {} Notification(s) for table '{}'", notifications.length, tableName);
                               return Flux.fromStream(Arrays.stream(notifications).map(PGNotification::getParameter));
                           } else {
                               log.trace("Didn't receive any Notifications for table '{}'", tableName);
                               return Flux.empty();
                           }
                       } catch (ConnectionException | SQLException e) {
                           log.error(msg("Failed to listen for notification for table '{}'", tableName),
                                     e);
                           // This may be due to Connection issue, so let's close the handle and reset the reference
                           try {
                               handle.close();
                           } catch (Exception ex) {
                               log.error(msg("Failed to close the listener Handle for table '{}'", tableName),
                                         e);
                               return Flux.error(ex);
                           }
                           handleReference.set(null);
                           return Flux.empty();
                       }
                   })
                   .doOnCancel(() -> {
                       var handle = handleReference.get();
                       if (handle != null) {
                           log.info("Removing LISTENER for table '{}'", tableName);
                           try {
                               handle.execute("UNLISTEN " + tableName);
                           } catch (Exception e) {
                               // Ignore

                           }
                           try {
                               handle.close();
                           } catch (Exception e) {
                               // Ignore
                           }
                       }
                   })
                   .repeatWhen(longFlux -> Flux.interval(pollingInterval)
                                               .onBackpressureDrop()
                                               .publishOn(Schedulers.newSingle("Postgresql-Listener-" + tableName, true)));
    }
}
