package plasma.docker

import dispatch.{Future => _, _}
import spray.json._

case class Version(Version : String, GitCommit : Int, GoVersion : String)

// Missing CpuShares, DNS, OnBuild
case class ContainerConfig(Image : String,
                           Hostname : String,
                           User : String,
                           Memory : Long,
                           MemorySwap : Long,
                           AttachStdin : Boolean,
                           AttachStdout : Boolean,
                           AttachStderr : Boolean,
                           StdinOnce : Boolean,
                           NetworkDisabled : Boolean,
                           Env : List[String],
                           WorkingDir : String,
                           Entrypoint : List[String],
                           Cmd : List[String],
                           Tty : Boolean,
                           OpenStdin : Boolean,
                           Volumes : Map[String, Option[JsValue]],
                           VolumesFrom : String,
                           ExposedPorts : Map[String, Option[JsValue]]) {

  def withCommand(command : String*) = this.copy(Cmd = command.toList)

  def withWorkingDir(dir : String) = this.copy(WorkingDir = dir)

}

case class ContainerPortConfig (PublicPort: Int, Type: String)
case class ContainerSearchResult(Command: String,
                                 Created: Int,
                                 Id: String,
                                 Image: String,
                                 Names: List[String],
                                 Ports: List[ContainerPortConfig],
                                 SizeRootFs: Option[Int],
                                 SizeRw: Option[Int],
                                 Status: String)


case class ContainerState(Running: Boolean,
                          Pid: Int,
                          ExitCode: Int,
                          StartedAt: String,
                          FinishedAt: String,
                          Ghost: Boolean)

case class ContainerNetworkSettings(IPAddress: String,
                                    IPPrefixLen: Int,
                                    Gateway: String,
                                    Bridge: String,
                                    Ports: Option[List[(String,String)]])

case class CreateContainerResponse(Id : String,
                                   Warnings : Option[List[String]])

case class HostConfig(Binds : Option[List[String]],
                      ContainerIDFile : String,
                      LxcConf : Option[List[String]],
                      Privileged : Boolean,
                      PublishAllPorts : Boolean,
                      PortBindings : Option[Map[String, String]],
                      Links : Option[JsValue])

case class InspectContainerResponse(ID: String,
                                    Created: String,
                                    Path: String,
                                    Args: List[String],
                                    Config: ContainerConfig,
                                    State: ContainerState,
                                    Image: String,
                                    NetworkSettings: ContainerNetworkSettings,
                                    ResolvConfPath: String,
                                    HostConfig: HostConfig)

/*
case class ContainerProcesses(Titles: List[String],
                              Processes: List[List[String]])
*/

/*
 * Kind 0 - Changed
 * Kind 1 - Added
 * Kind 2 - deleted
 */
case class FSChange(Path: String,
                    Kind: Int) {

  def isChange = 0 == Kind
  def isAdd    = 1 == Kind
  def idDelete = 2 == Kind
}

private case class WaitContainerResponse(StatusCode: Int)

private case class CopyResourceFromContainer(Resource: String)

private object JsonProtocol extends DefaultJsonProtocol {

  import DefaultJsonProtocol._

  implicit val versionFormat = jsonFormat3(Version)
  implicit val containerConfigFormat = jsonFormat19(ContainerConfig)
  implicit val createContainerResponseFormat = jsonFormat2(CreateContainerResponse)
  implicit val hostConfig = jsonFormat7(HostConfig)
  implicit val containerPortConfigFormat = jsonFormat2(ContainerPortConfig)
  implicit val ContainerSearchResultFormat = jsonFormat9(ContainerSearchResult)
  implicit val containerStateFormatFormat = jsonFormat6(ContainerState)
  implicit val cotainerNetworkSettingsFormat= jsonFormat5(ContainerNetworkSettings)
  implicit val inspectContainerResponseFormat = jsonFormat10(InspectContainerResponse)
  // implicit val containerProcessesResponseFormat = jsonFormat2(ContainerProcesses)
  implicit val fsChangeFormat = jsonFormat2(FSChange)
  implicit val waitContainerResponseFormat = jsonFormat1(WaitContainerResponse)
  implicit val copyResourceFromContainerFormat = jsonFormat1(CopyResourceFromContainer)

  def response[T](resp : Res)(implicit fmt : JsonFormat[T]) : T = {
    JsonParser(dispatch.as.String(resp)).convertTo[T]
  }
}
