package plasma.docker

import scala.collection.mutable.ArrayBuffer
import scala.concurrent._
import scala.async.Async.{async, await}
import scala.util.{Try, Success, Failure}
import dispatch.{Future => _, _}
import spray.json._
import java.io.{InputStream, DataInputStream, EOFException}

case class NoSuchContainer(id : String) extends Throwable
case class DockerException(req : Req, resp : Res) extends Throwable
case class BadParameter(all: Boolean, limit: Option[Int], since: Option[String], before: Option[String], showSize: Boolean) extends Throwable
case class ContainerNotRunning (cfg: ContainerConfig) extends Throwable

class Docker(dockerUrl : String) {

  import JsonProtocol._

  private val base = url(dockerUrl)

  def version()(implicit ec : ExecutionContext) : Future[Version] = {
    Http((base / "version").GET OK response[Version])
  }

  def deleteContainer(id : String,
                      volumes : Boolean = false,
                      force : Boolean = false)
    (implicit ec : ExecutionContext) = async {
    val req = base / "containers" / id <<? Map("force" -> force.toString,
                                               "v" -> volumes.toString)
    val resp = await(Http(req.DELETE))
    resp.getStatusCode match {
      case 204 => ()
      case 404 => throw NoSuchContainer(id)
      case _ => throw DockerException(req, resp)
    }
  }

  def listContainers(all: Boolean = false,
                     limit: Option[Int] = None,
                     since: Option[String] = None,
                     before: Option[String] = None,
                     showSize: Boolean = false)
    (implicit ec : ExecutionContext): Future[List[ContainerSearchResult]] = async {
    val req = base / "containers" / "json" <<? (Map("all" -> all.toString,
                                                    "size" -> showSize.toString)
                                                ++ limit.map("limit" -> _.toString)
                                                ++ since.map("since" -> _)
                                                ++ before.map("before" -> _))
    val resp = await (Http(req.GET))
    resp.getStatusCode match {
      case 200 => response[List[ContainerSearchResult]](resp)
      case 400 => throw BadParameter(all, limit, since, before, showSize)
      case _ => throw DockerException(req, resp)
    }
  }

  def createContainer(cfg: ContainerConfig,
                      name: Option[String] = None)
    (implicit ec : ExecutionContext): Future[CreateContainerResponse] = async {
      val req = (base / "containers" / "create" <<? name.map("name" -> _))
                .setContentType("application/json", "UTF-8") << cfg.toJson.compactPrint
      val resp = await(Http(req.POST))
      resp.getStatusCode match {
        case 201 => response[CreateContainerResponse](resp)
        case 404 => throw NoSuchContainer(cfg.Image)
        case 406 => throw ContainerNotRunning(cfg)
        case _ => throw DockerException(req, resp)
      }
    }

  def inspectContainer(id: String)
    (implicit ec : ExecutionContext): Future[InspectContainerResponse] = async {
      val req = base / "containers" / id / "json"
      val resp = await(Http(req.GET))
      resp.getStatusCode match {
        case 200 => response[InspectContainerResponse](resp)
        case 404 => throw NoSuchContainer(id)
        case _ => throw new DockerException(req, resp)
      }
    }

  /*
  def listContainerProcesses(id: String)
    (implicit ec : ExecutionContext): Future[ContainerProcesses] = async {
      val req = base / "containers" / id / "top"
      val resp= await(Http(req.GET))
      resp.getStatusCode match {
        case 200 => response[ContainerProcesses](resp)
        case 404 => throw NoSuchContainer(id)
        case _ => throw DockerException(req, resp)
      }
    }
  */

  def containerFileSystemChanges(id: String)
    (implicit ec : ExecutionContext): Future[List[FSChange]] = async {
      val req = base / "containers" / id / "changes"
      val resp = await(Http(req.GET))
      resp.getStatusCode match {
        case 200 => response[List[FSChange]](resp)
        case 404 => throw NoSuchContainer(id)
        case _ => throw DockerException(req, resp)
      }
    }

  def startContainer(id: String,
                     hostConf: HostConfig = HostConfig.empty)
    (implicit ec : ExecutionContext) = async {
      val req = ((base / "containers" / id / "start") <<
                    hostConf.toJson.compactPrint)
                .setContentType("application/json", "UTF-8")

      val resp = await(Http(req.POST))
      resp.getStatusCode match {
        case 204 => ()
        case 404 => throw NoSuchContainer(id)
        case _ => throw DockerException(req, resp)
      }
    }

  def stopContainer(id: String,
                    t: Option[Int] = None)
    (implicit ec : ExecutionContext) = async {
      val req = base / "containers" / id / "stop" <<? t.map("t" -> _.toString)
      val resp = await(Http(req.POST))
      resp.getStatusCode match {
        case 204 => ()
        case 404 => throw NoSuchContainer(id)
        case _ => throw DockerException(req, resp)
      }
    }

  def restartContainer(id: String,
                       t: Option[Int] = None)
    (implicit ec : ExecutionContext) = async {
      val req = base / "containers" / id / "restart" <<? t.map("t" -> _.toString)
      val resp = await(Http(req.POST))
      resp.getStatusCode match {
        case 204 => ()
        case 404 => throw NoSuchContainer(id)
        case _ => throw DockerException(req, resp)
      }
    }

  def killContainer(id: String,
                    sig: Option[Int] = None)
    (implicit ec : ExecutionContext) = async {
      val req = base / "containers" / id / "kill" <<? sig.map("signal" -> _.toString)
      val resp = await(Http(req.POST))
      resp.getStatusCode match {
        case 204 => ()
        case 404 => throw NoSuchContainer(id)
        case _ => throw DockerException(req, resp)
      }
    }

  def waitContainer(id: String)
    (implicit ec : ExecutionContext): Future[Int] = async {
      val req = base / "containers" / id / "wait"
      val resp = await(Http(req.POST))
      resp.getStatusCode match {
        case 200 => response[WaitContainerResponse](resp).StatusCode
        case 404 => throw NoSuchContainer(id)
        case _ => throw DockerException(req, resp)
      }
    }

  def copyFiles(id: String,
                res: String)
    (implicit ec : ExecutionContext): Future[InputStream] = async {
      val req = (base / "containers" / id / "copy")
                 .setContentType("application/json", "UTF-8") <<
                 CopyResourceFromContainer(res).toJson.compactPrint
      val resp = await(Http(req.POST))
      resp.getStatusCode match {
        case 200 => resp.getResponseBodyAsStream
        case 404 => throw NoSuchContainer(id)
        case _ => throw DockerException(req, resp)
      }
    }

  private def parseAttach(input : DataInputStream)
    (implicit ec : ExecutionContext) : Stream[Array[Byte]] = {
    val result = Try {
      val typ = input.readByte()
      val pad0 = input.readByte()
      val pad1 = input.readShort()
      val size = input.readInt()
      val buf = new Array[Byte](size)
      input.readFully(buf)
      buf
    }
    result match {
      case Success(buf) => buf #:: parseAttach(input)
      case Failure(_ : EOFException) => {
        input.close()
        Stream.Empty
      }
      case Failure(exn) => {
        input.close()
        throw exn
      }
    }
  }

  def logs(id : String, stderr : Boolean)
    (implicit ec : ExecutionContext) : Future[Array[Byte]] = async {
    val query = Map("logs" -> "true",
                    "stdout" -> (!stderr).toString(),
                    "stderr" -> stderr.toString())
    val req = base / "containers"/ id / "attach" <<? query
    val resp = await(Http(req.POST))
    resp.getStatusCode match {
      case 200 => {
        val stream = new DataInputStream(resp.getResponseBodyAsStream())
        val arrs = parseAttach(stream)
        val buf = new ArrayBuffer[Byte]()
        for (arr <- arrs) {
          buf.append(arr : _*)
        }
        buf.toArray
      }
      case 404 => throw NoSuchContainer(id)
      case _ => throw DockerException(req, resp)
    }
  }

}
