/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * JFlex 1.7.0-SNAPSHOT                                                    *
 * Copyright (C) 1998-2015  Gerwin Klein <lsf@jflex.de>                    *
 * All rights reserved.                                                    *
 *                                                                         *
 * License: BSD                                                            *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

package jflex;

import java.io.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This class manages the actual code generation, putting
 * the scanner together, filling in skeleton sections etc.
 *
 * Table compression, String packing etc. is also done here.
 *
 * @author Gerwin Klein
 * @version JFlex 1.7.0-SNAPSHOT
 */
public abstract class Emitter {
  protected static final Pattern JAVADOC_COMMENT_AND_MAYBE_ANNOTATIONS_PATTERN
      = Pattern.compile
          (".*/\\*\\*(.*)\\*/"               // javadoc comment, embedded '*/' disallowed
          +"(?:\\s*@[a-z][a-z0-9_]*(?:\\.[a-z][a-z0-9_]*)*"        // @[p.ack.age.]AnnotationClass
          +"   (?:\\s*\\(\\s*(?:\"(?:\\\"|[^\"])*\""               // ignore close parens in double quotes
          +"                   |'(?:[^']|\\\\(?:'|u[0-9a-f]{4}))'" // ignore close parens in single quotes
          +"                   |[^)])+\\))?"                       // optional annotation params
          +")*\\s*",                         // zero or more annotations, followed by optional whitespace
           Pattern.DOTALL | Pattern.CASE_INSENSITIVE | Pattern.COMMENTS);

  // bit masks for state attributes
  static final protected int FINAL = 1;
  static final protected int NOLOOK = 8;

  protected File inputFile;

  protected PrintWriter out;
  protected Skeleton skel;
  protected LexScan scanner;
  protected LexParse parser;
  protected DFA dfa;

  protected boolean isTransition[];
  
  // for row killing:
  protected int numRows;
  protected int [] rowMap;
  protected boolean [] rowKilled;
  
  // for col killing:
  protected int numCols;
  protected int [] colMap;
  protected boolean [] colKilled;
  

  /** maps actions to their switch label */
  protected Map<Action,Integer> actionTable = new LinkedHashMap<Action,Integer>();

  protected CharClassInterval [] intervals;

  /**
   * Computes base name of the class name. Needs to take into account generics.
   *
   * @param className Class name for which to construct the base name
   * @see LexScan#className
   * @return the
   */
  public static String getBaseName(String className) {
    int gen = className.indexOf('<');
    if (gen < 0) {
      return className;
    }
    else {
      return className.substring(0, gen);
    }
  }
  

  /**
   * Constructs a file in Options.getDir() or in the same directory as
   * another file. Makes a backup if the file already exists.
   *
   * @param name  the name (without path) of the file
   * @param input fall back location if path = <tt>null</tt>
   *              (expected to be a file in the directory to write to)   
   * @return The constructed File
   */
  public static File normalize(String name, File input) {
    File outputFile;

    if ( Options.getDir() == null ) 
      if ( input == null || input.getParent() == null )
        outputFile = new File(name);
      else
        outputFile = new File(input.getParent(), name);
    else 
      outputFile = new File(Options.getDir(), name);
        
    if ( outputFile.exists() && !Options.no_backup ) {      
      File backup = new File( outputFile.toString()+"~" );
      
      if ( backup.exists() ) backup.delete();
      
      if ( outputFile.renameTo( backup ) )
        Out.println("Old file \""+outputFile+"\" saved as \""+backup+"\"");
      else
        Out.println("Couldn't save old file \""+outputFile+"\", overwriting!");
    }

    return outputFile;
  }

  protected void println() {
    out.println();
  }

  protected void println(String line) {
    out.println(line);
  }

  protected void println(int i) {
    out.println(i);
  }

  protected void print(String line) {
    out.print(line);
  }

  protected void print(int i) {
    out.print(i);
  }

  protected void print(int i, int tab) {
    int exp;

    if (i < 0) 
      exp = 1;
    else
      exp = 10;

    while (tab-- > 1) {
      if (Math.abs(i) < exp) print(" ");
      exp*= 10;
    }

    print(i);
  }

  protected boolean hasGenLookAhead() {
    return dfa.lookaheadUsed;
  }
  
  protected abstract void emitLookBuffer();
  
  protected abstract void emitScanError();

  protected abstract void emitMain();
  
  protected abstract void emitNoMatch();
  
  protected abstract void emitNextInput();

  protected void emitHeader() {
    println("/* The following code was generated by JFlex " + Main.version + " */");
    println("");
  } 

  protected abstract void emitUserCode();

  protected abstract void emitClassName();

  /**
   * Try to find out if user code ends with a javadoc comment,
   * maybe followed by one or more annotations
   * 
   * @param usercode  the user code
   * @return true     if it ends with a javadoc comment and zero or more annotations
   */
  public static boolean endsWithJavadoc(StringBuilder usercode) {
    Matcher matcher = JAVADOC_COMMENT_AND_MAYBE_ANNOTATIONS_PATTERN.matcher(usercode);
    return matcher.matches() && ! matcher.group(1).contains("*/");
  }


  protected abstract void emitLexicalStates();

  protected abstract void emitDynamicInit();

  protected abstract void emitCharMapInitFunction(int packedCharMapPairs);

  protected abstract void emitCharMapArrayUnPacked();

  /**
   * Returns the number of elements in the packed char map
   * array, or zero if the char map array will be not be packed.
   * 
   * This will be more than intervals.length if the count
   * for any of the values is more than 0xFFFF, since
   * the number of char map array entries per value is
   * ceil(count / 0xFFFF)
   */
  protected abstract int emitCharMapArray();


  /**
   * Print number as octal/unicode escaped string character.
   * 
   * @param c   the value to print
   * @prec  0 <= c <= 0xFFFF 
   */
  protected void printUC(int c) {
    if (c > 255) {
      out.print("\\u");
      if (c < 0x1000) out.print("0");
      out.print(Integer.toHexString(c));
    }
    else {
      out.print("\\");
      out.print(Integer.toOctalString(c));
    }    
  }


  protected abstract void emitRowMapArray();


  protected abstract void emitAttributes();


  protected abstract void emitClassCode();

  protected abstract void emitConstructorDecl();
  
  protected abstract void emitConstructorDecl(boolean printCtorArgs);

  protected void emitCtorArgs() {
    for (int i = 0; i < scanner.ctorArgs.size(); i++) {
      print(", "+scanner.ctorTypes.get(i));
      print(" "+scanner.ctorArgs.get(i));
    }    
  }

  protected abstract void emitDoEOF();

  protected abstract void emitLexFunctHeader();
  
  protected abstract void emitGetRowMapNext();

  /**
   * Escapes all " ' \ tabs and newlines
   * 
   * @param s The string to escape
   * @return The escaped string
   */
  protected String escapify(String s) {
    StringBuilder result = new StringBuilder(s.length()*2);
    
    for (int i = 0; i < s.length(); i++) {
      char c = s.charAt(i);
      switch (c) {
      case '\'': result.append("\\\'"); break;
      case '\"': result.append("\\\""); break;
      case '\\': result.append("\\\\"); break;
      case '\t': result.append("\\t"); break;
      case '\r': if (i+1 == s.length() || s.charAt(i+1) != '\n') result.append("\"+ZZ_NL+\""); 
                 break;
      case '\n': result.append("\"+ZZ_NL+\""); break;
      default: result.append(c);
      }
    }

    return result.toString();
  }

  public abstract void emitActionTable();

  protected abstract void emitActions();

  protected abstract void emitEOFVal();
  
  protected void findActionStates() {
    isTransition = new boolean [dfa.numStates];
    
    for (int i = 0; i < dfa.numStates;  i++) {
      char j = 0;
      while ( !isTransition[i] && j < dfa.numInput )
        isTransition[i] = dfa.table[i][j++] != DFA.NO_TARGET;
    }
  }
  
  protected void reduceColumns() {
    colMap = new int [dfa.numInput];
    colKilled = new boolean [dfa.numInput];

    int i,j,k;
    int translate = 0;
    boolean equal;

    numCols = dfa.numInput;

    for (i = 0; i < dfa.numInput; i++) {
      
      colMap[i] = i-translate;
      
      for (j = 0; j < i; j++) {
        
        // test for equality:
        k = -1;
        equal = true;        
        while (equal && ++k < dfa.numStates) 
          equal = dfa.table[k][i] == dfa.table[k][j];
        
        if (equal) {
          translate++;
          colMap[i] = colMap[j];
          colKilled[i] = true;
          numCols--;
          break;
        } // if
      } // for j
    } // for i
  }
  
  protected void reduceRows() {
    rowMap = new int [dfa.numStates];
    rowKilled = new boolean [dfa.numStates];
    
    int i,j,k;
    int translate = 0;
    boolean equal;

    numRows = dfa.numStates;

    // i is the state to add to the new table
    for (i = 0; i < dfa.numStates; i++) {
      
      rowMap[i] = i-translate;
      
      // check if state i can be removed (i.e. already
      // exists in entries 0..i-1)
      for (j = 0; j < i; j++) {
        
        // test for equality:
        k = -1;
        equal = true;
        while (equal && ++k < dfa.numInput) 
          equal = dfa.table[i][k] == dfa.table[j][k];
        
        if (equal) {
          translate++;
          rowMap[i] = rowMap[j];
          rowKilled[i] = true;
          numRows--;
          break;
        } // if
      } // for j
    } // for i
    
  } 


  /**
   * Set up EOF code section according to scanner.eofcode 
   */
  protected abstract void setupEOFCode();


  /**
   * Main Emitter method.  
   */
  public abstract void emit();

}
