/* Copyright 2013 University of South Florida.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

package edu.usf.MessageService

import wslite.http.auth.*
import wslite.rest.*
import wslite.json.*
import groovy.json.JsonOutput
import groovy.util.logging.Slf4j

import static java.util.UUID.randomUUID

/**
* Groovy Client for <a href="https://github.com/USF-IT/MessageService">MessageService</a> <br>
* 
*
* @author      Eric Pierce (epierce@usf.edu)
* @version     0.6.1
**/

@Slf4j
class MessageServiceClient {

    private def messageService
    private def messageServiceHost
    private def messageServiceBase = 'MessageService/basic'
    private def messageServiceUser
    private def messageServicekeyStore
    private def messageServiceQueueDir = new File('/tmp')
    private def authenticated = false
    private def secureConnection = false

    /**
    *
    * @param messageHost    URL of the MessageService Host (Example: https://server.example.edu)
    *
    **/
    def MessageServiceClient(String messageHost){
        log.debug("Creating new MessageServiceClient for ${messageHost}")
        messageServiceHost = messageHost
        messageService = new RESTClient(messageServiceHost)
        messageService.defaultAcceptHeader = "application/json"
        messageService.defaultContentTypeHeader = "application/json"
        messageService.defaultCharset = "UTF-8"
    }

    /**
    * Set the directory to write local queue files to in case of a MessageService outage or error
    *
    * @param queueDirectory     Directory to save files in
    *
    * @throws FileNotFoundException 
    **/
    void setQueueFileLocation(String queueDirectory){
        log.debug("Setting Queue Directory to ${queueDirectory}")
        messageServiceQueueDir = new File(queueDirectory)
        if (! messageServiceQueueDir.isAbsolute()) throw new FileNotFoundException("${queueDirectory} must be an absolute path.")
        if (! messageServiceQueueDir.exists()) throw new FileNotFoundException("${queueDirectory} does not exist.")
        if (! messageServiceQueueDir.canWrite()) throw new FileNotFoundException("${queueDirectory} must be an writeable.")
    }

    /**
    * Set the username and password for MessageService access
    *
    * @param username
    * @param password
    *
    **/
    void setCredentials(String username, String password) {
        log.debug("Setting credentials for ${username}")
        messageServiceUser = username
        authenticated = true

        //Add credentials to RESTClient
        messageService.authorization = new HTTPBasicAuthorization(username, password)
    }

    /**
    * Set the location and password of the Java Trust Store to use for this connection.
    *
    * @param keyStorePath   Java key/trust store
    * @param keyStorePass   keystore Password
    *
    * @throws FileNotFoundException 
    **/
    void setKeyStoreLocation(String keyStorePath, String keyStorePass){
        log.debug("Using ${keyStorePath} as custom TrustStore")

        def keyStoreFile = new File(keyStorePath)
        if (keyStoreFile.exists() && keyStoreFile.canRead()){
            messageService.httpClient.sslTrustStoreFile = keyStorePath
            messageService.httpClient.sslTrustStorePassword = keyStorePass
            secureConnection = true
        } else {
            throw new FileNotFoundException("${keyStorePath} is not readable.")
        }
    }

    /**
    * Disable certificate validation and trust the certificate of any server we connect to.  
    * When dealing with lots of servers with self-signed certs (development), it can be 
    * helpful to bypass a custom trust store and trust all certs automatically.
    *
    * @since 0.6.0
    *
    **/
    void trustAllCerts(){
        log.warn("Trusting all SSL certificates. NOT RECOMMENDED FOR PRODUCTION")
        messageService.httpClient.sslTrustAllCerts = true
        secureConnection = true
    }
    

    /** 
    * Set the base location of the MessageService deployment.
    * 
    * @param path   MessageService path (Default: /MessageService/basic)
    *
    **/
    void setMessageBasePath(String path){
        log.debug("Setting base MessageService path to ${messageServiceBase}")
        messageServiceBase = path
    }

    /**
    * Get a list of all message topics
    *
    * @return JSONObject containing results from MessageService
    **/
    JSONObject getTopicList() {
        getContainerList('topic')
    }

    /**
    * Get a list of all message queues
    *
    * @return JSONObject containing results from MessageService
    **/
    JSONObject getQueueList() {
        getContainerList('queue')
    }

    /**
    * Private method for requesting queue/topic lists from MessageService
    *
    * @return JSONObject containing results from MessageService
    **/
    private JSONObject getContainerList(String type){
        try {
            checkConnectionOptions()            
        } catch(RuntimeException e) {
            throw new RuntimeException(e.message)
        }

        try {
            log.debug("Listing ${type}s for user ${messageServiceUser}")
            
           def response = messageService.get( path: "/${messageServiceBase}/${type}" )
           
           response.json

        } catch (RESTClientException ex){
            if (ex.cause){
                throw new Exception("${ex.cause} ${ex.message}")
            } else {
                throw new Exception(ex.message)
            }
        } catch (Exception ex) {
            throw new Exception(ex.message)
        }
    }

    /**
    * Get a list of all messages in a topic
    *
    * @param topic      Topic to retrive messages from
    * @param startTime  Time to start message list at.  Format yyyy-MM-dd'T'HH:mm:ss (optional)
    * @param endTime    Time to end message list at. Format yyyy-MM-dd'T'HH:mm:ss (optional)
    * @return           JSONArray containing results from MessageService
    **/  
    JSONArray getTopicMessages(String topic, startTime = null , endTime = null) {
        getMessages([type: 'topic', name: topic, startTime: startTime, endTime: endTime])
    }

    /**
    * Get oldest "pending" message in a queue
    *
    * @param queue      Queue to retrive messages from
    * @return           JSONArray containing message from MessageService
    **/ 
    JSONObject getQueueMessage(String queue) {
        getMessages([type: 'queue', name: queue])
    }

    /**
    * Get a list of messages in a queue without changing their status
    *
    * @param queue      queue to retrive messages from
    * @param num        Number of messages to retrieve (Default: 10)
    * @return           JSONArray containing results from MessageService
    **/ 
    JSONArray peek(String queue, Integer num = 10) {
        getMessages([type: 'queue', name: queue, peek: true, num: num])
    }

    /**
    * Get a list of all messages in a queue with the status 'in-progress'
    *
    * @param queue      Queue to retrive messages from
    * @return           JSONArray containing results from MessageService
    **/ 
    JSONArray getInProgressMessages(String queue) {
        getMessages([type: 'queue', name: queue, status: "in-progress"])
    }

    /**
    * Private method to retrieve messages from MessageService
    *
    * @param options    Map containing the following data
    *                      * Type of container (topic or queue)
    *                      * Name of the Topic/Queue to retrive messages from
    *                      * startTime (topic only)
    *                      * endTime (topic only)
    *                      * num (queue only)
    *                      * status (queue only)
    * @return           JSONObject or JSONArray containing results from MessageService
    * @throws Exception
    * @throws RuntimeException
    **/ 
    private def getMessages(options){
        try {
            checkConnectionOptions()            
        } catch(RuntimeException e) {
            throw new RuntimeException(e.message)
        }     

        try {
            def path = "/${messageServiceBase}/${options.type}/${options.name}"
            def query
            def response

            if (options.peek) {
                log.debug("Peeking at ${options.num} messages in ${options.type} ${options.name} for user ${messageServiceUser}")
                path = "${path}/peek"
                query = [count: options.num]
            } else if (options.status) {
                log.debug("Getting messages with status ${options.status} in ${options.type} ${options.name} for user ${messageServiceUser}")
                path = "${path}/${options.status}"
            } else if (options.startTime){
                log.debug("Getting messages starting at ${options.startTime} in ${options.type} ${options.name} for user ${messageServiceUser}")
                path = "${path}/filter"
                query = [startTime: options.startTime]
                if (options.endTime) query.put("endTime", options.endTime)
            } else {
                log.debug("Getting messages in ${options.type} ${options.name} for user ${messageServiceUser}")
            }
            
            if (query) {
                response = messageService.get( path: path , query: query)
            } else {
                response = messageService.get( path: path)
            }

            if ((options.type == 'queue') && (! options.peek) && (! options.status)) {
                return response.json.messages[0]
            } else {
                return response.json.messages
            }
        } catch (RESTClientException ex){
            if (ex.cause){
                throw new Exception("${ex.cause} ${ex.message}")
            } else {
                throw new Exception(ex.message)
            }
        } catch (Exception ex) {
            throw new Exception(ex.message)
        }
    }

    /**
    * Create a new topic
    * 
    * @param name
    * @param canRead    List of the user who can read messages from this topic
    * @param canWrite   List of the user who can write messages to this topic
    * @param canAdmin   List of the user who can modify this topic
    * @return           JSONObject of the topic data
    * @throws Exception
    * @throws RuntimeException
    **/
    JSONObject createTopic(String name, List canRead = [], List canWrite = [], List canAdmin = []) {
        createContainer('topic', name, canRead, canWrite, canAdmin)
    }

    /**
    * Create a new queue
    * 
    * @param name
    * @param canRead    List of the user who can read messages from this queue
    * @param canWrite   List of the user who can write messages to this queue
    * @param canAdmin   List of the user who can modify this queue
    * @return           JSONObject of the queue data
    * @throws Exception
    * @throws RuntimeException
    **/
    JSONObject createQueue(String name, List canRead = [], List canWrite = [], List canAdmin = []) {
        createContainer('queue', name, canRead, canWrite, canAdmin)
    }

    /**
    * Private method to create a topic/queue in MessageService
    * 
    * @param type
    * @param name
    * @param canRead    List of the user who can read messages from this queue/topic
    * @param canWrite   List of the user who can write messages to this queue/topic
    * @param canAdmin   List of the user who can modify this queue/topic
    * @return           JSONObject of the queue/topic data
    * @throws Exception
    * @throws RuntimeException
    **/
    private JSONObject createContainer(String type, String name, List canRead, List canWrite, List canAdmin) {
        try {
            checkConnectionOptions()            
        } catch(RuntimeException e) {
            throw new RuntimeException(e.message)
        }     

        def path = "/${messageServiceBase}/${type}"

        def messageData = [name: name, permissions: [canRead: canRead, canWrite: canWrite, canAdmin: canAdmin]]
        
        try {           
            log.debug("Creating ${type} ${name} for user ${messageServiceUser}")
            def response = messageService.post( path: path) { json messageData: messageData }
                        
            log.debug("Got response: ${response}")
            
            if (type == 'topic') {
                return response.json.topics    
            } else {
                return response.json.queues
            }

        } catch (RESTClientException ex){
            queueLocal("createContainer", type, name, [messageData: data])
            if (ex.cause){
                throw new Exception("${ex.cause} ${ex.message}")
            } else {
                throw new Exception(ex.message)
            }
        } catch (Exception ex) {
            queueLocal("createContainer", type, name, [messageData: data])            
            throw new Exception(ex.message)
        }
    }

    /**
    * Delete Topic
    *
    * @param name
    * @return       Name of the deleted topic
    * @throws Exception
    * @throws RuntimeException
    **/
    String deleteTopic(String name) {
        deleteContainer('topic', name)
    }

    /**
    * Delete Queue
    *
    * @param name
    * @return       Name of the deleted queue
    * @throws Exception
    * @throws RuntimeException
    **/
    String deleteQueue(String name) {
        deleteContainer('queue', name)
    }

    /**
    * Private method that deletes topics/queus in MessageService
    *
    * @param type   topic or queue
    * @param name
    * @return       Name of the deleted topic
    * @throws Exception
    * @throws RuntimeException
    **/
    private String deleteContainer(String type, String name) {
        try {
            checkConnectionOptions()            
        } catch(RuntimeException e) {
            throw new RuntimeException(e.message)
        }     

        def path = "/${messageServiceBase}/${type}/${name}"
        
        try {           
            log.info("Deleting ${type} ${name} for user ${messageServiceUser}")
            def response = messageService.delete( path: path)

            return response.json.topics.name

        } catch (RESTClientException ex){
            queueLocal("deleteContainer", type, name, [name: name])
            if (ex.cause){
                throw new Exception("${ex.cause} ${ex.message}")
            } else {
                throw new Exception(ex.message)
            }
        } catch (Exception ex) {
            queueLocal("deleteContainer", type, name, [name: name])
            throw new Exception(ex.message)
        }
    }


    /**
    * Create a new message in a topic
    * 
    * @param program    Name of the program creating this message
    * @param name       Topic to write the message to
    * @param data       Data to write to the message
    * @return           JSONObject of the created message
    * @throws Exception
    * @throws RuntimeException
    **/
    JSONObject createTopicMessage(String program, String name, Map data) {
        createMessage('topic', program, name, data)
    }

    /**
    * Create a new message in a queue
    * 
    * @param program    Name of the program creating this message
    * @param name       Queue to write the message to
    * @param data       Data to write to the message
    * @return           JSONObject of the created message
    * @throws Exception
    * @throws RuntimeException
    **/
    JSONObject createQueueMessage(String program, String name, Map data) {
        createMessage('queue', program, name, data)
    }

    /**
    * Private method to create new messages in MessageService
    * 
    * @param type       queue or topic
    * @param program    Name of the program creating this message
    * @param name       Queue to write the message to
    * @param data       Data to write to the message
    * @return           JSONObject of the created message
    * @throws Exception
    * @throws RuntimeException
    **/
    private JSONObject createMessage(String type, String program, String name, Map data) {
        try {
            checkConnectionOptions()            
        } catch(RuntimeException e) {
            throw new RuntimeException(e.message)
        }     

        def path = "/${messageServiceBase}/${type}/${name}"

        def body = [apiVersion: 1, createProg: program, messageData: data]
        
        try {           
            log.debug("Creating message on ${type} ${name} for user ${messageServiceUser}")
            def response = messageService.post( path: path) { json messageData: data, apiVersion: 1, createProg: program }
           // def response = messageService.post( path: path, body: body, requestContentType : JSON )

            return response.json.messages

        } catch (RESTClientException ex){
            queueLocal("createMessage", type, name, [messageData: data, apiVersion: 1, createProg: program])
            if (ex.cause){
                throw new Exception("${ex.cause} ${ex.message}")
            } else {
                throw new Exception(ex.message)
            }
        } catch (Exception ex) {
            queueLocal("createMessage", type, name, [messageData: data, apiVersion: 1, createProg: program])
            throw new Exception(ex.message)
        }
    }

    /**
    * Change the status on a message in a queue
    *
    * @param name       Queue containing the message
    * @param id         Message ID
    * @param status     Status to change the message to. (allowed: in-progress, pending, completed, error)
    * @return           JSONObject of the new state of the message
    * @throws Exception
    * @throws RuntimeException
    **/
    JSONObject changeMessageStatus(String name, String id, String status) {
        try {
            checkConnectionOptions()            
        } catch(RuntimeException e) {
            throw new RuntimeException(e.message)
        }     

        def path = "/${messageServiceBase}/queue/${name}/${id}"

        def body = [apiVersion: 1, status: status]
        
        try {           
            log.info("Updating status to ${status} on message ${id} in queue ${name} for user ${messageServiceUser}")
            def response = messageService.put( path: path) { json status: status, apiVersion: 1 }

            return response.json.messages

        } catch (RESTClientException ex){
            queueLocal("changeMessageStatus", "queue", name, [status: status, apiVersion: 1])
            if (ex.cause){
                throw new Exception("${ex.cause} ${ex.message}")
            } else {
                throw new Exception(ex.message)
            }
        } catch (Exception ex) {
            queueLocal("changeMessageStatus", "queue", name, [status: status, apiVersion: 1])
            throw new Exception(ex.message)
        }
    }

    /**
    * Delete a message from a queue
    *
    * @param name       Queue containing the message
    * @param id         Message ID
    * @return           id of the deleted message
    * @throws Exception
    * @throws RuntimeException
    **/
    String deleteQueueMessage(String name, String id) {
        deleteMessage('queue', name, id)
    }

    /**
    * Private method to delete a message from MessageService
    *
    * @param type       topic or queue
    * @param name       Queue containing the message
    * @param id         Message ID
    * @return           id of the deleted message
    * @throws Exception
    * @throws RuntimeException
    **/
    private String deleteMessage(String type, String name, String id) {
        try {
            checkConnectionOptions()            
        } catch(RuntimeException e) {
            throw new RuntimeException(e.message)
        }     

        def path = "/${messageServiceBase}/${type}/${name}/${id}"
        
        try {           
            log.info("Deleting message ${id} from ${type} ${name} for user ${messageServiceUser}")
            def response = messageService.delete( path: path)

            return id

        } catch (RESTClientException ex){
            if (ex.cause){
                throw new Exception("${ex.cause} ${ex.message}")
            } else {
                throw new Exception(ex.message)
            }
        } catch (Exception ex) {
            throw new Exception(ex.message)
        }
    }

    private void checkConnectionOptions() {
        if (! authenticated) {
            log.error("setCredentials() must be called before connecting to MessageService")
            throw new RuntimeException("setCredentials() must be called before connecting to MessageService")
        }
        if (! secureConnection) {
            log.error("setKeyStoreLocation() must be called before connecting to MessageService")
            throw new RuntimeException("setKeyStoreLocation() must be called before connecting to MessageService")
        }
    }

    private String createQueueFilename(String action, String type, String name){
        def time = System.currentTimeMillis()/1000l as Integer
        def uuid = randomUUID() as String
        "${time}:${action}:${name}:${uuid}.${type}"
    }

    private def queueLocal(action, type, name, requestBody){
        queueLocal(action, type, name, requestBody, null)
    }

    private def queueLocal(action, type, name, requestBody, response){
        def queueFilename = createQueueFilename(action,type,name)
        def jsonOut = new JsonOutput()

        log.warn("Saving message to a local queue file: ${messageServiceQueueDir}${File.separator}${queueFilename}")
        def queueFile = new File("${messageServiceQueueDir}${File.separator}${queueFilename}")
        queueFile.append "### Request Body ### \n"
        queueFile.append "${jsonOut.prettyPrint(jsonOut.toJson(requestBody))} \n"

        if (response) {
            queueFile.append "### Response Headers ### \n"
            response.headers.each {
                queueFile.append "${it.name} : ${it.value} \n"
            }

            queueFile.append "### Response Body ### \n"
            queueFile.append "${response.data} \n"  
        }

    }



 }