/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.aws;

import com.google.common.annotations.VisibleForTesting;
import edu.wisc.library.ocfl.api.exception.OcflIOException;
import edu.wisc.library.ocfl.api.exception.OcflInputException;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.storage.cloud.CloudClient;
import edu.wisc.library.ocfl.core.storage.cloud.CloudObjectKey;
import edu.wisc.library.ocfl.core.storage.cloud.HeadResult;
import edu.wisc.library.ocfl.core.storage.cloud.KeyNotFoundException;
import edu.wisc.library.ocfl.core.storage.cloud.ListResult;
import edu.wisc.library.ocfl.core.util.UncheckedFiles;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.AbortMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompleteMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.CompletedMultipartUpload;
import software.amazon.awssdk.services.s3.model.CompletedPart;
import software.amazon.awssdk.services.s3.model.CopyObjectRequest;
import software.amazon.awssdk.services.s3.model.CreateMultipartUploadRequest;
import software.amazon.awssdk.services.s3.model.Delete;
import software.amazon.awssdk.services.s3.model.DeleteObjectsRequest;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadBucketRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Response;
import software.amazon.awssdk.services.s3.model.NoSuchBucketException;
import software.amazon.awssdk.services.s3.model.NoSuchKeyException;
import software.amazon.awssdk.services.s3.model.ObjectIdentifier;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.UploadPartCopyRequest;
import software.amazon.awssdk.services.s3.model.UploadPartCopyResponse;
import software.amazon.awssdk.services.s3.model.UploadPartRequest;
import software.amazon.awssdk.services.s3.model.UploadPartResponse;
import software.amazon.awssdk.utils.http.SdkHttpUtils;

public class OcflS3Client
implements CloudClient {
    private static final Logger LOG = LoggerFactory.getLogger(OcflS3Client.class);
    private static final int KB = 1024;
    private static final int MB = 0x100000;
    private static final long GB = 0x40000000L;
    private static final long TB = 0x10000000000L;
    private static final long MAX_FILE_BYTES = 0x50000000000L;
    private static final int MAX_PART_BYTES = 0x6400000;
    private static final int PART_SIZE_BYTES = 0xA00000;
    private static final int MAX_PARTS = 100;
    private static final int PART_SIZE_INCREMENT = 10;
    private static final int PARTS_INCREMENT = 100;
    private final S3Client s3Client;
    private final String bucket;
    private final String repoPrefix;
    private final CloudObjectKey.Builder keyBuilder;
    private final BiConsumer<String, PutObjectRequest.Builder> putObjectModifier;
    private final BiConsumer<String, CreateMultipartUploadRequest.Builder> createMultipartModifier;
    private int maxPartBytes = 0x6400000;
    private int partSizeBytes = 0xA00000;

    public static Builder builder() {
        return new Builder();
    }

    public OcflS3Client(S3Client s3Client, String bucket) {
        this(s3Client, bucket, null, null, null);
    }

    public OcflS3Client(S3Client s3Client, String bucket, String prefix, BiConsumer<String, PutObjectRequest.Builder> putObjectModifier, BiConsumer<String, CreateMultipartUploadRequest.Builder> createMultipartModifier) {
        this.s3Client = (S3Client)Enforce.notNull((Object)s3Client, (String)"s3Client cannot be null");
        this.bucket = Enforce.notBlank((String)bucket, (String)"bucket cannot be blank");
        this.repoPrefix = OcflS3Client.sanitizeRepoPrefix(prefix == null ? "" : prefix);
        this.keyBuilder = CloudObjectKey.builder().prefix(this.repoPrefix);
        this.putObjectModifier = putObjectModifier != null ? putObjectModifier : (k, b) -> {};
        this.createMultipartModifier = createMultipartModifier != null ? createMultipartModifier : (k, b) -> {};
    }

    private static String sanitizeRepoPrefix(String repoPrefix) {
        return repoPrefix.substring(0, OcflS3Client.indexLastNonSlash(repoPrefix));
    }

    private static int indexLastNonSlash(String string) {
        for (int i = string.length(); i > 0; --i) {
            if (string.charAt(i - 1) == '/') continue;
            return i;
        }
        return 0;
    }

    public String bucket() {
        return this.bucket;
    }

    public String prefix() {
        return this.repoPrefix;
    }

    public CloudObjectKey uploadFile(Path srcPath, String dstPath) {
        return this.uploadFile(srcPath, dstPath, null);
    }

    public CloudObjectKey uploadFile(Path srcPath, String dstPath, String contentType) {
        long fileSize = UncheckedFiles.size((Path)srcPath);
        CloudObjectKey dstKey = this.keyBuilder.buildFromPath(dstPath);
        if (fileSize >= 0x50000000000L) {
            throw new OcflInputException(String.format("Cannot store file %s because it exceeds the maximum file size.", srcPath));
        }
        if (fileSize > (long)this.maxPartBytes) {
            this.multipartUpload(srcPath, dstKey, fileSize, contentType);
        } else {
            LOG.debug("Uploading {} to bucket {} key {} size {}", new Object[]{srcPath, this.bucket, dstKey, fileSize});
            PutObjectRequest.Builder builder = PutObjectRequest.builder().contentType(contentType);
            this.putObjectModifier.accept(dstKey.getKey(), builder);
            this.s3Client.putObject((PutObjectRequest)builder.bucket(this.bucket).key(dstKey.getKey()).contentLength(Long.valueOf(fileSize)).build(), srcPath);
        }
        return dstKey;
    }

    private void multipartUpload(Path srcPath, CloudObjectKey dstKey, long fileSize, String contentType) {
        int partSize = this.determinePartSize(fileSize);
        LOG.debug("Multipart upload of {} to bucket {} key {}. File size: {}; part size: {}", new Object[]{srcPath, this.bucket, dstKey, fileSize, partSize});
        String uploadId = this.beginMultipartUpload(dstKey, contentType);
        ArrayList<CompletedPart> completedParts = new ArrayList<CompletedPart>();
        try {
            try (FileChannel channel = FileChannel.open(srcPath, StandardOpenOption.READ);){
                ByteBuffer buffer = ByteBuffer.allocate(partSize);
                int i = 1;
                while (channel.read(buffer) > 0) {
                    buffer.flip();
                    UploadPartResponse partResponse = this.s3Client.uploadPart((UploadPartRequest)UploadPartRequest.builder().bucket(this.bucket).key(dstKey.getKey()).uploadId(uploadId).partNumber(Integer.valueOf(i)).build(), RequestBody.fromByteBuffer((ByteBuffer)buffer));
                    completedParts.add((CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(i)).eTag(partResponse.eTag()).build());
                    buffer.clear();
                    ++i;
                }
            }
            catch (IOException e) {
                throw new OcflIOException((Exception)e);
            }
            this.completeMultipartUpload(uploadId, dstKey, completedParts);
        }
        catch (RuntimeException e) {
            this.abortMultipartUpload(uploadId, dstKey);
            throw e;
        }
    }

    public CloudObjectKey uploadBytes(String dstPath, byte[] bytes, String contentType) {
        CloudObjectKey dstKey = this.keyBuilder.buildFromPath(dstPath);
        LOG.debug("Writing string to bucket {} key {}", (Object)this.bucket, (Object)dstKey);
        PutObjectRequest.Builder builder = PutObjectRequest.builder().contentType(contentType);
        this.putObjectModifier.accept(dstKey.getKey(), builder);
        this.s3Client.putObject((PutObjectRequest)builder.bucket(this.bucket).key(dstKey.getKey()).build(), RequestBody.fromBytes((byte[])bytes));
        return dstKey;
    }

    public CloudObjectKey copyObject(String srcPath, String dstPath) {
        CloudObjectKey srcKey = this.keyBuilder.buildFromPath(srcPath);
        CloudObjectKey dstKey = this.keyBuilder.buildFromPath(dstPath);
        LOG.debug("Copying {} to {} in bucket {}", new Object[]{srcKey, dstKey, this.bucket});
        try {
            this.s3Client.copyObject((CopyObjectRequest)CopyObjectRequest.builder().destinationBucket(this.bucket).destinationKey(dstKey.getKey()).copySource(this.keyWithBucketName(srcKey.getKey())).build());
        }
        catch (NoSuchKeyException e) {
            throw new KeyNotFoundException((Throwable)e);
        }
        catch (SdkException e) {
            if (e.getMessage().contains("copy source is larger than the maximum allowable size")) {
                this.multipartCopy(srcKey, dstKey);
            }
            throw e;
        }
        return dstKey;
    }

    private void multipartCopy(CloudObjectKey srcKey, CloudObjectKey dstKey) {
        HeadObjectResponse head = this.headObject(srcKey);
        Long fileSize = head.contentLength();
        int partSize = this.determinePartSize(fileSize);
        LOG.debug("Multipart copy of {} to {} in bucket {}: File size {}; part size: {}", new Object[]{srcKey, dstKey, this.bucket, fileSize, partSize});
        String uploadId = this.beginMultipartUpload(dstKey, null);
        try {
            ArrayList<CompletedPart> completedParts = new ArrayList<CompletedPart>();
            int part = 1;
            long position = 0L;
            while (position < fileSize) {
                long end = Math.min(fileSize - 1L, (long)(part * partSize - 1));
                UploadPartCopyResponse partResponse = this.s3Client.uploadPartCopy((UploadPartCopyRequest)UploadPartCopyRequest.builder().bucket(this.bucket).key(dstKey.getKey()).copySource(this.keyWithBucketName(srcKey.getKey())).partNumber(Integer.valueOf(part)).uploadId(uploadId).copySourceRange(String.format("bytes=%s-%s", position, end)).build());
                completedParts.add((CompletedPart)CompletedPart.builder().partNumber(Integer.valueOf(part)).eTag(partResponse.copyPartResult().eTag()).build());
                ++part;
                position = end + 1L;
            }
            this.completeMultipartUpload(uploadId, dstKey, completedParts);
        }
        catch (RuntimeException e) {
            this.abortMultipartUpload(uploadId, dstKey);
            throw e;
        }
    }

    private HeadObjectResponse headObject(CloudObjectKey key) {
        return this.s3Client.headObject((HeadObjectRequest)HeadObjectRequest.builder().bucket(this.bucket).key(key.getKey()).build());
    }

    public Path downloadFile(String srcPath, Path dstPath) {
        CloudObjectKey srcKey = this.keyBuilder.buildFromPath(srcPath);
        LOG.debug("Downloading bucket {} key {} to {}", new Object[]{this.bucket, srcKey, dstPath});
        try {
            this.s3Client.getObject((GetObjectRequest)GetObjectRequest.builder().bucket(this.bucket).key(srcKey.getKey()).build(), dstPath);
        }
        catch (NoSuchKeyException e) {
            throw new KeyNotFoundException((Throwable)e);
        }
        return dstPath;
    }

    public InputStream downloadStream(String srcPath) {
        CloudObjectKey srcKey = this.keyBuilder.buildFromPath(srcPath);
        LOG.debug("Streaming bucket {} key {}", (Object)this.bucket, (Object)srcKey);
        try {
            return this.s3Client.getObject((GetObjectRequest)GetObjectRequest.builder().bucket(this.bucket).key(srcKey.getKey()).build());
        }
        catch (NoSuchKeyException e) {
            throw new KeyNotFoundException(String.format("Key %s not found in bucket %s.", srcKey, this.bucket), (Throwable)e);
        }
    }

    public String downloadString(String srcPath) {
        String string;
        block8: {
            InputStream stream = this.downloadStream(srcPath);
            try {
                string = new String(stream.readAllBytes(), StandardCharsets.UTF_8);
                if (stream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new OcflIOException((Exception)e);
                }
            }
            stream.close();
        }
        return string;
    }

    public HeadResult head(String path) {
        CloudObjectKey key = this.keyBuilder.buildFromPath(path);
        try {
            HeadObjectResponse s3Result = this.s3Client.headObject((HeadObjectRequest)HeadObjectRequest.builder().bucket(this.bucket).key(key.getKey()).build());
            return new HeadResult().setContentEncoding(s3Result.contentEncoding()).setContentLength(s3Result.contentLength()).setETag(s3Result.eTag()).setLastModified(s3Result.lastModified());
        }
        catch (NoSuchKeyException e) {
            throw new KeyNotFoundException(String.format("Key %s not found in bucket %s.", key, this.bucket), (Throwable)e);
        }
    }

    public ListResult list(String prefix) {
        CloudObjectKey prefixedPrefix = this.keyBuilder.buildFromPath(prefix);
        return this.toListResult(ListObjectsV2Request.builder().bucket(this.bucket).prefix(prefixedPrefix.getKey()));
    }

    public ListResult listDirectory(String path) {
        Object prefix = this.keyBuilder.buildFromPath(path).getKey();
        if (!((String)prefix).isEmpty() && !((String)prefix).endsWith("/")) {
            prefix = (String)prefix + "/";
        }
        LOG.debug("Listing directory {} in bucket {}", prefix, (Object)this.bucket);
        return this.toListResult(ListObjectsV2Request.builder().bucket(this.bucket).delimiter("/").prefix((String)prefix));
    }

    public void deletePath(String path) {
        LOG.debug("Deleting path {} in bucket {}", (Object)path, (Object)this.bucket);
        List<CloudObjectKey> keys = this.list(path).getObjects().stream().map(ListResult.ObjectListing::getKey).collect(Collectors.toList());
        this.deleteObjectsInternal(keys);
    }

    public void deleteObjects(Collection<String> objectPaths) {
        if (!objectPaths.isEmpty()) {
            List<CloudObjectKey> objectKeys = objectPaths.stream().filter(Objects::nonNull).map(arg_0 -> ((CloudObjectKey.Builder)this.keyBuilder).buildFromPath(arg_0)).collect(Collectors.toList());
            this.deleteObjectsInternal(objectKeys);
        }
    }

    private void deleteObjectsInternal(Collection<CloudObjectKey> objectKeys) {
        LOG.debug("Deleting objects in bucket {}: {}", (Object)this.bucket, objectKeys);
        if (!objectKeys.isEmpty()) {
            List objectIds = objectKeys.stream().map(key -> (ObjectIdentifier)ObjectIdentifier.builder().key(key.getKey()).build()).collect(Collectors.toList());
            this.s3Client.deleteObjects((DeleteObjectsRequest)DeleteObjectsRequest.builder().bucket(this.bucket).delete((Delete)Delete.builder().objects(objectIds).build()).build());
        }
    }

    public void safeDeleteObjects(String ... objectPaths) {
        this.safeDeleteObjects(Arrays.asList(objectPaths));
    }

    public void safeDeleteObjects(Collection<String> objectPaths) {
        try {
            this.deleteObjects(objectPaths);
        }
        catch (RuntimeException e) {
            LOG.error("Failed to cleanup objects in bucket {}: {}", new Object[]{this.bucket, objectPaths, e});
        }
    }

    public boolean bucketExists() {
        try {
            this.s3Client.headBucket((HeadBucketRequest)HeadBucketRequest.builder().bucket(this.bucket).build());
            return true;
        }
        catch (NoSuchBucketException e) {
            return false;
        }
    }

    private String beginMultipartUpload(CloudObjectKey key, String contentType) {
        CreateMultipartUploadRequest.Builder builder = CreateMultipartUploadRequest.builder().contentType(contentType);
        this.createMultipartModifier.accept(key.getKey(), builder);
        return this.s3Client.createMultipartUpload((CreateMultipartUploadRequest)builder.bucket(this.bucket).key(key.getKey()).build()).uploadId();
    }

    private void completeMultipartUpload(String uploadId, CloudObjectKey key, List<CompletedPart> parts) {
        this.s3Client.completeMultipartUpload((CompleteMultipartUploadRequest)CompleteMultipartUploadRequest.builder().bucket(this.bucket).key(key.getKey()).uploadId(uploadId).multipartUpload((CompletedMultipartUpload)CompletedMultipartUpload.builder().parts(parts).build()).build());
    }

    private void abortMultipartUpload(String uploadId, CloudObjectKey key) {
        try {
            this.s3Client.abortMultipartUpload((AbortMultipartUploadRequest)AbortMultipartUploadRequest.builder().bucket(this.bucket).key(key.getKey()).uploadId(uploadId).build());
        }
        catch (RuntimeException e) {
            LOG.error("Failed to abort multipart upload. Bucket: {}; Key: {}; Upload Id: {}", new Object[]{this.bucket, key, uploadId, e});
        }
    }

    private String keyWithBucketName(String key) {
        return SdkHttpUtils.urlEncode((String)String.format("%s/%s", this.bucket, key));
    }

    private int determinePartSize(long fileSize) {
        int partSize = this.partSizeBytes;
        int maxParts = 100;
        while (fileSize / (long)partSize > (long)maxParts) {
            if ((partSize += 10) <= this.maxPartBytes) continue;
            maxParts += 100;
            partSize /= 2;
        }
        return partSize;
    }

    private ListResult toListResult(ListObjectsV2Request.Builder requestBuilder) {
        ListObjectsV2Response result = this.s3Client.listObjectsV2((ListObjectsV2Request)requestBuilder.build());
        int prefixLength = this.prefixLength(result.prefix());
        int repoPrefixLength = this.repoPrefix.isBlank() ? 0 : this.repoPrefix.length() + 1;
        List<ListResult.ObjectListing> objects = this.toObjectListings(result, prefixLength);
        List<ListResult.DirectoryListing> dirs = this.toDirectoryListings(result, repoPrefixLength);
        while (result.isTruncated().booleanValue()) {
            result = this.s3Client.listObjectsV2((ListObjectsV2Request)requestBuilder.continuationToken(result.nextContinuationToken()).build());
            objects.addAll(this.toObjectListings(result, prefixLength));
            dirs.addAll(this.toDirectoryListings(result, repoPrefixLength));
        }
        return new ListResult().setObjects(objects).setDirectories(dirs);
    }

    private List<ListResult.ObjectListing> toObjectListings(ListObjectsV2Response result, int prefixLength) {
        return result.contents().stream().map(o -> {
            String key = o.key();
            return new ListResult.ObjectListing().setKey(this.keyBuilder.buildFromKey(key)).setKeySuffix(key.substring(prefixLength));
        }).collect(Collectors.toList());
    }

    private List<ListResult.DirectoryListing> toDirectoryListings(ListObjectsV2Response result, int repoPrefixLength) {
        return result.commonPrefixes().stream().filter(p -> p.prefix() != null).map(p -> {
            String path = p.prefix();
            return new ListResult.DirectoryListing().setPath(path.substring(repoPrefixLength));
        }).collect(Collectors.toList());
    }

    private int prefixLength(String prefix) {
        int prefixLength = 0;
        if (prefix != null && !prefix.isEmpty()) {
            prefixLength = prefix.length();
            if (!prefix.endsWith("/")) {
                ++prefixLength;
            }
        }
        return prefixLength;
    }

    @VisibleForTesting
    void setMaxPartBytes(int maxPartBytes) {
        this.maxPartBytes = maxPartBytes;
    }

    @VisibleForTesting
    void setPartSizeBytes(int partSizeBytes) {
        this.partSizeBytes = partSizeBytes;
    }

    public static class Builder {
        private S3Client s3Client;
        private String bucket;
        private String repoPrefix;
        private BiConsumer<String, PutObjectRequest.Builder> putObjectModifier;
        private BiConsumer<String, CreateMultipartUploadRequest.Builder> createMultipartModifier;

        public Builder s3Client(S3Client s3Client) {
            this.s3Client = (S3Client)Enforce.notNull((Object)s3Client, (String)"s3Client cannot be null");
            return this;
        }

        public Builder bucket(String bucket) {
            this.bucket = Enforce.notBlank((String)bucket, (String)"bucket cannot be blank");
            return this;
        }

        public Builder repoPrefix(String repoPrefix) {
            this.repoPrefix = repoPrefix;
            return this;
        }

        public Builder putObjectModifier(BiConsumer<String, PutObjectRequest.Builder> putObjectModifier) {
            this.putObjectModifier = putObjectModifier;
            return this;
        }

        public Builder createMultipartModifier(BiConsumer<String, CreateMultipartUploadRequest.Builder> createMultipartModifier) {
            this.createMultipartModifier = createMultipartModifier;
            return this;
        }

        public OcflS3Client build() {
            return new OcflS3Client(this.s3Client, this.bucket, this.repoPrefix, this.putObjectModifier, this.createMultipartModifier);
        }
    }
}

