/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.core.storage.filesystem;

import edu.wisc.library.ocfl.api.OcflFileRetriever;
import edu.wisc.library.ocfl.api.exception.CorruptObjectException;
import edu.wisc.library.ocfl.api.exception.FixityCheckException;
import edu.wisc.library.ocfl.api.exception.NotFoundException;
import edu.wisc.library.ocfl.api.exception.ObjectOutOfSyncException;
import edu.wisc.library.ocfl.api.exception.OcflIOException;
import edu.wisc.library.ocfl.api.exception.OcflNoSuchFileException;
import edu.wisc.library.ocfl.api.exception.OcflStateException;
import edu.wisc.library.ocfl.api.io.FixityCheckInputStream;
import edu.wisc.library.ocfl.api.model.DigestAlgorithm;
import edu.wisc.library.ocfl.api.model.ObjectVersionId;
import edu.wisc.library.ocfl.api.model.OcflVersion;
import edu.wisc.library.ocfl.api.model.ValidationResults;
import edu.wisc.library.ocfl.api.model.VersionNum;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.ObjectPaths;
import edu.wisc.library.ocfl.core.extension.OcflExtensionConfig;
import edu.wisc.library.ocfl.core.extension.storage.layout.OcflStorageLayoutExtension;
import edu.wisc.library.ocfl.core.inventory.SidecarMapper;
import edu.wisc.library.ocfl.core.model.Inventory;
import edu.wisc.library.ocfl.core.model.RevisionNum;
import edu.wisc.library.ocfl.core.model.Version;
import edu.wisc.library.ocfl.core.path.constraint.LogicalPathConstraints;
import edu.wisc.library.ocfl.core.path.constraint.PathConstraintProcessor;
import edu.wisc.library.ocfl.core.storage.AbstractOcflStorage;
import edu.wisc.library.ocfl.core.storage.ObjectProperties;
import edu.wisc.library.ocfl.core.storage.filesystem.FileSystemOcflFileRetriever;
import edu.wisc.library.ocfl.core.storage.filesystem.FileSystemOcflObjectRootDirIterator;
import edu.wisc.library.ocfl.core.storage.filesystem.FileSystemOcflStorageBuilder;
import edu.wisc.library.ocfl.core.storage.filesystem.FileSystemOcflStorageInitializer;
import edu.wisc.library.ocfl.core.util.FileUtil;
import edu.wisc.library.ocfl.core.util.NamasteTypeFile;
import edu.wisc.library.ocfl.core.util.UncheckedFiles;
import edu.wisc.library.ocfl.core.validation.Validator;
import edu.wisc.library.ocfl.core.validation.storage.FileSystemStorage;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.temporal.ChronoUnit;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.Policy;
import net.jodah.failsafe.RetryPolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemOcflStorage
extends AbstractOcflStorage {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemOcflStorage.class);
    private final PathConstraintProcessor logicalPathConstraints;
    private final Path repositoryRoot;
    private final FileSystemOcflStorageInitializer initializer;
    private final Validator validator;
    private final boolean verifyInventoryDigest;
    private OcflStorageLayoutExtension storageLayoutExtension;
    private final RetryPolicy<Void> ioRetry;

    public static FileSystemOcflStorageBuilder builder() {
        return new FileSystemOcflStorageBuilder();
    }

    public FileSystemOcflStorage(Path repositoryRoot, boolean verifyInventoryDigest, FileSystemOcflStorageInitializer initializer) {
        this.repositoryRoot = (Path)Enforce.notNull((Object)repositoryRoot, (String)"repositoryRoot cannot be null");
        this.initializer = (FileSystemOcflStorageInitializer)Enforce.notNull((Object)initializer, (String)"initializer cannot be null");
        this.verifyInventoryDigest = verifyInventoryDigest;
        this.logicalPathConstraints = LogicalPathConstraints.constraintsWithBackslashCheck();
        this.ioRetry = ((RetryPolicy)new RetryPolicy().handle(new Class[]{UncheckedIOException.class, IOException.class})).withBackoff(5L, 200L, ChronoUnit.MILLIS, 1.5).withMaxRetries(5);
        this.validator = new Validator(new FileSystemStorage(repositoryRoot));
    }

    @Override
    public Inventory loadInventory(String objectId) {
        this.ensureOpen();
        LOG.debug("Load inventory for object <{}>", (Object)objectId);
        Inventory inventory = null;
        String objectRootPathStr = this.objectRootPath(objectId);
        Path objectRootPathAbsolute = this.repositoryRoot.resolve(objectRootPathStr);
        ObjectProperties objectProps = this.examineObject(objectRootPathAbsolute);
        if (objectProps.getOcflVersion() != null) {
            if (objectProps.getDigestAlgorithm() == null) {
                throw new CorruptObjectException(String.format("Object %s is missing its root sidecar file", objectId));
            }
            boolean hasMutableHead = false;
            if (objectProps.hasExtensions()) {
                hasMutableHead = this.loadObjectExtensions(objectRootPathAbsolute).contains("0005-mutable-head");
            }
            if (hasMutableHead) {
                Path mutableHeadInventoryPath = ObjectPaths.mutableHeadInventoryPath(objectRootPathAbsolute);
                inventory = this.parseMutableHeadInventory(objectRootPathStr, objectRootPathAbsolute, objectProps.getDigestAlgorithm(), mutableHeadInventoryPath);
                this.ensureRootObjectHasNotChanged(objectId, objectRootPathAbsolute);
            } else {
                inventory = this.parseInventory(objectRootPathStr, objectProps.getDigestAlgorithm(), ObjectPaths.inventoryPath(objectRootPathAbsolute));
            }
            if (!Objects.equals(objectId, inventory.getId())) {
                throw new CorruptObjectException(String.format("Expected object at %s to have id %s. Found: %s", objectRootPathStr, objectId, inventory.getId()));
            }
        }
        return inventory;
    }

    @Override
    public byte[] getInventoryBytes(String objectId, VersionNum versionNum) {
        this.ensureOpen();
        Enforce.notBlank((String)objectId, (String)"objectId cannot be blank");
        Enforce.notNull((Object)versionNum, (String)"versionNum cannot be null");
        LOG.debug("Loading inventory bytes for object {} version {}", (Object)objectId, (Object)versionNum);
        Path objectRootPath = this.repositoryRoot.resolve(this.objectRootPath(objectId));
        Path versionPath = objectRootPath.resolve(versionNum.toString());
        try {
            return Files.readAllBytes(ObjectPaths.inventoryPath(versionPath));
        }
        catch (NoSuchFileException e) {
            try {
                byte[] bytes = Files.readAllBytes(ObjectPaths.mutableHeadInventoryPath(objectRootPath));
                Inventory inv = this.inventoryMapper.readMutableHeadNoDigest("root", RevisionNum.R1, new ByteArrayInputStream(bytes));
                if (versionNum.equals((Object)inv.getHead())) {
                    return bytes;
                }
            }
            catch (NoSuchFileException bytes) {
            }
            catch (IOException e2) {
                throw new OcflIOException(e2);
            }
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
        throw new NotFoundException(String.format("No inventory could be found for object %s version %s", objectId, versionNum));
    }

    @Override
    public Stream<String> listObjectIds() {
        LOG.debug("List object ids");
        return this.findOcflObjectRootDirs(this.repositoryRoot).map(rootPath -> {
            String relativeRootStr = FileUtil.pathToStringStandardSeparator(this.repositoryRoot.relativize((Path)rootPath));
            Path inventoryPath = ObjectPaths.inventoryPath(rootPath);
            Inventory inventory = this.inventoryMapper.readNoDigest(relativeRootStr, inventoryPath);
            return inventory.getId();
        });
    }

    @Override
    public void storeNewVersion(Inventory inventory, Path stagingDir) {
        this.ensureOpen();
        LOG.debug("Store new version of object <{}> version <{}> revision <{}> from staging directory <{}>", new Object[]{inventory.getId(), inventory.getHead(), inventory.getRevisionNum(), stagingDir});
        Path objectRootPath = this.objectRootPathFull(inventory.getId());
        ObjectPaths.ObjectRoot objectRoot = ObjectPaths.objectRoot(inventory, objectRootPath);
        if (inventory.hasMutableHead()) {
            this.storeNewMutableHeadVersion(inventory, objectRoot, stagingDir);
        } else {
            this.storeNewImmutableVersion(inventory, objectRoot, stagingDir);
        }
    }

    @Override
    public Map<String, OcflFileRetriever> getObjectStreams(Inventory inventory, VersionNum versionNum) {
        this.ensureOpen();
        LOG.debug("Get file streams for object <{}> version <{}>", (Object)inventory.getId(), (Object)versionNum);
        Path objectRootPath = this.objectRootPathFull(inventory.getId());
        Version version = inventory.ensureVersion(versionNum);
        DigestAlgorithm algorithm = inventory.getDigestAlgorithm();
        HashMap<String, OcflFileRetriever> map = new HashMap<String, OcflFileRetriever>(version.getState().size());
        version.getState().forEach((digest, paths) -> {
            Path srcPath = objectRootPath.resolve(inventory.ensureContentPath((String)digest));
            paths.forEach(path -> map.put((String)path, new FileSystemOcflFileRetriever(srcPath, algorithm, (String)digest)));
        });
        return map;
    }

    @Override
    public void reconstructObjectVersion(Inventory inventory, VersionNum versionNum, Path stagingDir) {
        this.ensureOpen();
        LOG.debug("Reconstruct object <{}> version <{}> in directory <{}>", new Object[]{inventory.getId(), versionNum, stagingDir});
        Path objectRootPath = this.objectRootPathFull(inventory.getId());
        Version version = inventory.ensureVersion(versionNum);
        String digestAlgorithm = inventory.getDigestAlgorithm().getJavaStandardName();
        version.getState().forEach((id, files) -> {
            Path srcPath = objectRootPath.resolve(inventory.ensureContentPath((String)id));
            for (String logicalPath : files) {
                this.logicalPathConstraints.apply(logicalPath);
                Path destination = Paths.get(FileUtil.pathJoinFailEmpty(stagingDir.toString(), logicalPath), new String[0]);
                UncheckedFiles.createDirectories(destination.getParent());
                try (FixityCheckInputStream stream = new FixityCheckInputStream((InputStream)new BufferedInputStream(Files.newInputStream(srcPath, new OpenOption[0])), digestAlgorithm, id);){
                    Files.copy((InputStream)stream, destination, new CopyOption[0]);
                    stream.checkFixity();
                }
                catch (IOException e) {
                    throw new OcflIOException(e);
                }
                catch (FixityCheckException e) {
                    throw new FixityCheckException(String.format("File %s in object %s failed its fixity check.", logicalPath, inventory.getId()), (Throwable)e);
                }
            }
        });
    }

    @Override
    public void purgeObject(String objectId) {
        this.ensureOpen();
        LOG.info("Purge object <{}>", (Object)objectId);
        Path objectRootPath = this.objectRootPathFull(objectId);
        try {
            FileUtil.deleteDirectory(objectRootPath);
        }
        catch (Exception e) {
            throw new CorruptObjectException(String.format("Failed to purge object %s at %s. The object may need to be deleted manually.", objectId, objectRootPath), (Throwable)e);
        }
        if (Files.exists(objectRootPath.getParent(), new LinkOption[0])) {
            try {
                FileUtil.deleteDirAndParentsIfEmpty(objectRootPath.getParent());
            }
            catch (OcflIOException e) {
                LOG.error(String.format("Failed to cleanup all empty directories in path %s. There may be empty directories remaining in the OCFL storage hierarchy.", objectRootPath), (Throwable)e);
            }
        }
    }

    @Override
    public void rollbackToVersion(Inventory inventory, VersionNum versionNum) {
        this.ensureOpen();
        LOG.info("Rollback object <{}> to version {}", (Object)inventory.getId(), (Object)versionNum);
        Path objectRootPath = this.objectRootPathFull(inventory.getId());
        ObjectPaths.ObjectRoot objectRoot = ObjectPaths.objectRoot(inventory, objectRootPath);
        ObjectPaths.VersionRoot versionRoot = objectRoot.version(versionNum);
        try {
            this.copyInventory(versionRoot, objectRoot);
        }
        catch (Exception e) {
            try {
                this.copyInventory(objectRoot.headVersion(), objectRoot);
            }
            catch (Exception e1) {
                LOG.error("Failed to rollback inventory at {}", (Object)objectRoot.inventoryFile(), (Object)e1);
            }
            throw e;
        }
        try {
            VersionNum currentVersion = inventory.getHead();
            while (currentVersion.compareTo(versionNum) > 0) {
                LOG.info("Purging object {} version {}", (Object)inventory.getId(), (Object)currentVersion);
                Path currentVersionPath = objectRoot.versionPath(currentVersion);
                FileUtil.deleteDirectory(currentVersionPath);
                currentVersion = currentVersion.previousVersionNum();
            }
            FileUtil.deleteDirectory(objectRoot.mutableHeadExtensionPath());
        }
        catch (Exception e) {
            throw new CorruptObjectException(String.format("Object %s was corrupted while attempting to rollback to version %s. It must be manually remediated.", inventory.getId(), versionNum), (Throwable)e);
        }
    }

    @Override
    public void commitMutableHead(Inventory oldInventory, Inventory newInventory, Path stagingDir) {
        this.ensureOpen();
        LOG.debug("Commit mutable HEAD on object <{}>", (Object)newInventory.getId());
        Path objectRootPath = this.objectRootPathFull(newInventory.getId());
        ObjectPaths.ObjectRoot objectRoot = ObjectPaths.objectRoot(newInventory, objectRootPath);
        this.ensureRootObjectHasNotChanged(newInventory, objectRoot);
        if (!Files.exists(objectRoot.mutableHeadVersion().inventoryFile(), new LinkOption[0])) {
            throw new ObjectOutOfSyncException(String.format("Cannot commit mutable HEAD of object %s because a mutable HEAD does not exist.", newInventory.getId()));
        }
        ObjectPaths.VersionRoot versionRoot = objectRoot.headVersion();
        ObjectPaths.VersionRoot stagingRoot = ObjectPaths.version(newInventory, stagingDir);
        this.deleteMutableHeadFilesNotInManifest(oldInventory, objectRoot, versionRoot);
        this.moveToVersionDirectory(newInventory, objectRoot.mutableHeadPath(), versionRoot);
        try {
            try {
                this.copyInventoryToRootWithRollback(stagingRoot, objectRoot, newInventory);
                this.copyInventory(stagingRoot, versionRoot);
            }
            catch (RuntimeException e) {
                try {
                    FileUtil.moveDirectory(versionRoot.path(), objectRoot.mutableHeadPath());
                }
                catch (RuntimeException | FileAlreadyExistsException e1) {
                    LOG.error("Failed to move {} back to {}", new Object[]{versionRoot.path(), objectRoot.mutableHeadPath(), e1});
                }
                throw e;
            }
            this.deleteEmptyDirs(versionRoot.contentPath());
        }
        catch (RuntimeException e) {
            FileUtil.safeDeleteDirectory(versionRoot.path());
            throw e;
        }
        try {
            FileUtil.deleteDirectory(objectRoot.mutableHeadExtensionPath());
        }
        catch (RuntimeException e) {
            LOG.error("Failed to cleanup mutable HEAD of object {} at {}. It must be deleted manually.", new Object[]{newInventory.getId(), objectRoot.mutableHeadExtensionPath(), e});
        }
    }

    @Override
    public void purgeMutableHead(String objectId) {
        this.ensureOpen();
        LOG.info("Purge mutable HEAD on object <{}>", (Object)objectId);
        Path objectRootPath = this.objectRootPathFull(objectId);
        Path extensionRoot = ObjectPaths.mutableHeadExtensionRoot(objectRootPath);
        if (Files.exists(extensionRoot, new LinkOption[0])) {
            try (Stream<Path> paths = Files.walk(extensionRoot, new FileVisitOption[0]);){
                paths.sorted(Comparator.reverseOrder()).forEach(f -> {
                    try {
                        Files.delete(f);
                    }
                    catch (IOException e) {
                        throw new OcflIOException(String.format("Failed to delete file %s while purging mutable HEAD of object %s. The purge failed and the mutable HEAD may need to be deleted manually.", f, objectId), e);
                    }
                });
            }
            catch (IOException e) {
                throw new OcflIOException(String.format("Failed to purge mutable HEAD of object %s at %s. The object may need to be deleted manually.", objectId, extensionRoot), e);
            }
        }
    }

    @Override
    public boolean containsObject(String objectId) {
        this.ensureOpen();
        ObjectProperties objectProps = this.examineObject(this.objectRootPathFull(objectId));
        boolean exists = objectProps.getOcflVersion() != null;
        LOG.debug("OCFL repository contains object <{}>: {}", (Object)objectId, (Object)exists);
        return exists;
    }

    @Override
    public String objectRootPath(String objectId) {
        this.ensureOpen();
        String objectRootPath = this.storageLayoutExtension.mapObjectId(objectId);
        LOG.debug("Object root path for object <{}>: {}", (Object)objectId, (Object)objectRootPath);
        return objectRootPath;
    }

    @Override
    public void exportVersion(ObjectVersionId objectVersionId, Path outputPath) {
        this.ensureOpen();
        Enforce.notNull((Object)objectVersionId.getVersionNum(), (String)"versionNum cannot be null");
        Path objectRootPath = this.objectRootPathFull(objectVersionId.getObjectId());
        if (Files.notExists(objectRootPath, new LinkOption[0])) {
            throw new NotFoundException(String.format("Object %s was not found.", objectVersionId.getObjectId()));
        }
        Path versionRoot = objectRootPath.resolve(objectVersionId.getVersionNum().toString());
        if (Files.notExists(versionRoot, new LinkOption[0])) {
            throw new NotFoundException(String.format("Object %s version %s was not found.", objectVersionId.getObjectId(), objectVersionId.getVersionNum()));
        }
        LOG.debug("Copying <{}> to <{}>", (Object)versionRoot, (Object)outputPath);
        FileUtil.recursiveCopy(versionRoot, outputPath, new StandardCopyOption[0]);
    }

    @Override
    public void exportObject(String objectId, Path outputPath) {
        this.ensureOpen();
        Path objectRootPath = this.objectRootPathFull(objectId);
        if (Files.notExists(objectRootPath, new LinkOption[0])) {
            throw new NotFoundException(String.format("Object %s was not found.", objectId));
        }
        LOG.debug("Copying <{}> to <{}>", (Object)objectRootPath, (Object)outputPath);
        FileUtil.recursiveCopy(objectRootPath, outputPath, new StandardCopyOption[0]);
    }

    @Override
    public void importObject(String objectId, Path objectPath) {
        this.ensureOpen();
        Path objectRootPath = this.objectRootPathFull(objectId);
        LOG.debug("Importing <{}> to <{}>", (Object)objectId, (Object)objectRootPath);
        UncheckedFiles.createDirectories(objectRootPath.getParent());
        try {
            FileUtil.moveDirectory(objectPath, objectRootPath);
        }
        catch (FileAlreadyExistsException e) {
            throw new ObjectOutOfSyncException(String.format("Cannot import object %s because the object already exists.", objectId));
        }
        catch (RuntimeException e) {
            try {
                this.purgeObject(objectId);
            }
            catch (RuntimeException e1) {
                LOG.error("Failed to rollback object {} import", (Object)objectId, (Object)e1);
            }
            throw e;
        }
    }

    @Override
    public ValidationResults validateObject(String objectId, boolean contentFixityCheck) {
        this.ensureOpen();
        String objectRoot = this.objectRootPath(objectId);
        Path objectRootPath = this.repositoryRoot.resolve(objectRoot);
        if (Files.notExists(objectRootPath, new LinkOption[0])) {
            throw new NotFoundException(String.format("Object %s was not found.", objectId));
        }
        LOG.debug("Validating object <{}> at <{}>", (Object)objectId, (Object)objectRoot);
        return this.validator.validateObject(objectRoot, contentFixityCheck);
    }

    @Override
    protected void doInitialize(OcflExtensionConfig layoutConfig) {
        this.storageLayoutExtension = this.initializer.initializeStorage(this.repositoryRoot, this.ocflVersion, layoutConfig, this.supportEvaluator);
    }

    @Override
    public void close() {
        LOG.debug("Closing " + this.getClass().getName());
    }

    private Path objectRootPathFull(String objectId) {
        return this.repositoryRoot.resolve(this.storageLayoutExtension.mapObjectId(objectId));
    }

    private Inventory parseInventory(String objectRootPath, DigestAlgorithm digestAlgorithm, Path inventoryPath) {
        String expectedDigest;
        Inventory inventory = this.inventoryMapper.read(objectRootPath, digestAlgorithm, inventoryPath);
        if (this.verifyInventoryDigest && !(expectedDigest = SidecarMapper.readDigestRequired(ObjectPaths.inventorySidecarPath(inventoryPath.getParent(), inventory))).equalsIgnoreCase(inventory.getInventoryDigest())) {
            throw new CorruptObjectException(String.format("Root inventory in object %s does not match expected digest", inventory.getId()));
        }
        return inventory;
    }

    private Inventory parseMutableHeadInventory(String objectRootPath, Path objectRootPathAbsolute, DigestAlgorithm digestAlgorithm, Path inventoryPath) {
        String expectedDigest;
        RevisionNum revisionNum = this.identifyLatestRevision(objectRootPathAbsolute);
        Inventory inventory = this.inventoryMapper.readMutableHead(objectRootPath, revisionNum, digestAlgorithm, inventoryPath);
        if (this.verifyInventoryDigest && !(expectedDigest = SidecarMapper.readDigestRequired(ObjectPaths.inventorySidecarPath(inventoryPath.getParent(), inventory))).equalsIgnoreCase(inventory.getInventoryDigest())) {
            throw new CorruptObjectException(String.format("Mutable HEAD inventory in object %s does not match expected digest", inventory.getId()));
        }
        return inventory;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private RevisionNum identifyLatestRevision(Path objectRootPath) {
        Path revisionsPath = ObjectPaths.mutableHeadRevisionsPath(objectRootPath);
        try (Stream<Path> files = Files.list(revisionsPath);){
            Optional<RevisionNum> result = files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).map(Path::getFileName).map(Path::toString).filter(RevisionNum::isRevisionNum).map(RevisionNum::fromString).max(Comparator.naturalOrder());
            if (result.isEmpty()) {
                RevisionNum revisionNum2 = null;
                return revisionNum2;
            }
            RevisionNum revisionNum = result.get();
            return revisionNum;
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
    }

    private void storeNewImmutableVersion(Inventory inventory, ObjectPaths.ObjectRoot objectRoot, Path stagingDir) {
        this.ensureNoMutableHead(objectRoot);
        ObjectPaths.VersionRoot versionRoot = objectRoot.headVersion();
        boolean isFirstVersion = this.isFirstVersion(inventory);
        try {
            if (isFirstVersion) {
                this.setupNewObjectDirs(objectRoot.path());
            }
            this.moveToVersionDirectory(inventory, stagingDir, versionRoot);
            try {
                this.verifyPriorInventory(inventory, objectRoot.inventorySidecar());
                this.copyInventoryToRootWithRollback(versionRoot, objectRoot, inventory);
            }
            catch (RuntimeException e) {
                FileUtil.safeDeleteDirectory(versionRoot.path());
                throw e;
            }
        }
        catch (RuntimeException e) {
            if (isFirstVersion && Files.notExists(objectRoot.inventoryFile(), new LinkOption[0])) {
                try {
                    this.purgeObject(inventory.getId());
                }
                catch (RuntimeException e1) {
                    LOG.error("Failed to rollback object {} creation", (Object)inventory.getId(), (Object)e1);
                }
            }
            throw e;
        }
    }

    private void storeNewMutableHeadVersion(Inventory inventory, ObjectPaths.ObjectRoot objectRoot, Path stagingDir) {
        this.ensureRootObjectHasNotChanged(inventory, objectRoot);
        ObjectPaths.VersionRoot versionRoot = objectRoot.headVersion();
        Path revisionPath = versionRoot.contentRoot().headRevisionPath();
        ObjectPaths.VersionRoot stagingVersionRoot = ObjectPaths.version(inventory, stagingDir);
        Path revisionsDirPath = objectRoot.mutableHeadRevisionsPath();
        boolean isNewMutableHead = Files.notExists(versionRoot.inventoryFile(), new LinkOption[0]);
        try {
            Path revisionMarker = this.createRevisionMarker(inventory, revisionsDirPath);
            try {
                this.moveToRevisionDirectory(inventory, stagingVersionRoot, versionRoot);
                if (isNewMutableHead) {
                    this.copyRootInventorySidecar(objectRoot, versionRoot);
                }
                try {
                    this.verifyPriorInventoryMutable(inventory, objectRoot, isNewMutableHead);
                    this.copyInventory(stagingVersionRoot, versionRoot);
                }
                catch (RuntimeException e) {
                    FileUtil.safeDeleteDirectory(revisionPath);
                    throw e;
                }
            }
            catch (RuntimeException e) {
                FileUtil.safeDeleteDirectory(revisionMarker);
                throw e;
            }
        }
        catch (RuntimeException e) {
            if (isNewMutableHead) {
                FileUtil.safeDeleteDirectory(versionRoot.path().getParent());
            }
            throw e;
        }
        this.deleteEmptyDirs(versionRoot.contentPath());
        this.deleteMutableHeadFilesNotInManifest(inventory, objectRoot, versionRoot);
    }

    private void copyRootInventorySidecar(ObjectPaths.ObjectRoot objectRoot, ObjectPaths.VersionRoot versionRoot) {
        Path rootSidecar = objectRoot.inventorySidecar();
        UncheckedFiles.copy(rootSidecar, versionRoot.path().getParent().resolve("root-" + rootSidecar.getFileName().toString()), StandardCopyOption.REPLACE_EXISTING);
    }

    private void moveToVersionDirectory(Inventory inventory, Path source, ObjectPaths.VersionRoot versionRoot) {
        try {
            Files.createDirectories(versionRoot.path().getParent(), new FileAttribute[0]);
            FileUtil.moveDirectory(source, versionRoot.path());
        }
        catch (FileAlreadyExistsException e) {
            throw new ObjectOutOfSyncException(String.format("Failed to create a new version of object %s. Changes are out of sync with the current object state.", inventory.getId()));
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
    }

    private Path createRevisionMarker(Inventory inventory, Path revisionsPath) {
        UncheckedFiles.createDirectories(revisionsPath);
        String revision = inventory.getRevisionNum().toString();
        try {
            Path revisionMarker = Files.createFile(revisionsPath.resolve(revision), new FileAttribute[0]);
            return Files.writeString(revisionMarker, (CharSequence)revision, new OpenOption[0]);
        }
        catch (FileAlreadyExistsException e) {
            throw new ObjectOutOfSyncException(String.format("Failed to update mutable HEAD of object %s. Changes are out of sync with the current object state.", inventory.getId()));
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
    }

    private void moveToRevisionDirectory(Inventory inventory, ObjectPaths.VersionRoot source, ObjectPaths.VersionRoot destination) {
        try {
            Files.createDirectories(destination.contentPath(), new FileAttribute[0]);
            FileUtil.moveDirectory(source.contentRoot().headRevisionPath(), destination.contentRoot().headRevisionPath());
        }
        catch (FileAlreadyExistsException e) {
            throw new ObjectOutOfSyncException(String.format("Failed to update mutable HEAD of object %s. Changes are out of sync with the current object state.", inventory.getId()));
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
    }

    private boolean isFirstVersion(Inventory inventory) {
        return VersionNum.V1.equals((Object)inventory.getHead());
    }

    private void setupNewObjectDirs(Path objectRootPath) {
        UncheckedFiles.createDirectories(objectRootPath);
        new NamasteTypeFile(this.ocflVersion.getOcflObjectVersion()).writeFile(objectRootPath);
    }

    private void copyInventory(ObjectPaths.HasInventory source, ObjectPaths.HasInventory destination) {
        Failsafe.with(this.ioRetry, (Policy[])new RetryPolicy[0]).run(() -> {
            LOG.debug("Copying {} to {}", (Object)source.inventoryFile(), (Object)destination.inventoryFile());
            UncheckedFiles.copy(source.inventoryFile(), destination.inventoryFile(), StandardCopyOption.REPLACE_EXISTING);
            UncheckedFiles.copy(source.inventorySidecar(), destination.inventorySidecar(), StandardCopyOption.REPLACE_EXISTING);
        });
    }

    private void copyInventoryToRootWithRollback(ObjectPaths.HasInventory source, ObjectPaths.ObjectRoot objectRoot, Inventory inventory) {
        try {
            this.copyInventory(source, objectRoot);
        }
        catch (RuntimeException e) {
            if (!this.isFirstVersion(inventory)) {
                try {
                    ObjectPaths.VersionRoot previousVersionRoot = objectRoot.version(inventory.getHead().previousVersionNum());
                    this.copyInventory(previousVersionRoot, objectRoot);
                }
                catch (RuntimeException e1) {
                    LOG.error("Failed to rollback inventory at {}", (Object)objectRoot.inventoryFile(), (Object)e1);
                }
            }
            throw e;
        }
    }

    private void deleteEmptyDirs(Path path) {
        try {
            FileUtil.deleteEmptyDirs(path);
        }
        catch (RuntimeException e) {
            LOG.error("Failed to delete an empty directory. It may need to be deleted manually.", (Throwable)e);
        }
    }

    private void deleteMutableHeadFilesNotInManifest(Inventory inventory, ObjectPaths.ObjectRoot objectRoot, ObjectPaths.VersionRoot versionRoot) {
        List<Path> files = FileUtil.findFiles(versionRoot.contentPath());
        files.forEach(file -> {
            if (inventory.getFileId(objectRoot.path().relativize((Path)file)) == null) {
                try {
                    Files.deleteIfExists(file);
                }
                catch (IOException e) {
                    LOG.warn("Failed to delete file: {}. It should be manually deleted.", file, (Object)e);
                }
            }
        });
    }

    private void verifyPriorInventoryMutable(Inventory inventory, ObjectPaths.ObjectRoot objectRoot, boolean isNewMutableHead) {
        Path sidecarPath = isNewMutableHead ? objectRoot.inventorySidecar() : objectRoot.mutableHeadVersion().inventorySidecar();
        this.verifyPriorInventory(inventory, sidecarPath);
    }

    private void verifyPriorInventory(Inventory inventory, Path sidecarPath) {
        if (inventory.getPreviousDigest() != null) {
            String actualDigest = SidecarMapper.readDigestRequired(sidecarPath);
            if (!actualDigest.equalsIgnoreCase(inventory.getPreviousDigest())) {
                throw new ObjectOutOfSyncException(String.format("Cannot update object %s because the update is out of sync with the current object state. The digest of the current inventory is %s, but the digest %s was expected.", inventory.getId(), actualDigest, inventory.getPreviousDigest()));
            }
        } else if (!inventory.getHead().equals((Object)VersionNum.V1)) {
            LOG.debug("Cannot verify prior inventory for object {} because its digest is unknown.", (Object)inventory.getId());
        }
    }

    private Stream<Path> findOcflObjectRootDirs(Path start) {
        FileSystemOcflObjectRootDirIterator iterator = new FileSystemOcflObjectRootDirIterator(start);
        try {
            Spliterator<String> spliterator = Spliterators.spliteratorUnknownSize(iterator, 1041);
            return (Stream)StreamSupport.stream(spliterator, false).map(x$0 -> Paths.get(x$0, new String[0])).onClose(iterator::close);
        }
        catch (RuntimeException e) {
            iterator.close();
            throw e;
        }
    }

    private void ensureNoMutableHead(ObjectPaths.ObjectRoot objectRoot) {
        if (Files.exists(objectRoot.mutableHeadVersion().inventoryFile(), new LinkOption[0])) {
            throw new OcflStateException(String.format("Cannot create a new version of object %s because it has an active mutable HEAD.", objectRoot.objectId()));
        }
    }

    private void ensureRootObjectHasNotChanged(Inventory inventory, ObjectPaths.ObjectRoot objectRoot) {
        String expectedDigest;
        Path savedSidecarPath = ObjectPaths.inventorySidecarPath(objectRoot.mutableHeadExtensionPath(), inventory);
        try {
            expectedDigest = SidecarMapper.readDigestOptional(savedSidecarPath);
        }
        catch (OcflNoSuchFileException e) {
            return;
        }
        String actualDigest = SidecarMapper.readDigestRequired(objectRoot.inventorySidecar());
        if (!expectedDigest.equalsIgnoreCase(actualDigest)) {
            throw new ObjectOutOfSyncException(String.format("The mutable HEAD of object %s is out of sync with the root object state.", inventory.getId()));
        }
    }

    private void ensureRootObjectHasNotChanged(String objectId, Path objectRootPath) {
        String expectedDigest;
        Path savedSidecarPath = ObjectPaths.findMutableHeadRootInventorySidecarPath(objectRootPath.resolve("extensions/0005-mutable-head"));
        try {
            expectedDigest = SidecarMapper.readDigestOptional(savedSidecarPath);
        }
        catch (OcflNoSuchFileException e) {
            return;
        }
        Path rootSidecarPath = ObjectPaths.findInventorySidecarPath(objectRootPath);
        String actualDigest = SidecarMapper.readDigestRequired(rootSidecarPath);
        if (!expectedDigest.equalsIgnoreCase(actualDigest)) {
            throw new ObjectOutOfSyncException(String.format("The mutable HEAD of object %s is out of sync with the root object state.", objectId));
        }
    }

    private ObjectProperties examineObject(Path objectRootPath) {
        ObjectProperties properties = new ObjectProperties();
        try (Stream<Path> paths2 = Files.list(objectRootPath);){
            List allPaths = paths2.collect(Collectors.toList());
            for (Path path : allPaths) {
                String filename = path.getFileName().toString();
                if (Files.isRegularFile(path, new LinkOption[0])) {
                    if (filename.startsWith("inventory.json.")) {
                        properties.setDigestAlgorithm(SidecarMapper.getDigestAlgorithmFromSidecar(path));
                    } else if (filename.startsWith("0=ocfl_object_")) {
                        properties.setOcflVersion(OcflVersion.fromOcflObjectVersionFilename((String)filename));
                    }
                } else if (Files.isDirectory(path, new LinkOption[0]) && "extensions".equals(filename)) {
                    properties.setExtensions(true);
                }
                if (properties.getOcflVersion() == null || properties.getDigestAlgorithm() == null || !properties.hasExtensions()) continue;
                break;
            }
        }
        catch (NoSuchFileException paths2) {
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
        return properties;
    }

    private Set<String> loadObjectExtensions(Path objectRoot) {
        HashSet<String> extensions = new HashSet<String>();
        Path extensionsDir = ObjectPaths.extensionsPath(objectRoot);
        try (Stream<Path> list2 = Files.list(extensionsDir);){
            list2.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(dir -> {
                String extensionName = dir.getFileName().toString();
                if (this.supportEvaluator.checkSupport(extensionName)) {
                    extensions.add(extensionName);
                }
            });
        }
        catch (NoSuchFileException list2) {
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
        return extensions;
    }
}

