/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.core.storage.filesystem;

import com.fasterxml.jackson.databind.ObjectMapper;
import edu.wisc.library.ocfl.api.exception.CorruptObjectException;
import edu.wisc.library.ocfl.api.exception.InvalidInventoryException;
import edu.wisc.library.ocfl.api.exception.OcflIOException;
import edu.wisc.library.ocfl.api.exception.OcflNoSuchFileException;
import edu.wisc.library.ocfl.api.exception.RepositoryConfigurationException;
import edu.wisc.library.ocfl.api.model.OcflVersion;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.ObjectPaths;
import edu.wisc.library.ocfl.core.extension.ExtensionSupportEvaluator;
import edu.wisc.library.ocfl.core.extension.OcflExtensionConfig;
import edu.wisc.library.ocfl.core.extension.OcflExtensionRegistry;
import edu.wisc.library.ocfl.core.extension.storage.layout.OcflLayout;
import edu.wisc.library.ocfl.core.extension.storage.layout.OcflStorageLayoutExtension;
import edu.wisc.library.ocfl.core.util.FileUtil;
import edu.wisc.library.ocfl.core.util.NamasteTypeFile;
import edu.wisc.library.ocfl.core.util.UncheckedFiles;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemOcflStorageInitializer {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemOcflStorageInitializer.class);
    private static final String SPECS_DIR = "ocfl-specs/";
    private static final String EXT_SPEC = "ocfl_extensions_1.0.md";
    private final ObjectMapper objectMapper;

    public FileSystemOcflStorageInitializer(ObjectMapper objectMapper) {
        this.objectMapper = (ObjectMapper)Enforce.notNull((Object)objectMapper, (String)"objectMapper cannot be null");
    }

    public OcflStorageLayoutExtension initializeStorage(Path repositoryRoot, OcflVersion ocflVersion, OcflExtensionConfig layoutConfig, ExtensionSupportEvaluator supportEvaluator) {
        OcflStorageLayoutExtension layoutExtension;
        Enforce.notNull((Object)repositoryRoot, (String)"repositoryRoot cannot be null");
        Enforce.notNull((Object)ocflVersion, (String)"ocflVersion cannot be null");
        if (!Files.exists(repositoryRoot, new LinkOption[0])) {
            UncheckedFiles.createDirectories(repositoryRoot);
        } else {
            Enforce.expressionTrue((boolean)Files.isDirectory(repositoryRoot, new LinkOption[0]), (Object)repositoryRoot, (String)"repositoryRoot must be a directory");
        }
        if (!FileUtil.hasChildren(repositoryRoot)) {
            layoutExtension = this.initNewRepo(repositoryRoot, ocflVersion, layoutConfig);
        } else {
            layoutExtension = this.validateAndLoadExistingRepo(repositoryRoot, ocflVersion, layoutConfig);
            this.loadRepositoryExtensions(repositoryRoot, supportEvaluator);
        }
        LOG.info("OCFL repository is configured to use OCFL storage layout extension {} implemented by {}", (Object)layoutExtension.getExtensionName(), layoutExtension.getClass());
        return layoutExtension;
    }

    private OcflStorageLayoutExtension validateAndLoadExistingRepo(Path repositoryRoot, OcflVersion ocflVersion, OcflExtensionConfig layoutConfig) {
        this.validateOcflVersion(repositoryRoot, ocflVersion);
        OcflLayout ocflLayout = this.readOcflLayout(repositoryRoot);
        if (ocflLayout == null) {
            LOG.debug("OCFL layout extension not specified");
            return this.validateLayoutByInspection(repositoryRoot, layoutConfig);
        }
        LOG.debug("Found specified OCFL layout extension: {}", (Object)ocflLayout.getExtension());
        return this.loadLayoutFromConfig(repositoryRoot, ocflLayout);
    }

    private void validateOcflVersion(Path repositoryRoot, OcflVersion ocflVersion) {
        OcflVersion existingOcflVersion = null;
        for (File file : repositoryRoot.toFile().listFiles()) {
            if (!file.isFile() || !file.getName().startsWith("0=")) continue;
            existingOcflVersion = OcflVersion.fromOcflVersionFilename((String)file.getName());
            break;
        }
        if (existingOcflVersion == null) {
            throw new RepositoryConfigurationException("OCFL root is missing its namaste file, eg. 0=ocfl_1.0.");
        }
        if (existingOcflVersion != ocflVersion) {
            throw new RepositoryConfigurationException(String.format("OCFL version mismatch. Expected: %s; Found: %s", ocflVersion, existingOcflVersion));
        }
    }

    private OcflStorageLayoutExtension loadLayoutFromConfig(Path repositoryRoot, OcflLayout ocflLayout) {
        OcflStorageLayoutExtension layoutExtension = this.loadLayoutExtension(ocflLayout.getExtension());
        OcflExtensionConfig expectedConfig = this.readLayoutConfig(repositoryRoot, ocflLayout, layoutExtension.getExtensionConfigClass());
        layoutExtension.init(expectedConfig);
        return layoutExtension;
    }

    private OcflStorageLayoutExtension validateLayoutByInspection(Path repositoryRoot, OcflExtensionConfig layoutConfig) {
        Path actualPath;
        String objectId;
        Path expectedPath;
        if (layoutConfig == null) {
            throw new RepositoryConfigurationException(String.format("No storage layout configuration is defined in the OCFL repository at %s. Layout must be configured programmatically.", repositoryRoot));
        }
        OcflStorageLayoutExtension layoutExtension = this.loadAndInitLayoutExtension(layoutConfig);
        Path objectRoot = this.identifyRandomObjectRoot(repositoryRoot);
        if (objectRoot != null && !(expectedPath = Paths.get(layoutExtension.mapObjectId(objectId = this.extractObjectId(ObjectPaths.inventoryPath(objectRoot))), new String[0])).equals(actualPath = repositoryRoot.relativize(objectRoot))) {
            throw new RepositoryConfigurationException(String.format("The OCFL client was configured to use the following layout: %s. This layout does not match the layout of existing objects in the repository. Found object %s stored at %s, but was expecting it to be stored at %s.", layoutConfig, objectId, actualPath, expectedPath));
        }
        return layoutExtension;
    }

    private Path identifyRandomObjectRoot(Path root) {
        final AtomicReference ref = new AtomicReference();
        try {
            Files.walkFileTree(root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (file.getFileName().toString().startsWith("0=ocfl_object_")) {
                        ref.set(file.getParent());
                        return FileVisitResult.TERMINATE;
                    }
                    return super.visitFile(file, attrs);
                }
            });
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
        return (Path)ref.get();
    }

    private String extractObjectId(Path inventoryPath) {
        if (Files.notExists(inventoryPath, new LinkOption[0])) {
            throw new CorruptObjectException("Missing inventory at " + inventoryPath);
        }
        Map map = this.read(inventoryPath, Map.class);
        Object id = map.get("id");
        if (id == null) {
            throw new InvalidInventoryException(String.format("Inventory file at %s does not contain an id.", inventoryPath));
        }
        return (String)id;
    }

    private OcflStorageLayoutExtension initNewRepo(Path repositoryRoot, OcflVersion ocflVersion, OcflExtensionConfig layoutConfig) {
        Enforce.notNull((Object)layoutConfig, (String)"layoutConfig cannot be null when initializing a new repo");
        LOG.info("Initializing new OCFL repository at {}", (Object)repositoryRoot);
        OcflStorageLayoutExtension layoutExtension = this.loadAndInitLayoutExtension(layoutConfig);
        try {
            new NamasteTypeFile(ocflVersion.getOcflVersion()).writeFile(repositoryRoot);
            this.writeOcflSpec(repositoryRoot, ocflVersion);
            this.writeOcflLayout(repositoryRoot, layoutConfig, layoutExtension.getDescription());
            this.writeOcflLayoutSpec(repositoryRoot, layoutConfig);
            this.writeSpecFile(this.getClass().getClassLoader(), repositoryRoot, EXT_SPEC);
            return layoutExtension;
        }
        catch (RuntimeException e) {
            LOG.error("Failed to initialize OCFL repository at {}", (Object)repositoryRoot, (Object)e);
            FileUtil.safeDeleteDirectory(repositoryRoot);
            throw e;
        }
    }

    private void writeOcflSpec(Path repositoryRoot, OcflVersion ocflVersion) {
        String ocflSpecFile = ocflVersion.getOcflVersion() + ".txt";
        this.writeSpecFile(this.getClass().getClassLoader(), repositoryRoot, ocflSpecFile);
    }

    private void writeOcflLayoutSpec(Path repositoryRoot, OcflExtensionConfig layoutConfig) {
        String specFile = layoutConfig.getExtensionName() + ".md";
        try {
            this.writeSpecFile(layoutConfig.getClass().getClassLoader(), repositoryRoot, specFile);
        }
        catch (RuntimeException e) {
            LOG.warn("Failed to write spec file for layout extension {}", (Object)layoutConfig.getExtensionName(), (Object)e);
        }
    }

    private void writeSpecFile(ClassLoader classLoader, Path repositoryRoot, String fileName) {
        block9: {
            try (InputStream stream = classLoader.getResourceAsStream(SPECS_DIR + fileName);){
                if (stream != null) {
                    Files.copy(stream, repositoryRoot.resolve(fileName), new CopyOption[0]);
                    break block9;
                }
                throw new RuntimeException("No spec file found for " + fileName);
            }
            catch (IOException e) {
                throw new OcflIOException(e);
            }
        }
    }

    private void writeOcflLayout(Path repositoryRoot, OcflExtensionConfig layoutConfig, String description) {
        OcflLayout spec = new OcflLayout().setExtension(layoutConfig.getExtensionName()).setDescription(description);
        try {
            this.objectMapper.writeValue(this.ocflLayoutPath(repositoryRoot).toFile(), (Object)spec);
            if (layoutConfig.hasParameters()) {
                Path configFile = this.layoutExtensionConfigPath(repositoryRoot, layoutConfig.getExtensionName());
                UncheckedFiles.createDirectories(configFile.getParent());
                this.objectMapper.writeValue(configFile.toFile(), (Object)layoutConfig);
            }
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
    }

    private OcflStorageLayoutExtension loadAndInitLayoutExtension(OcflExtensionConfig layoutConfig) {
        OcflStorageLayoutExtension layoutExtension = this.loadLayoutExtension(layoutConfig.getExtensionName());
        layoutExtension.init(layoutConfig);
        return layoutExtension;
    }

    private OcflStorageLayoutExtension loadLayoutExtension(String extensionName) {
        return (OcflStorageLayoutExtension)OcflExtensionRegistry.lookup(extensionName).orElseThrow(() -> new IllegalStateException(String.format("Failed to find an implementation for storage layout extension %s", extensionName)));
    }

    private OcflLayout readOcflLayout(Path repositoryRoot) {
        Path layoutPath = this.ocflLayoutPath(repositoryRoot);
        try {
            return this.read(layoutPath, OcflLayout.class);
        }
        catch (OcflNoSuchFileException e) {
            return null;
        }
    }

    private OcflExtensionConfig readLayoutConfig(Path repositoryRoot, OcflLayout ocflLayout, Class<? extends OcflExtensionConfig> clazz) {
        Path configPath = this.layoutExtensionConfigPath(repositoryRoot, ocflLayout.getExtension());
        try {
            return this.read(configPath, clazz);
        }
        catch (OcflNoSuchFileException e) {
            return this.initClass(clazz);
        }
    }

    private void loadRepositoryExtensions(Path repositoryRoot, ExtensionSupportEvaluator supportEvaluator) {
        Path extensionsDir = repositoryRoot.resolve("extensions");
        try (Stream<Path> list2 = Files.list(extensionsDir);){
            list2.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(dir -> {
                String extensionName = dir.getFileName().toString();
                supportEvaluator.checkSupport(extensionName);
            });
        }
        catch (NoSuchFileException list2) {
        }
        catch (IOException e) {
            throw new OcflIOException(e);
        }
    }

    private <T> T initClass(Class<T> clazz) {
        try {
            return clazz.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (Exception e) {
            throw new RepositoryConfigurationException(String.format("Failed to init OCFL storage layout extension configuration class %s", clazz), (Throwable)e);
        }
    }

    private <T> T read(Path path, Class<T> clazz) {
        try {
            return (T)this.objectMapper.readValue(path.toFile(), clazz);
        }
        catch (IOException e) {
            throw OcflIOException.from((IOException)e);
        }
    }

    private Path ocflLayoutPath(Path repositoryRoot) {
        return repositoryRoot.resolve("ocfl_layout.json");
    }

    private Path layoutExtensionConfigPath(Path repositoryRoot, String extensionName) {
        return repositoryRoot.resolve("extensions").resolve(extensionName).resolve("config.json");
    }
}

