/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.core.storage;

import com.fasterxml.jackson.databind.ObjectMapper;
import edu.wisc.library.ocfl.api.OcflConstants;
import edu.wisc.library.ocfl.api.exception.CorruptObjectException;
import edu.wisc.library.ocfl.api.exception.InvalidInventoryException;
import edu.wisc.library.ocfl.api.exception.OcflIOException;
import edu.wisc.library.ocfl.api.exception.OcflJavaException;
import edu.wisc.library.ocfl.api.exception.OcflNoSuchFileException;
import edu.wisc.library.ocfl.api.exception.RepositoryConfigurationException;
import edu.wisc.library.ocfl.api.model.OcflVersion;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.ObjectPaths;
import edu.wisc.library.ocfl.core.extension.ExtensionSupportEvaluator;
import edu.wisc.library.ocfl.core.extension.OcflExtensionConfig;
import edu.wisc.library.ocfl.core.extension.OcflExtensionRegistry;
import edu.wisc.library.ocfl.core.extension.storage.layout.OcflLayout;
import edu.wisc.library.ocfl.core.extension.storage.layout.OcflStorageLayoutExtension;
import edu.wisc.library.ocfl.core.storage.OcflStorageInitializer;
import edu.wisc.library.ocfl.core.storage.RepositoryConfig;
import edu.wisc.library.ocfl.core.storage.common.Listing;
import edu.wisc.library.ocfl.core.storage.common.OcflObjectRootDirIterator;
import edu.wisc.library.ocfl.core.storage.common.Storage;
import edu.wisc.library.ocfl.core.util.FileUtil;
import edu.wisc.library.ocfl.core.util.NamasteTypeFile;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultOcflStorageInitializer
implements OcflStorageInitializer {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultOcflStorageInitializer.class);
    private static final String SPECS_DIR = "ocfl-specs/";
    private static final String EXT_SPEC = "ocfl_extensions_1.0.md";
    private static final String MEDIA_TYPE_TEXT = "text/plain; charset=UTF-8";
    private static final String MEDIA_TYPE_JSON = "application/json; charset=UTF-8";
    private final Storage storage;
    private final ObjectMapper objectMapper;

    public DefaultOcflStorageInitializer(Storage storage, ObjectMapper objectMapper) {
        this.storage = (Storage)Enforce.notNull((Object)storage, (String)"storage cannot be null");
        this.objectMapper = (ObjectMapper)Enforce.notNull((Object)objectMapper, (String)"objectMapper cannot be null");
    }

    @Override
    public RepositoryConfig initializeStorage(OcflVersion ocflVersion, OcflExtensionConfig layoutConfig, ExtensionSupportEvaluator supportEvaluator) {
        RepositoryConfig config;
        if (this.directoryIsEmpty("")) {
            config = this.initNewRepo(ocflVersion, layoutConfig);
        } else {
            config = this.loadAndValidateExistingRepo(ocflVersion, layoutConfig);
            this.loadRepositoryExtensions(supportEvaluator);
        }
        LOG.info("OCFL repository is configured to adhere to OCFL {} and use OCFL storage layout extension {}", (Object)config.getOcflVersion().getRawVersion(), (Object)config.getStorageLayoutExtension().getExtensionName());
        return config;
    }

    private RepositoryConfig loadAndValidateExistingRepo(OcflVersion ocflVersion, OcflExtensionConfig layoutConfig) {
        OcflStorageLayoutExtension extension;
        OcflLayout ocflLayout;
        OcflVersion existingVersion;
        OcflVersion resolvedVersion = existingVersion = this.identifyExistingVersion();
        if (ocflVersion != null && existingVersion.compareTo((Enum)ocflVersion) < 0) {
            this.upgradeOcflRepo(existingVersion, ocflVersion);
            resolvedVersion = ocflVersion;
        }
        if ((ocflLayout = this.readOcflLayout()) == null) {
            LOG.debug("OCFL layout extension not specified");
            extension = this.validateLayoutByInspection(layoutConfig);
        } else {
            LOG.debug("Found specified OCFL layout extension: {}", (Object)ocflLayout.getExtension());
            extension = this.loadLayoutByConfig(ocflLayout);
        }
        return new RepositoryConfig(resolvedVersion, extension);
    }

    private OcflVersion identifyExistingVersion() {
        OcflVersion foundVersion = null;
        for (OcflVersion version : OcflVersion.values()) {
            String fileName = new NamasteTypeFile(version.getOcflVersion()).fileName();
            try {
                String contents = this.storage.readToString(fileName);
                foundVersion = OcflVersion.fromOcflVersionString((String)contents);
                break;
            }
            catch (OcflNoSuchFileException e) {
                LOG.debug("OCFL root namaste file {} does not exist", (Object)fileName);
            }
        }
        if (foundVersion == null) {
            throw new RepositoryConfigurationException("OCFL root is missing a namaste file, eg. 0=ocfl_1.0.");
        }
        return foundVersion;
    }

    private void upgradeOcflRepo(OcflVersion currentVersion, OcflVersion newVersion) {
        LOG.info("This is an OCFL {} repository, but was programmatically configured to create OCFL {} objects. Upgrading the OCFL repository to {}. Note, existing objects will NOT be upgraded.", new Object[]{currentVersion.getRawVersion(), newVersion.getRawVersion(), newVersion.getRawVersion()});
        try {
            this.writeNamasteFile(newVersion);
            this.writeOcflSpec(newVersion);
            this.storage.deleteFile(new NamasteTypeFile(currentVersion.getOcflVersion()).fileName());
        }
        catch (RuntimeException e) {
            throw new OcflJavaException(String.format("Failed to upgrade OCFL repository to version %s", newVersion.getRawVersion()), (Throwable)e);
        }
    }

    private OcflStorageLayoutExtension loadLayoutByConfig(OcflLayout ocflLayout) {
        OcflStorageLayoutExtension layoutExtension = this.loadLayoutExtension(ocflLayout.getExtension());
        OcflExtensionConfig expectedConfig = this.readLayoutConfig(ocflLayout, layoutExtension.getExtensionConfigClass());
        layoutExtension.init(expectedConfig);
        return layoutExtension;
    }

    private OcflStorageLayoutExtension validateLayoutByInspection(OcflExtensionConfig layoutConfig) {
        String objectId;
        String expectedPath;
        if (layoutConfig == null) {
            throw new RepositoryConfigurationException("No storage layout configuration is defined in the OCFL repository. Layout must be configured programmatically.");
        }
        OcflStorageLayoutExtension layoutExtension = this.loadAndInitLayoutExtension(layoutConfig);
        String objectRoot = this.identifyRandomObjectRoot();
        if (objectRoot != null && !(expectedPath = layoutExtension.mapObjectId(objectId = this.extractObjectId(ObjectPaths.inventoryPath(objectRoot)))).equals(objectRoot)) {
            throw new RepositoryConfigurationException(String.format("The OCFL client was configured to use the following layout: %s. This layout does not match the layout of existing objects in the repository. Found object %s stored at %s, but was expecting it to be stored at %s.", layoutConfig, objectId, objectRoot, expectedPath));
        }
        return layoutExtension;
    }

    private String identifyRandomObjectRoot() {
        try (OcflObjectRootDirIterator iter = this.storage.iterateObjects();){
            if (iter.hasNext()) {
                String string = iter.next();
                return string;
            }
            String string = null;
            return string;
        }
    }

    private String extractObjectId(String inventoryPath) {
        String string;
        block10: {
            InputStream stream = this.storage.read(inventoryPath);
            try {
                Map map = this.read(stream, Map.class);
                Object id = map.get("id");
                if (id == null) {
                    throw new InvalidInventoryException(String.format("Inventory file at %s does not contain an id.", inventoryPath));
                }
                string = (String)id;
                if (stream == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw OcflIOException.from((IOException)e);
                }
                catch (OcflNoSuchFileException e) {
                    throw new CorruptObjectException(String.format("Missing inventory at %s", inventoryPath));
                }
            }
            stream.close();
        }
        return string;
    }

    private RepositoryConfig initNewRepo(OcflVersion ocflVersion, OcflExtensionConfig layoutConfig) {
        Enforce.notNull((Object)layoutConfig, (String)"layoutConfig cannot be null when initializing a new repo");
        if (ocflVersion == null) {
            ocflVersion = OcflConstants.DEFAULT_OCFL_VERSION;
        }
        LOG.info("Initializing new OCFL repository");
        OcflStorageLayoutExtension layoutExtension = this.loadAndInitLayoutExtension(layoutConfig);
        try {
            this.storage.createDirectories("");
            this.writeNamasteFile(ocflVersion);
            this.writeOcflSpec(ocflVersion);
            this.writeOcflLayout(layoutConfig, layoutExtension.getDescription());
            this.writeOcflLayoutSpec(layoutConfig);
            this.writeSpecFile(this.getClass().getClassLoader(), EXT_SPEC);
            return new RepositoryConfig(ocflVersion, layoutExtension);
        }
        catch (RuntimeException e) {
            LOG.error("Failed to initialize OCFL repository", (Throwable)e);
            try {
                this.storage.deleteDirectory("");
            }
            catch (RuntimeException e1) {
                LOG.error("Failed to cleanup OCFL repository root", (Throwable)e1);
            }
            throw e;
        }
    }

    private void loadRepositoryExtensions(ExtensionSupportEvaluator supportEvaluator) {
        this.list("extensions").stream().filter(Listing::isDirectory).forEach(dir -> supportEvaluator.checkSupport(dir.getRelativePath()));
    }

    private void writeOcflSpec(OcflVersion ocflVersion) {
        String ext = ".md";
        if (ocflVersion == OcflVersion.OCFL_1_0) {
            ext = ".txt";
        }
        this.writeSpecFile(this.getClass().getClassLoader(), ocflVersion.getOcflVersion() + ext);
    }

    private void writeOcflLayoutSpec(OcflExtensionConfig layoutConfig) {
        try {
            this.writeSpecFile(layoutConfig.getClass().getClassLoader(), layoutConfig.getExtensionName() + ".md");
        }
        catch (RuntimeException e) {
            LOG.warn("Failed to write spec file for layout extension {}", (Object)layoutConfig.getExtensionName(), (Object)e);
        }
    }

    private void writeSpecFile(ClassLoader classLoader, String fileName) {
        block9: {
            try (InputStream stream = classLoader.getResourceAsStream(SPECS_DIR + fileName);){
                if (stream != null) {
                    this.writeStream(fileName, stream);
                    break block9;
                }
                throw new OcflJavaException("No spec file found for " + fileName);
            }
            catch (IOException e) {
                throw new OcflIOException((Exception)e);
            }
        }
    }

    private void writeNamasteFile(OcflVersion ocflVersion) {
        NamasteTypeFile namasteFile = new NamasteTypeFile(ocflVersion.getOcflVersion());
        this.storage.write(namasteFile.fileName(), namasteFile.fileContent().getBytes(StandardCharsets.UTF_8), MEDIA_TYPE_TEXT);
    }

    private void writeOcflLayout(OcflExtensionConfig layoutConfig, String description) {
        OcflLayout spec = new OcflLayout().setExtension(layoutConfig.getExtensionName()).setDescription(description);
        try {
            this.storage.write("ocfl_layout.json", this.objectMapper.writeValueAsBytes((Object)spec), MEDIA_TYPE_JSON);
            if (layoutConfig.hasParameters()) {
                this.storage.createDirectories(FileUtil.pathJoinFailEmpty("extensions", layoutConfig.getExtensionName()));
                this.storage.write(this.layoutConfigFile(layoutConfig.getExtensionName()), this.objectMapper.writeValueAsBytes((Object)layoutConfig), MEDIA_TYPE_JSON);
            }
        }
        catch (IOException e) {
            throw new OcflIOException((Exception)e);
        }
    }

    private OcflStorageLayoutExtension loadAndInitLayoutExtension(OcflExtensionConfig layoutConfig) {
        OcflStorageLayoutExtension layoutExtension = this.loadLayoutExtension(layoutConfig.getExtensionName());
        layoutExtension.init(layoutConfig);
        return layoutExtension;
    }

    private OcflStorageLayoutExtension loadLayoutExtension(String extensionName) {
        return (OcflStorageLayoutExtension)OcflExtensionRegistry.lookup(extensionName).orElseThrow(() -> new IllegalStateException(String.format("Failed to find an implementation for storage layout extension %s", extensionName)));
    }

    private OcflLayout readOcflLayout() {
        OcflLayout ocflLayout;
        block9: {
            InputStream stream = this.storage.read("ocfl_layout.json");
            try {
                ocflLayout = this.read(stream, OcflLayout.class);
                if (stream == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (OcflNoSuchFileException e) {
                    return null;
                }
                catch (IOException e) {
                    throw new OcflIOException((Exception)e);
                }
            }
            stream.close();
        }
        return ocflLayout;
    }

    private OcflExtensionConfig readLayoutConfig(OcflLayout ocflLayout, Class<? extends OcflExtensionConfig> clazz) {
        OcflExtensionConfig ocflExtensionConfig;
        block9: {
            InputStream stream = this.storage.read(this.layoutConfigFile(ocflLayout.getExtension()));
            try {
                ocflExtensionConfig = this.read(stream, clazz);
                if (stream == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (OcflNoSuchFileException e) {
                    return this.initClass(clazz);
                }
                catch (IOException e) {
                    throw new OcflIOException((Exception)e);
                }
            }
            stream.close();
        }
        return ocflExtensionConfig;
    }

    private <T> T initClass(Class<T> clazz) {
        try {
            return clazz.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        }
        catch (Exception e) {
            throw new RepositoryConfigurationException(String.format("Failed to init OCFL storage layout extension configuration class %s", clazz), (Throwable)e);
        }
    }

    private String layoutConfigFile(String extensionName) {
        return FileUtil.pathJoinFailEmpty("extensions", extensionName, "config.json");
    }

    private <T> T read(InputStream stream, Class<T> clazz) {
        try {
            return (T)this.objectMapper.readValue(stream, clazz);
        }
        catch (IOException e) {
            throw new OcflIOException((Exception)e);
        }
    }

    private void writeStream(String remotePath, InputStream stream) {
        try {
            this.storage.write(remotePath, stream.readAllBytes(), MEDIA_TYPE_TEXT);
        }
        catch (IOException e) {
            throw new OcflIOException((Exception)e);
        }
    }

    private List<Listing> list(String path) {
        try {
            return this.storage.listDirectory(path);
        }
        catch (OcflNoSuchFileException e) {
            return Collections.emptyList();
        }
    }

    private boolean directoryIsEmpty(String path) {
        try {
            return this.storage.directoryIsEmpty(path);
        }
        catch (OcflNoSuchFileException e) {
            return true;
        }
    }
}

