/*
 * Decompiled with CFR 0.152.
 */
package edu.wisc.library.ocfl.core.validation;

import edu.wisc.library.ocfl.api.model.DigestAlgorithm;
import edu.wisc.library.ocfl.api.model.InventoryType;
import edu.wisc.library.ocfl.api.model.OcflVersion;
import edu.wisc.library.ocfl.api.model.ValidationCode;
import edu.wisc.library.ocfl.api.model.ValidationIssue;
import edu.wisc.library.ocfl.api.model.ValidationResults;
import edu.wisc.library.ocfl.api.model.VersionNum;
import edu.wisc.library.ocfl.api.util.Enforce;
import edu.wisc.library.ocfl.core.validation.ValidationResultsBuilder;
import edu.wisc.library.ocfl.core.validation.model.SimpleInventory;
import edu.wisc.library.ocfl.core.validation.model.SimpleUser;
import edu.wisc.library.ocfl.core.validation.model.SimpleVersion;
import java.net.URI;
import java.net.URISyntaxException;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.util.BitSet;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeSet;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.regex.Pattern;

public class SimpleInventoryValidator {
    private static final Pattern VALID_VERSION = Pattern.compile("^v\\d+$");
    private static final VersionNum VERSION_ZERO = VersionNum.fromInt((int)0);
    private static final List<String> ALLOWED_CONTENT_DIGESTS = List.of(DigestAlgorithm.sha512.getOcflName(), DigestAlgorithm.sha256.getOcflName());
    private static final Map<String, Integer> DIGEST_LENGTHS = Map.of(DigestAlgorithm.md5.getOcflName(), 32, DigestAlgorithm.sha1.getOcflName(), 40, DigestAlgorithm.sha256.getOcflName(), 64, DigestAlgorithm.sha512.getOcflName(), 128, DigestAlgorithm.blake2b512.getOcflName(), 128, DigestAlgorithm.blake2b160.getOcflName(), 40, DigestAlgorithm.blake2b256.getOcflName(), 64, DigestAlgorithm.blake2b384.getOcflName(), 96, DigestAlgorithm.sha512_256.getOcflName(), 64);
    private static final DateTimeFormatter RFC3339_FORMAT = new DateTimeFormatterBuilder().parseCaseInsensitive().appendValue(ChronoField.YEAR, 4).appendLiteral('-').appendValue(ChronoField.MONTH_OF_YEAR, 2).appendLiteral('-').appendValue(ChronoField.DAY_OF_MONTH, 2).appendLiteral('T').appendValue(ChronoField.HOUR_OF_DAY, 2).appendLiteral(':').appendValue(ChronoField.MINUTE_OF_HOUR, 2).appendLiteral(':').appendValue(ChronoField.SECOND_OF_MINUTE, 2).optionalStart().appendFraction(ChronoField.NANO_OF_SECOND, 0, 9, true).optionalEnd().appendOffset("+HH:MM", "Z").toFormatter();
    private final BitSet lowerHexChars = new BitSet();

    public SimpleInventoryValidator() {
        int i;
        for (i = 48; i <= 57; ++i) {
            this.lowerHexChars.set(i);
        }
        for (i = 97; i <= 102; ++i) {
            this.lowerHexChars.set(i);
        }
    }

    public ValidationResults validateInventory(SimpleInventory inventory, String inventoryPath, OcflVersion ocflVersion, VersionEquality equality) {
        Enforce.notNull((Object)inventory, (String)"inventory cannot be null");
        Enforce.notNull((Object)inventoryPath, (String)"inventoryPath cannot be null");
        ValidationResultsBuilder results = new ValidationResultsBuilder();
        results.addIssue(this.notBlank(inventory.getId(), ValidationCode.E036, "Inventory id must be set in %s", inventoryPath)).addIssue(this.ifNotNull(inventory.getId(), () -> this.isTrue(this.isUri(inventory.getId()), ValidationCode.W005, "Inventory id should be a URI in %s. Found: %s", inventoryPath, inventory.getId()))).addIssue(this.notNull(inventory.getType(), ValidationCode.E036, "Inventory type must be set in %s", inventoryPath)).addIssue(this.notNull(inventory.getDigestAlgorithm(), ValidationCode.E036, "Inventory digest algorithm must be set in %s", inventoryPath)).addIssue(this.notNull(inventory.getHead(), ValidationCode.E036, "Inventory head must be set in %s", inventoryPath));
        this.validateType(inventory, inventoryPath, ocflVersion, equality, results);
        if (inventory.getDigestAlgorithm() != null) {
            if (!ALLOWED_CONTENT_DIGESTS.contains(inventory.getDigestAlgorithm())) {
                results.addIssue(ValidationCode.E025, "Inventory digest algorithm must be one of %s in %s. Found: %s", ALLOWED_CONTENT_DIGESTS, inventoryPath, inventory.getDigestAlgorithm());
            } else {
                results.addIssue(this.isTrue(DigestAlgorithm.sha512.getOcflName().equals(inventory.getDigestAlgorithm()), ValidationCode.W004, "Inventory digest algorithm should be %s in %s. Found: %s", DigestAlgorithm.sha512.getOcflName(), inventoryPath, inventory.getDigestAlgorithm()));
            }
        }
        if (inventory.getHead() != null) {
            this.parseAndValidateVersionNum(inventory.getHead(), inventoryPath, results);
        }
        if (inventory.getContentDirectory() != null) {
            String content = inventory.getContentDirectory();
            results.addIssue(this.isFalse(content.contains("/"), ValidationCode.E017, "Inventory content directory cannot contain '/' in %s", inventoryPath)).addIssue(this.isFalse(content.equals(".") || content.equals(".."), ValidationCode.E018, "Inventory content directory cannot equal '.' or '..' in %s", inventoryPath));
        }
        this.validateInventoryVersionNumbers(inventory, inventoryPath, results);
        this.validateInventoryManifest(inventory, inventoryPath, results);
        this.validateInventoryVersions(inventory, inventoryPath, results);
        this.validateInventoryFixity(inventory, inventoryPath, results);
        return results.build();
    }

    private void validateType(SimpleInventory inventory, String inventoryPath, OcflVersion ocflVersion, VersionEquality equality, ValidationResultsBuilder results) {
        if (inventory.getType() != null) {
            try {
                InventoryType type = InventoryType.fromValue((String)inventory.getType());
                if (ocflVersion != null) {
                    Enforce.notNull((Object)((Object)equality), (String)"equality cannot be null");
                    if (equality == VersionEquality.EQUAL) {
                        if (type != ocflVersion.getInventoryType()) {
                            results.addIssue(ValidationCode.E038, "Inventory type must equal '%s' in %s", ocflVersion.getInventoryType().getId(), inventoryPath);
                        }
                    } else if (equality == VersionEquality.LESS_THAN_OR_EQUAL && type.compareTo((Enum)ocflVersion.getInventoryType()) > 0) {
                        results.addIssue(ValidationCode.E103, "Inventory type must be for version %s or lower in %s. Found: %s", ocflVersion.getRawVersion(), inventoryPath, type.getId());
                    }
                }
            }
            catch (RuntimeException e) {
                results.addIssue(ValidationCode.E038, "Invalid inventory type in %s. Found: %s", inventoryPath, inventory.getType());
            }
        }
    }

    private void validateInventoryManifest(SimpleInventory inventory, String inventoryPath, ValidationResultsBuilder results) {
        if (inventory.getManifest() != null) {
            HashSet<String> digests = new HashSet<String>(inventory.getManifest().size());
            for (String digest : inventory.getManifest().keySet()) {
                String digestLower = digest.toLowerCase();
                if (!this.isDigestValidHex(digestLower, inventory.getDigestAlgorithm())) {
                    results.addIssue(ValidationCode.E096, "Inventory manifest digests must be valid in %s. Found: %s", inventoryPath, digest);
                }
                if (digests.contains(digestLower)) {
                    results.addIssue(ValidationCode.E096, "Inventory manifest cannot contain duplicates of digest %s in %s", digestLower, inventoryPath);
                    continue;
                }
                digests.add(digestLower);
            }
            this.validateDigestPathsMap(inventory.getManifest(), path -> results.addIssue(ValidationCode.E100, "Inventory manifest cannot contain content paths that begin or end with '/' in %s. Found: %s", inventoryPath, path), path -> results.addIssue(ValidationCode.E101, "Inventory manifest content paths must be unique in %s. Found: %s", inventoryPath, path), path -> results.addIssue(ValidationCode.E099, "Inventory manifest cannot contain blank content path parts in %s. Found: %s", inventoryPath, path), path -> results.addIssue(ValidationCode.E099, "Inventory manifest cannot contain content path parts equal to '.' or '..' in %s. Found: %s", inventoryPath, path), path -> results.addIssue(ValidationCode.E101, "Inventory manifest content paths must be non-conflicting in %s. Found conflicting path: %s", inventoryPath, path));
        } else {
            results.addIssue(ValidationCode.E041, "Inventory manifest must be set in %s", inventoryPath);
        }
    }

    private void validateInventoryVersions(SimpleInventory inventory, String inventoryPath, ValidationResultsBuilder results) {
        if (inventory.getVersions() != null) {
            Map<Object, Object> manifest = inventory.getManifest() == null ? Collections.emptyMap() : inventory.getManifest();
            HashSet<Object> unseenDigests = new HashSet<Object>(manifest.keySet());
            for (Map.Entry<String, SimpleVersion> entry : inventory.getVersions().entrySet()) {
                String versionNum = entry.getKey();
                SimpleVersion version = entry.getValue();
                if (version.getCreated() != null) {
                    try {
                        RFC3339_FORMAT.parse(version.getCreated());
                    }
                    catch (DateTimeParseException e) {
                        results.addIssue(ValidationCode.E049, "Inventory version %s created timestamp must be formatted in accordance to RFC3339 in %s. Found: %s", versionNum, inventoryPath, version.getCreated());
                    }
                } else {
                    results.addIssue(ValidationCode.E048, "Inventory version %s must contain a created timestamp in %s", versionNum, inventoryPath);
                }
                if (version.getUser() != null) {
                    SimpleUser user = version.getUser();
                    results.addIssue(this.notBlank(user.getName(), ValidationCode.E054, "Inventory version %s user name must be set in %s", versionNum, inventoryPath)).addIssue(this.notNull(user.getAddress(), ValidationCode.W008, "Inventory version %s user address should be set in %s", versionNum, inventoryPath));
                    if (user.getAddress() != null) {
                        results.addIssue(this.isTrue(this.isUri(user.getAddress()), ValidationCode.W009, "Inventory version %s user address should be a URI in %s. Found: %s", versionNum, inventoryPath, user.getAddress()));
                    }
                } else {
                    results.addIssue(ValidationCode.W007, "Inventory version %s should contain a user in %s", versionNum, inventoryPath);
                }
                if (version.getMessage() == null) {
                    results.addIssue(ValidationCode.W007, "Inventory version %s should contain a message in %s", versionNum, inventoryPath);
                }
                if (version.getState() != null) {
                    for (String digest : version.getState().keySet()) {
                        unseenDigests.remove(digest);
                        results.addIssue(this.isTrue(manifest.containsKey(digest), ValidationCode.E050, "Inventory version %s contains digest %s that does not exist in the manifest in %s", versionNum, digest, inventoryPath));
                    }
                    this.validateDigestPathsMap(version.getState(), path -> results.addIssue(ValidationCode.E053, "Inventory version %s cannot contain paths that begin or end with '/' in %s. Found: %s", versionNum, inventoryPath, path), path -> results.addIssue(ValidationCode.E095, "Inventory version %s paths must be unique in %s. Found: %s", versionNum, inventoryPath, path), path -> results.addIssue(ValidationCode.E052, "Inventory version %s cannot contain blank path parts in %s. Found: %s", versionNum, inventoryPath, path), path -> results.addIssue(ValidationCode.E052, "Inventory version %s cannot contain path parts equal to '.' or '..' in %s. Found: %s", versionNum, inventoryPath, path), path -> results.addIssue(ValidationCode.E095, "Inventory version %s paths must be non-conflicting in %s. Found conflicting path: %s", versionNum, inventoryPath, path));
                    continue;
                }
                results.addIssue(ValidationCode.E048, "Inventory version %s must contain a state in %s", versionNum, inventoryPath);
            }
            for (Object digest : unseenDigests) {
                results.addIssue(ValidationCode.E107, "Inventory manifest in %s contains an entry that is not referenced in any version. Found: %s", inventoryPath, digest);
            }
        } else {
            results.addIssue(ValidationCode.E043, "Inventory versions must be set in %s", inventoryPath);
        }
    }

    private void validateInventoryVersionNumbers(SimpleInventory inventory, String inventoryPath, ValidationResultsBuilder results) {
        block4: {
            block5: {
                VersionNum highestVersion;
                if (inventory.getVersions() == null) break block4;
                if (inventory.getHead() != null && !inventory.getVersions().containsKey(inventory.getHead())) {
                    results.addIssue(ValidationCode.E010, "Inventory versions is missing an entry for version %s in %s", inventory.getHead(), inventoryPath);
                }
                if (inventory.getVersions().size() <= 0) break block5;
                TreeSet versions = new TreeSet(Comparator.naturalOrder());
                inventory.getVersions().keySet().forEach(version -> this.parseAndValidateVersionNum((String)version, inventoryPath, results).ifPresent(versions::add));
                long previousNum = 0L;
                Integer paddingWidth = null;
                boolean inconsistentPadding = false;
                for (VersionNum currentNum : versions) {
                    block8: {
                        long nextNum;
                        block6: {
                            block7: {
                                nextNum = previousNum + 1L;
                                if (currentNum.getVersionNum() != nextNum) break block6;
                                if (paddingWidth != null) break block7;
                                paddingWidth = currentNum.getZeroPaddingWidth();
                                break block8;
                            }
                            if (inconsistentPadding) break block8;
                            inconsistentPadding = paddingWidth.intValue() != currentNum.getZeroPaddingWidth();
                            break block8;
                        }
                        VersionNum missing = new VersionNum(nextNum, currentNum.getZeroPaddingWidth());
                        while (!missing.equals((Object)currentNum)) {
                            results.addIssue(ValidationCode.E010, "Inventory versions is missing an entry for version %s in %s", missing, inventoryPath);
                            missing = missing.nextVersionNum();
                        }
                    }
                    previousNum = currentNum.getVersionNum();
                }
                results.addIssue(this.isFalse(inconsistentPadding, ValidationCode.E013, "Inventory versions contain inconsistently padded version numbers in %s", inventoryPath));
                VersionNum versionNum = highestVersion = versions.isEmpty() ? null : (VersionNum)versions.last();
                if (highestVersion != null && inventory.getHead() != null) {
                    results.addIssue(this.isTrue(highestVersion.toString().equals(inventory.getHead()), ValidationCode.E040, "Inventory head must be the highest version number in %s. Expected: %s; Found: %s", inventoryPath, highestVersion, inventory.getHead()));
                }
                break block4;
            }
            results.addIssue(ValidationCode.E008, "Inventory must contain at least one version %s", inventoryPath);
        }
    }

    private void validateInventoryFixity(SimpleInventory inventory, String inventoryPath, ValidationResultsBuilder results) {
        if (inventory.getFixity() != null) {
            Map<String, Map<String, List<String>>> fixity = inventory.getFixity();
            for (Map.Entry<String, Map<String, List<String>>> entry : fixity.entrySet()) {
                String algorithm = entry.getKey();
                Map<String, List<String>> digestMap = entry.getValue();
                if (digestMap == null) continue;
                HashSet<String> digests = new HashSet<String>(digestMap.size());
                for (String digest : digestMap.keySet()) {
                    String digestLower = digest.toLowerCase();
                    if (!this.isDigestValidHex(digestLower, algorithm)) {
                        results.addIssue(ValidationCode.E057, "Inventory fixity block digests must be valid in %s. Found: %s", inventoryPath, digest);
                    }
                    if (digests.contains(digestLower)) {
                        results.addIssue(ValidationCode.E097, "Inventory fixity block cannot contain duplicates of digest %s in %s", digestLower, inventoryPath);
                        continue;
                    }
                    digests.add(digestLower);
                }
                this.validateDigestPathsMap(digestMap, path -> results.addIssue(ValidationCode.E100, "Inventory fixity block cannot contain content paths that begin or end with '/' in %s. Found: %s", inventoryPath, path), path -> results.addIssue(ValidationCode.E101, "Inventory fixity block content paths must be unique in %s. Found: %s", inventoryPath, path), path -> results.addIssue(ValidationCode.E099, "Inventory fixity block cannot contain blank content path parts in %s. Found: %s", inventoryPath, path), path -> results.addIssue(ValidationCode.E099, "Inventory fixity block cannot contain content path parts equal to '.' or '..' in %s. Found: %s", inventoryPath, path), path -> results.addIssue(ValidationCode.E101, "Inventory fixity block content paths must be non-conflicting in %s. Found conflicting path: %s", inventoryPath, path));
            }
        }
    }

    private void validateDigestPathsMap(Map<String, List<String>> map, Consumer<String> leadingTrailingSlashes, Consumer<String> nonUnique, Consumer<String> blankPart, Consumer<String> dotPart, Consumer<String> conflicting) {
        HashSet<String> check;
        HashSet<String> iter;
        HashSet<String> files = new HashSet<String>();
        HashSet<String> dirs = new HashSet<String>();
        for (List<String> paths : map.values()) {
            Iterator<String> iterator = paths.iterator();
            while (iterator.hasNext()) {
                String path2;
                String trimmedPath = path2 = iterator.next();
                boolean startsWith = path2.startsWith("/");
                boolean endsWith = path2.endsWith("/");
                if (startsWith || endsWith) {
                    leadingTrailingSlashes.accept(path2);
                    if (startsWith) {
                        trimmedPath = trimmedPath.substring(1);
                    }
                    if (endsWith) {
                        trimmedPath = trimmedPath.substring(0, trimmedPath.length() - 1);
                    }
                }
                if (files.contains(path2)) {
                    nonUnique.accept(path2);
                } else {
                    files.add(path2);
                }
                String[] parts = trimmedPath.split("/");
                StringBuilder pathBuilder = new StringBuilder();
                boolean erroredBlank = false;
                boolean erroredDot = false;
                for (int i = 0; i < parts.length; ++i) {
                    String part = parts[i];
                    if (!erroredBlank && part.isEmpty()) {
                        blankPart.accept(path2);
                        erroredBlank = true;
                    } else if (!erroredDot && (part.equals(".") || part.equals(".."))) {
                        dotPart.accept(path2);
                        erroredDot = true;
                    }
                    if (i >= parts.length - 1) continue;
                    if (i > 0) {
                        pathBuilder.append("/");
                    }
                    pathBuilder.append(part);
                    dirs.add(pathBuilder.toString());
                }
            }
        }
        if (files.size() > dirs.size()) {
            iter = dirs;
            check = files;
        } else {
            iter = files;
            check = dirs;
        }
        iter.forEach(path -> {
            if (check.contains(path)) {
                conflicting.accept((String)path);
            }
        });
    }

    private Optional<VersionNum> parseAndValidateVersionNum(String num, String inventoryPath, ValidationResultsBuilder results) {
        Optional<VersionNum> versionNum = Optional.empty();
        if (this.isInvalidVersionNum(num)) {
            results.addIssue(ValidationCode.E104, "Inventory contains invalid version number in %s. Found: %s", inventoryPath, num);
        } else {
            VersionNum parsed = VersionNum.fromString((String)num);
            if (parsed.equals((Object)VERSION_ZERO)) {
                results.addIssue(ValidationCode.E009, "Inventory version numbers must be greater than 0 in %s. Found: %s", inventoryPath, num);
            } else {
                versionNum = Optional.of(parsed);
            }
        }
        return versionNum;
    }

    private Optional<ValidationIssue> notBlank(String value, ValidationCode code, String messageTemplate, Object ... args) {
        if (value == null || value.isBlank()) {
            return Optional.of(this.createIssue(code, messageTemplate, args));
        }
        return Optional.empty();
    }

    private Optional<ValidationIssue> notNull(Object value, ValidationCode code, String messageTemplate, Object ... args) {
        if (value == null) {
            return Optional.of(this.createIssue(code, messageTemplate, args));
        }
        return Optional.empty();
    }

    private Optional<ValidationIssue> isTrue(boolean condition, ValidationCode code, String messageTemplate, Object ... args) {
        if (!condition) {
            return Optional.of(this.createIssue(code, messageTemplate, args));
        }
        return Optional.empty();
    }

    private Optional<ValidationIssue> isFalse(boolean condition, ValidationCode code, String messageTemplate, Object ... args) {
        if (condition) {
            return Optional.of(this.createIssue(code, messageTemplate, args));
        }
        return Optional.empty();
    }

    private boolean isUri(String value) {
        try {
            URI uri = new URI(value);
            return uri.getScheme() != null && !uri.getScheme().isBlank();
        }
        catch (URISyntaxException e) {
            return false;
        }
    }

    private boolean isInvalidVersionNum(String num) {
        return num == null || !VALID_VERSION.matcher(num).matches();
    }

    private Optional<ValidationIssue> ifNotNull(Object value, Supplier<Optional<ValidationIssue>> condition) {
        if (value != null) {
            return condition.get();
        }
        return Optional.empty();
    }

    private boolean isDigestValidHex(String lowerDigest, String algorithm) {
        if (algorithm != null && DIGEST_LENGTHS.containsKey(algorithm)) {
            Integer length = DIGEST_LENGTHS.get(algorithm);
            if (lowerDigest.length() != length.intValue()) {
                return false;
            }
            for (int i = 0; i < lowerDigest.length(); ++i) {
                if (this.lowerHexChars.get(lowerDigest.charAt(i))) continue;
                return false;
            }
        }
        return true;
    }

    private ValidationIssue createIssue(ValidationCode code, String messageTemplate, Object ... args) {
        String message = messageTemplate;
        if (args != null && args.length > 0) {
            message = String.format(messageTemplate, args);
        }
        return new ValidationIssue(code, message);
    }

    public static enum VersionEquality {
        EQUAL,
        LESS_THAN_OR_EQUAL;

    }
}

