package enterprises.iwakura.economysystem.api;

import enterprises.iwakura.economysystem.api.dto.AccountingTransactionDTO;
import enterprises.iwakura.economysystem.api.dto.BankingAccountDTO;
import enterprises.iwakura.economysystem.api.dto.BankingAccountTypeEnum;
import enterprises.iwakura.economysystem.api.dto.StandingOrderDTO;
import enterprises.iwakura.economysystem.api.events.AccountingTransactionEventDTO;
import gol.economy.exception.CannotCreateUniqueStandingOrderException;
import gol.economy.exception.EntityDoesNotExistException;
import gol.economy.listener.AccountingTransactionListener;
import gol.economy.service.BankingService;
import gol.economy.utils.AsyncUtils;
import gol.economy.utils.SentryIntegration;
import lombok.Data;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;
import org.bukkit.plugin.Plugin;

import java.math.BigDecimal;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

/**
 * API class for accessing Economy System's features.
 */
@Data
@Slf4j
public class EconomySystemAPI {

    private final Plugin plugin;

    /**
     * Creates an instance of EconomySystemAPI with your plugin.
     *
     * @param plugin The plugin instance that is using the API.
     */
    public EconomySystemAPI(Plugin plugin) {
        this.plugin = plugin;
        AccountingTransactionListener.apiEventConsumer = (event) -> AsyncUtils.callEventAsync(AccountingTransactionEventDTO.fromEntity(event));
    }

    /**
     * Transfers money between two entities.
     *
     * @param fromEntity            From entity
     * @param fromEntityAccountType From entity account type
     * @param toEntity              To entity
     * @param toEntityAccountType   To entity account type
     * @param amount                Amount to transfer
     * @param detail                Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> transfer(
            UUID fromEntity, BankingAccountTypeEnum fromEntityAccountType,
            UUID toEntity, BankingAccountTypeEnum toEntityAccountType,
            BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_transfer_1", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is transferring {} from {}[{}] to {}[{}]",
                plugin.getName(), amount, fromEntity, fromEntityAccountType,
                toEntity, toEntityAccountType
            );
            final var transaction = BankingService.INSTANCE.transfer(
                    fromEntity, fromEntityAccountType.toEntity(),
                    toEntity, toEntityAccountType.toEntity(),
                    amount, detail
            );
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Transfers money between two accounts.
     *
     * @param fromAccountNumber From account number
     * @param toAccountNumber   To account number
     * @param amount            Amount to transfer
     * @param detail            Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     */
    public CompletableFuture<AccountingTransactionDTO> transfer(
            Long fromAccountNumber, Long toAccountNumber,
            BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_transfer_2", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is transferring {} from {} to {}",
                plugin.getName(), amount, fromAccountNumber, toAccountNumber
            );
            final var transaction = BankingService.INSTANCE.transfer(fromAccountNumber, toAccountNumber, amount, detail, false);
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Withdraws money from an account.
     *
     * @param accountNumber Account number
     * @param amount        Amount to withdraw
     * @param detail        Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> withdraw(
            Long accountNumber, BigDecimal amount, String detail, boolean silent, boolean allowNegative
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_withdraw_1", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is withdrawing {} from {}",
                plugin.getName(), amount, accountNumber
            );
            final var transaction = BankingService.INSTANCE.withdrawBalance(accountNumber, amount, detail, silent, allowNegative);
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Withdraws money from an account.
     *
     * @param entity Entity ID
     * @param type   Account type
     * @param amount Amount to withdraw
     * @param detail Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> withdraw(
            UUID entity, BankingAccountTypeEnum type, BigDecimal amount, String detail, boolean silent, boolean allowNegative
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_withdraw_2", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is withdrawing {} from {}[{}]",
                plugin.getName(), amount, entity, type
            );
            final var transaction = BankingService.INSTANCE.withdrawBalance(
                    entity, type.toEntity(), amount, detail, silent, allowNegative
            );
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Deposits money to an account.
     *
     * @param accountNumber Account number
     * @param amount        Amount to deposit
     * @param detail        Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> deposit(
            Long accountNumber, BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_deposit_1", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is depositing {} to {}",
                plugin.getName(), amount, accountNumber
            );
            final var transaction = BankingService.INSTANCE.depositBalance(accountNumber, amount, detail, false);
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Deposits money to an account.
     *
     * @param entity Entity ID
     * @param type   Account type
     * @param amount Amount to deposit
     * @param detail Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> deposit(
            UUID entity, BankingAccountTypeEnum type, BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_deposit_2", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is depositing {} to {}[{}]",
                plugin.getName(), amount, entity, type
            );
            final var transaction = BankingService.INSTANCE.depositBalance(
                    entity, type.toEntity(), amount, detail, false
            );
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Gets the accounts for the given entity ID.
     *
     * @param entityId Entity ID
     *
     * @return A CompletableFuture that will be completed with the list of account DTOs for the given entity ID. If entity has no accounts, the list will
     * be empty.
     */
    public CompletableFuture<List<BankingAccountDTO>> getAccounts(UUID entityId) {
        final var completableFuture = new CompletableFuture<List<BankingAccountDTO>>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_get_accounts", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            final var accounts = BankingService.INSTANCE.getAccountsForEntityId(entityId);
            completableFuture.complete(accounts.stream().map(BankingAccountDTO::fromEntity).toList());
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Gets the account for the given entity ID and account type.
     *
     * @param entityId    Entity ID
     * @param accountType Account type
     *
     * @return A CompletableFuture that will be completed with the account DTO for the given entity ID and account type. If entity has no account of the
     * given type, the future will be completed exceptionally.
     */
    public CompletableFuture<BankingAccountDTO> getAccount(UUID entityId, BankingAccountTypeEnum accountType) {
        final var completableFuture = new CompletableFuture<BankingAccountDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_get_account", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            final var account = BankingService.INSTANCE.getAccountForEntityId(entityId, accountType.toEntity()).orElseThrow(() -> new EntityDoesNotExistException("account", entityId, accountType));
            completableFuture.complete(BankingAccountDTO.fromEntity(account));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Gets the infinite account for the given name.
     * @param name Name of the infinite account to retrieve.
     * @return A CompletableFuture that will be completed with the BankingAccountDTO for the infinite account.
     */
    public CompletableFuture<BankingAccountDTO> getInfiniteAccount(String name) {
        final var completableFuture = new CompletableFuture<BankingAccountDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_get_infinite_account", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is getting infinite account for name {}", plugin.getName(), name);
            final var account = BankingService.INSTANCE.getInfiniteAccount(name).orElseThrow(() -> new EntityDoesNotExistException("infinite account", name));
            completableFuture.complete(BankingAccountDTO.fromEntity(account));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Creates a basic standing order.
     *
     * @param fromAccountNumber From account number
     * @param toAccountNumber To account number
     * @param amount Amount to transfer
     * @param gameDayPeriod Game day period for the standing order (how often it should be processed)
     * @param fromAccountEntityMustBeOnline If true, the from account entity must be online for the standing order to be processed
     * @param note Note for the standing order
     *
     * @return A CompletableFuture that will be completed with the StandingOrderDTO. If the standing order creation fails due to invalid arguments or
     */
    public CompletableFuture<StandingOrderDTO> createBasicStandingOrder(
            long fromAccountNumber,
            long toAccountNumber,
            BigDecimal amount,
            int gameDayPeriod,
            boolean fromAccountEntityMustBeOnline,
            @NonNull String note
    ) {
        final var completableFuture = new CompletableFuture<StandingOrderDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_create_basic_standing_order", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is creating a basic standing order from {} to {} every {} game days (must be online: {}) with note: {}",
                plugin.getName(), fromAccountNumber, toAccountNumber, gameDayPeriod, fromAccountEntityMustBeOnline, note
            );
            final var standingOrder = BankingService.INSTANCE.createBasicStandingOrder(
                    fromAccountNumber,
                    toAccountNumber,
                    amount,
                    gameDayPeriod,
                    fromAccountEntityMustBeOnline,
                    note, false
            );
            completableFuture.complete(StandingOrderDTO.fromEntity(standingOrder));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Creates a basic standing order.
     *
     * @param fromEntityUuid From entity UUID
     * @param fromEntityAccountType From entity account type
     * @param toEntityUuid To entity UUID
     * @param toEntityAccountType To entity account type
     * @param amount Amount to transfer
     * @param gameDayPeriod Game day period for the standing order (how often it should be processed)
     * @param fromAccountEntityMustBeOnline If true, the from account entity must be online for the standing order to be processed
     * @param note Note for the standing order
     *
     * @return A CompletableFuture that will be completed with the StandingOrderDTO. If the standing order creation fails due to invalid arguments or
     */
    public CompletableFuture<StandingOrderDTO> createBasicStandingOrder(
            UUID fromEntityUuid, BankingAccountTypeEnum fromEntityAccountType,
            UUID toEntityUuid, BankingAccountTypeEnum toEntityAccountType,
            BigDecimal amount,
            int gameDayPeriod,
            boolean fromAccountEntityMustBeOnline,
            @NonNull String note
    ) {
        final var completableFuture = new CompletableFuture<StandingOrderDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_create_basic_standing_order_2", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is creating a basic standing order from {}[{}] to {}[{}] every {} game days (must be online: {}) with note: {}",
                plugin.getName(), fromEntityUuid, fromEntityAccountType, toEntityUuid, toEntityAccountType, gameDayPeriod, fromAccountEntityMustBeOnline, note
            );
            final var standingOrder = BankingService.INSTANCE.createBasicStandingOrder(
                    fromEntityUuid, fromEntityAccountType.toEntity(),
                    toEntityUuid, toEntityAccountType.toEntity(),
                    amount,
                    gameDayPeriod,
                    fromAccountEntityMustBeOnline,
                    note, false
            );
            completableFuture.complete(StandingOrderDTO.fromEntity(standingOrder));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Creates a unique standing order. If there's already a standing order with the same note, it will be updated instead of creating a new one.
     * If the specified amount is negative, it deletes the existing standing order. If no standing order exists with the specified note and the amount
     * is less than or equal to 0, it throws an CannotCreateUniqueStandingOrderException.
     *
     * @param fromAccountNumber From account number
     * @param toAccountNumber To account number
     * @param amount Amount to transfer
     * @param gameDayPeriod Game day period for the standing order (how often it should be processed)
     * @param fromAccountEntityMustBeOnline If true, the from account entity must be online for the standing order to be processed
     * @param uniqueNote Unique note for the standing order
     * @param silent If true, the standing order will be processed silently without sending any messages to the players involved.
     *
     * @return A CompletableFuture that will be completed with the StandingOrderDTO. If the standing order creation fails due to invalid arguments or any other way,
     * the future will be completed exceptionally.
     * @throws CannotCreateUniqueStandingOrderException if no standing order exists with the specified note and the amount is less than or equal to 0.
     */
    public CompletableFuture<StandingOrderDTO> createUniqueStandingOrder(
            long fromAccountNumber,
            long toAccountNumber,
            BigDecimal amount,
            int gameDayPeriod,
            boolean fromAccountEntityMustBeOnline,
            @NonNull String uniqueNote, boolean silent
    ) {
        final var completableFuture = new CompletableFuture<StandingOrderDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_create_unique_standing_order", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is creating a unique standing order from {} to {} every {} game days (must be online: {}) with note: {}",
                     plugin.getName(), fromAccountNumber, toAccountNumber, gameDayPeriod, fromAccountEntityMustBeOnline, uniqueNote
            );
            final var standingOrder = BankingService.INSTANCE.createUniqueStandingOrder(
                    fromAccountNumber,
                    toAccountNumber,
                    amount,
                    gameDayPeriod,
                    fromAccountEntityMustBeOnline,
                    uniqueNote, silent
            );
            completableFuture.complete(StandingOrderDTO.fromEntity(standingOrder));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Creates a unique standing order. If there's already a standing order with the same note, it will be updated instead of creating a new one.
     * If the specified amount is negative, it deletes the existing standing order. If no standing order exists with the specified note and the amount
     * is less than or equal to 0, it throws an CannotCreateUniqueStandingOrderException.
     *
     * @param fromEntityUuid From entity UUID
     * @param fromEntityAccountType From entity account type
     * @param toEntityUuid To entity UUID
     * @param toEntityAccountType To entity account type
     * @param amount Amount to transfer
     * @param gameDayPeriod Game day period for the standing order (how often it should be processed)
     * @param fromAccountEntityMustBeOnline If true, the from account entity must be online for the standing order to be processed
     * @param uniqueNote Unique note for the standing order
     * @param silent If true, the standing order will be processed silently without sending any messages to the players involved.
     *
     * @return A CompletableFuture that will be completed with the StandingOrderDTO. If the standing order creation fails due to invalid arguments or any other way,
     * the future will be completed exceptionally.
     * @throws CannotCreateUniqueStandingOrderException if no standing order exists with the specified note and the amount is less than or equal to 0.
     */
    public CompletableFuture<StandingOrderDTO> createUniqueStandingOrder(
            UUID fromEntityUuid, BankingAccountTypeEnum fromEntityAccountType,
            UUID toEntityUuid, BankingAccountTypeEnum toEntityAccountType,
            BigDecimal amount,
            int gameDayPeriod,
            boolean fromAccountEntityMustBeOnline,
            @NonNull String uniqueNote, boolean silent
    ) {
        final var completableFuture = new CompletableFuture<StandingOrderDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_create_unique_standing_order_2", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is creating a unique standing order from {}[{}] to {}[{}] every {} game days (must be online: {}) with note: {}",
                     plugin.getName(), fromEntityUuid, fromEntityAccountType, toEntityUuid, toEntityAccountType, gameDayPeriod, fromAccountEntityMustBeOnline, uniqueNote
            );
            final var standingOrder = BankingService.INSTANCE.createUniqueStandingOrder(
                    fromEntityUuid, fromEntityAccountType.toEntity(),
                    toEntityUuid, toEntityAccountType.toEntity(),
                    amount,
                    gameDayPeriod,
                    fromAccountEntityMustBeOnline,
                    uniqueNote, silent
            );
            completableFuture.complete(StandingOrderDTO.fromEntity(standingOrder));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Creates a unique standing order. If there's already a standing order with the same note, it will be updated instead of creating a new one.
     * If the specified amount is negative, it deletes the existing standing order. If no standing order exists with the specified note and the amount
     * is less than or equal to 0, it throws an CannotCreateUniqueStandingOrderException.
     *
     * @param fromEntityUuid From entity UUID
     * @param fromEntityAccountType From entity account type
     * @param infiniteAccountName Name of the infinite account
     * @param amount Amount to transfer, should be positive to create a standing order, or negative to delete an existing one.
     * @param gameDayPeriod Game day period for the standing order (how often it should be processed)
     * @param fromAccountEntityMustBeOnline If true, the from account entity must be online for the standing order to be processed
     * @param uniqueNote Unique note for the standing order
     * @param silent If true, the standing order will be processed silently without sending any messages to the players involved.
     *
     * @return A CompletableFuture that will be completed with the StandingOrderDTO. If the standing order creation fails due to invalid arguments or any other way,
     * the future will be completed exceptionally.
     * @throws CannotCreateUniqueStandingOrderException if no standing order exists with the specified note and the amount is less than or equal to 0.
     */
    public CompletableFuture<StandingOrderDTO> createUniqueStandingOrder(
            UUID fromEntityUuid, BankingAccountTypeEnum fromEntityAccountType,
            String infiniteAccountName,
            BigDecimal amount,
            int gameDayPeriod,
            boolean fromAccountEntityMustBeOnline,
            @NonNull String uniqueNote, boolean silent
    ) {
        final var completableFuture = new CompletableFuture<StandingOrderDTO>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_create_unique_standing_order_3", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is creating a unique standing order from {}[{}] to infinite account {} every {} game days (must be online: {}) with note: {}",
                     plugin.getName(), fromEntityUuid, fromEntityAccountType, infiniteAccountName, gameDayPeriod, fromAccountEntityMustBeOnline, uniqueNote
            );
            final var standingOrder = BankingService.INSTANCE.createUniqueStandingOrder(
                    fromEntityUuid, fromEntityAccountType.toEntity(),
                    infiniteAccountName,
                    amount,
                    gameDayPeriod,
                    fromAccountEntityMustBeOnline,
                    uniqueNote, silent
            );
            completableFuture.complete(StandingOrderDTO.fromEntity(standingOrder));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Gets standing orders based on the provided criteria.
     * @param fromAccountNumber From account number to filter standing orders.
     * @return A CompletableFuture that will be completed with a list of StandingOrderDTOs matching the criteria.
     */
    public CompletableFuture<List<StandingOrderDTO>> getStandingOrders(
            long fromAccountNumber
    ) {
        final var completableFuture = new CompletableFuture<List<StandingOrderDTO>>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_get_standing_order", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is getting standing orders from {}",
                     plugin.getName(), fromAccountNumber
            );
            final var standingOrders = BankingService.INSTANCE.getStandingOrders(fromAccountNumber);
            completableFuture.complete(standingOrders.stream()
                                                     .map(StandingOrderDTO::fromEntity)
                                                     .toList());
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Gets standing orders based on the provided criteria.
     * @param fromAccountNumber From account number to filter standing orders.
     * @param toAccountNumber To account number to filter standing orders.
     * @param note Note to filter standing orders
     * @return A CompletableFuture that will be completed with a list of StandingOrderDTOs matching the criteria.
     */
    public CompletableFuture<List<StandingOrderDTO>> getStandingOrders(
            long fromAccountNumber,
            long toAccountNumber,
            @NonNull String note
    ) {
        final var completableFuture = new CompletableFuture<List<StandingOrderDTO>>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_get_standing_order", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is getting standing orders from {} to {} with note: {}",
                plugin.getName(), fromAccountNumber, toAccountNumber, note
            );
            final var standingOrders = BankingService.INSTANCE.getStandingOrders(fromAccountNumber, toAccountNumber, note);
            completableFuture.complete(standingOrders.stream()
                    .map(StandingOrderDTO::fromEntity)
                    .toList());
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Removes a standing order by its ID.
     * @param standingOrderId ID of the standing order to remove.
     * @return A CompletableFuture that will be completed when the standing order is removed. If the removal fails (e.g., standing order does not exist),
     * the future will be completed exceptionally.
     */
    public CompletableFuture<Void> removeStandingOrderById(long standingOrderId) {
        final var completableFuture = new CompletableFuture<Void>();
        //@formatter:off
        AsyncUtils.runAsync("external_plugin_remove_standing_order", () -> {
            SentryIntegration.setSpanData("external_plugin", plugin.getName());

            log.debug("Plugin {} is removing standing order with ID: {}", plugin.getName(), standingOrderId);
            BankingService.INSTANCE.removeStandingOrderById(standingOrderId);
            completableFuture.complete(null);
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }
}
