package enterprises.iwakura.economysystem.api.events;

import enterprises.iwakura.economysystem.api.dto.AccountingTransactionDTO;
import enterprises.iwakura.economysystem.api.dto.LoanDTO;
import enterprises.iwakura.economysystem.api.dto.StandingOrderDTO;
import gol.economy.event.AccountingTransactionEvent;
import lombok.Data;
import lombok.EqualsAndHashCode;
import org.bukkit.event.Event;
import org.bukkit.event.HandlerList;
import org.jetbrains.annotations.NotNull;

/**
 * Event representing an accounting transaction.
 * This event is triggered when a transaction occurs, such as a transfer of balance.<br>
 * Please note that this even is not cancellable and is asynchronous.<br>
 * <b>In order for this event to be fired, you must create EconomySystemAPI</b> (or any other plugin)
 */
@EqualsAndHashCode(callSuper = true)
@Data
public class AccountingTransactionEventDTO extends Event {

    private static final HandlerList handlers = new HandlerList();

    private final AccountingTransactionDTO transactionDto;
    private final StandingOrderDTO causedByStandingOrder;
    private final LoanDTO causedByLoan;
    private final boolean silent;

    public AccountingTransactionEventDTO(
            AccountingTransactionDTO transactionDto,
            StandingOrderDTO causedByStandingOrder,
            LoanDTO causedByLoan,
            boolean silent) {
        super(true);
        this.transactionDto = transactionDto;
        this.causedByStandingOrder = causedByStandingOrder;
        this.causedByLoan = causedByLoan;
        this.silent = silent;
    }

    public static HandlerList getHandlerList() {
        return handlers;
    }

    /**
     * Creates a new AccountingTransactionEventDTO from the given event.
     *
     * @param event The event to convert
     * @return A new instance of AccountingTransactionEventDTO
     */
    public static AccountingTransactionEventDTO fromEntity(AccountingTransactionEvent event) {
        return new AccountingTransactionEventDTO(
                AccountingTransactionDTO.fromEntity(event.getTransactionEntity()),
                event.getStandingOrderEntity() != null ? StandingOrderDTO.fromEntity(event.getStandingOrderEntity()) : null,
                event.getLoanEntity() != null ? LoanDTO.fromEntity(event.getLoanEntity()) : null,
                event.isSilent()
        );
    }

    @Override
    public @NotNull HandlerList getHandlers() {
        return handlers;
    }

    /**
     * Determines if the event was caused by a standing order.
     *
     * @return true if the event was caused by a standing order, false otherwise.
     */
    public boolean isCausedByStandingOrder() {
        return causedByStandingOrder != null;
    }

    /**
     * Determines if the event was caused by a loan.
     *
     * @return true if the event was caused by a loan, false otherwise.
     */
    public boolean isCausedByLoan() {
        return causedByLoan != null;
    }
}
