package enterprises.iwakura.economysystem.api;

import enterprises.iwakura.economysystem.api.dto.AccountingTransactionDTO;
import enterprises.iwakura.economysystem.api.dto.BankingAccountDTO;
import enterprises.iwakura.economysystem.api.dto.BankingAccountTypeEnum;
import gol.economy.service.BankingService;
import gol.economy.utils.AsyncUtils;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.bukkit.plugin.Plugin;

import java.math.BigDecimal;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

/**
 * API class for accessing Economy System's features.
 */
@Data
@Slf4j
public class EconomySystemAPI {

    private final Plugin plugin;

    /**
     * Creates an instance of EconomySystemAPI with your plugin.
     *
     * @param plugin The plugin instance that is using the API.
     */
    public EconomySystemAPI(Plugin plugin) {
        this.plugin = plugin;
    }

    /**
     * Transfers money between two entities.
     *
     * @param fromEntity            From entity
     * @param fromEntityAccountType From entity account type
     * @param toEntity              To entity
     * @param toEntityAccountType   To entity account type
     * @param amount                Amount to transfer
     * @param detail                Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> transfer(
            UUID fromEntity, BankingAccountTypeEnum fromEntityAccountType,
            UUID toEntity, BankingAccountTypeEnum toEntityAccountType,
            BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync(() -> {
            log.info("Plugin {} is transferring {} from {}[{}] to {}[{}]",
                plugin.getName(), amount, fromEntity, fromEntityAccountType,
                toEntity, toEntityAccountType
            );
            final var transaction = BankingService.INSTANCE.transfer(
                    fromEntity, fromEntityAccountType.toEntity(),
                    toEntity, toEntityAccountType.toEntity(),
                    amount, detail
            );
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Transfers money between two accounts.
     *
     * @param fromAccountNumber From account number
     * @param toAccountNumber   To account number
     * @param amount            Amount to transfer
     * @param detail            Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     */
    public CompletableFuture<AccountingTransactionDTO> transfer(
            Long fromAccountNumber, Long toAccountNumber,
            BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync(() -> {
            log.info("Plugin {} is transferring {} from {} to {}",
                plugin.getName(), amount, fromAccountNumber, toAccountNumber
            );
            final var transaction = BankingService.INSTANCE.transfer(fromAccountNumber, toAccountNumber, amount, detail);
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Withdraws money from an account.
     *
     * @param accountNumber Account number
     * @param amount        Amount to withdraw
     * @param detail        Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> withdraw(
            Long accountNumber, BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync(() -> {
            log.info("Plugin {} is withdrawing {} from {}",
                plugin.getName(), amount, accountNumber
            );
            final var transaction = BankingService.INSTANCE.withdrawBalance(accountNumber, amount, detail);
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Withdraws money from an account.
     *
     * @param entity Entity ID
     * @param type   Account type
     * @param amount Amount to withdraw
     * @param detail Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> withdraw(
            UUID entity, BankingAccountTypeEnum type, BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync(() -> {
            log.info("Plugin {} is withdrawing {} from {}[{}]",
                plugin.getName(), amount, entity, type
            );
            final var transaction = BankingService.INSTANCE.withdrawBalance(
                    entity, type.toEntity(), amount, detail
            );
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Deposits money to an account.
     *
     * @param accountNumber Account number
     * @param amount        Amount to deposit
     * @param detail        Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> deposit(
            Long accountNumber, BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync(() -> {
            log.info("Plugin {} is depositing {} to {}",
                plugin.getName(), amount, accountNumber
            );
            final var transaction = BankingService.INSTANCE.depositBalance(accountNumber, amount, detail);
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Deposits money to an account.
     *
     * @param entity Entity ID
     * @param type   Account type
     * @param amount Amount to deposit
     * @param detail Transaction detail
     *
     * @return A CompletableFuture that will be completed with the transaction DTO. If transactions fails due to non-existing accounts or invalid
     * arguments (such as non-positive amount), the future will be completed exceptionally.
     */
    public CompletableFuture<AccountingTransactionDTO> deposit(
            UUID entity, BankingAccountTypeEnum type, BigDecimal amount, String detail
    ) {
        final var completableFuture = new CompletableFuture<AccountingTransactionDTO>();
        //@formatter:off
        AsyncUtils.runAsync(() -> {
            log.info("Plugin {} is depositing {} to {}[{}]",
                plugin.getName(), amount, entity, type
            );
            final var transaction = BankingService.INSTANCE.depositBalance(
                    entity, type.toEntity(), amount, detail
            );
            completableFuture.complete(AccountingTransactionDTO.fromEntity(transaction));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Gets the accounts for the given entity ID.
     *
     * @param entityId Entity ID
     *
     * @return A CompletableFuture that will be completed with the list of account DTOs for the given entity ID. If entity has no accounts, the list will
     * be empty.
     */
    public CompletableFuture<List<BankingAccountDTO>> getAccounts(UUID entityId) {
        final var completableFuture = new CompletableFuture<List<BankingAccountDTO>>();
        //@formatter:off
        AsyncUtils.runAsync(() -> {
            final var accounts = BankingService.INSTANCE.getAccountsForEntityId(entityId);
            completableFuture.complete(accounts.stream().map(BankingAccountDTO::fromEntity).toList());
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }

    /**
     * Gets the account for the given entity ID and account type.
     *
     * @param entityId    Entity ID
     * @param accountType Account type
     *
     * @return A CompletableFuture that will be completed with the account DTO for the given entity ID and account type. If entity has no account of the
     * given type, the future will be completed exceptionally.
     */
    public CompletableFuture<BankingAccountDTO> getAccount(UUID entityId, BankingAccountTypeEnum accountType) {
        final var completableFuture = new CompletableFuture<BankingAccountDTO>();
        //@formatter:off
        AsyncUtils.runAsync(() -> {
            final var account = BankingService.INSTANCE.getAccountForEntityId(entityId, accountType.toEntity()).orElseThrow();
            completableFuture.complete(BankingAccountDTO.fromEntity(account));
        }, completableFuture::completeExceptionally);
        //@formatter:on
        return completableFuture;
    }
}
