/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package eus.ixa.ixa.pipe.ml.parse;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import opennlp.tools.dictionary.Dictionary;
import opennlp.tools.ml.model.Event;
import opennlp.tools.parser.ParserEventTypeEnum;
import opennlp.tools.util.ObjectStream;

/**
 * Abstract class extended by parser event streams which perform tagging and chunking.
 */
public abstract class AbstractParserEventStream extends opennlp.tools.util.AbstractEventStream<Parse> {

  protected HeadRules rules;
  protected Set<String> punctSet;

  /**
   * The type of events being generated by this event stream.
   */
  protected ParserEventTypeEnum etype;
  protected boolean fixPossesives;
  protected Dictionary dict;

  public AbstractParserEventStream(ObjectStream<Parse> dataStream, HeadRules rules, ParserEventTypeEnum etype) {
    super(dataStream);
    //this.dict = dict;
    this.rules = rules;
    punctSet = rules.getPunctuationTags();
    this.etype = etype;
    init();
  }

  @Override
  protected Iterator<Event> createEvents(Parse sample) {
    List<Event> newEvents = new ArrayList<Event>();

    Parse.pruneParse(sample);
    if (fixPossesives) {
      Parse.fixPossesives(sample);
    }
    sample.updateHeads(rules);
    Parse[] chunks = getInitialChunks(sample);
    addParseEvents(newEvents, ShiftReduceParser.collapsePunctuation(chunks, punctSet));
    return newEvents.iterator();
  }

  protected void init() {
    fixPossesives = false;
  }

  public static Parse[] getInitialChunks(Parse p) {
    List<Parse> chunks = new ArrayList<Parse>();
    getInitialChunks(p, chunks);
    return chunks.toArray(new Parse[chunks.size()]);
  }

  private static void getInitialChunks(Parse p, List<Parse> ichunks) {
    if (p.isPosTag()) {
      ichunks.add(p);
    }
    else {
      Parse[] kids = p.getChildren();
      boolean allKidsAreTags = true;
      for (int ci = 0, cl = kids.length; ci < cl; ci++) {
        if (!kids[ci].isPosTag()) {
          allKidsAreTags = false;
          break;
        }
      }
      if (allKidsAreTags) {
        ichunks.add(p);
      }
      else {
        for (int ci = 0, cl = kids.length; ci < cl; ci++) {
          getInitialChunks(kids[ci], ichunks);
        }
      }
    }
  }

  /**
   * Produces all events for the specified sentence chunks
   * and adds them to the specified list.
   * @param newEvents A list of events to be added to.
   * @param chunks Pre-chunked constituents of a sentence.
   */
  protected abstract void addParseEvents(List<Event> newEvents, Parse[] chunks);

  /**
   * Returns true if the specified child is the last child of the specified parent.
   * @param child The child parse.
   * @param parent The parent parse.
   * @return true if the specified child is the last child of the specified parent; false otherwise.
   */
  protected boolean lastChild(Parse child, Parse parent) {
    Parse[] kids = ShiftReduceParser.collapsePunctuation(parent.getChildren(),punctSet);
    return (kids[kids.length - 1] == child);
  }

}

