/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eus.ixa.ixa.pipe.ml.parse;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import eus.ixa.ixa.pipe.ml.sequence.BioCodec;
import opennlp.tools.ml.model.Event;
import opennlp.tools.parser.ParserEventTypeEnum;
import opennlp.tools.util.ObjectStream;

/**
 * Wrapper class for one of four parser event streams. The particular event
 * stream is specified at construction.
 */
public class ParserEventStream
    extends opennlp.tools.util.AbstractEventStream<Parse> {

  private final HeadRules rules;
  private final Set<String> punctSet;

  /**
   * The type of events being generated by this event stream.
   */
  private final ParserEventTypeEnum etype;
  private boolean fixPossesives;
  private BuildContextGenerator bcg;
  private CheckContextGenerator kcg;

  public ParserEventStream(final ObjectStream<Parse> samples,
      final HeadRules rules, final ParserEventTypeEnum etype,
      final ParserFactory factory) {
    super(samples);
    this.rules = rules;
    this.punctSet = rules.getPunctuationTags();
    this.etype = etype;
    init();
  }

  public ParserEventStream(final ObjectStream<Parse> d, final HeadRules rules,
      final ParserEventTypeEnum etype) {
    this(d, rules, etype, null);
  }

  private void init() {
    if (this.etype == ParserEventTypeEnum.BUILD) {
      this.bcg = new BuildContextGenerator();
    } else if (this.etype == ParserEventTypeEnum.CHECK) {
      this.kcg = new CheckContextGenerator();
    }
    this.fixPossesives = false;
  }

  @Override
  protected Iterator<Event> createEvents(final Parse sample) {
    final List<Event> newEvents = new ArrayList<Event>();

    Parse.pruneParse(sample);
    if (this.fixPossesives) {
      Parse.fixPossesives(sample);
    }
    sample.updateHeads(this.rules);
    final Parse[] chunks = getInitialChunks(sample);
    addParseEvents(newEvents,
        ShiftReduceParser.collapsePunctuation(chunks, this.punctSet));
    return newEvents.iterator();
  }

  /**
   * Adds events for parsing (post tagging and chunking to the specified list of
   * events for the specified parse chunks.
   *
   * @param parseEvents
   *          The events for the specified chunks.
   * @param chunks
   *          The incomplete parses to be parsed.
   */
  private void addParseEvents(final List<Event> parseEvents, Parse[] chunks) {
    int ci = 0;
    while (ci < chunks.length) {
      // System.err.println("parserEventStream.addParseEvents: chunks=" +
      // Arrays.asList(chunks));
      final Parse c = chunks[ci];
      final Parse parent = c.getParent();
      if (parent != null) {
        final String type = parent.getType();
        String outcome;
        if (firstChild(c, parent)) {
          outcome = type + "-" + BioCodec.START;
        } else {
          outcome = type + "-" + BioCodec.CONTINUE;
        }
        // System.err.println("parserEventStream.addParseEvents:
        // chunks["+ci+"]="+c+" label=" +outcome + " bcg=" + bcg);
        c.setLabel(outcome);
        if (this.etype == ParserEventTypeEnum.BUILD) {
          parseEvents.add(new Event(outcome, this.bcg.getContext(chunks, ci)));
        }
        int start = ci - 1;
        while (start >= 0 && chunks[start].getParent() == parent) {
          start--;
        }
        if (lastChild(c, parent)) {
          if (this.etype == ParserEventTypeEnum.CHECK) {
            parseEvents.add(new Event(ShiftReduceParser.COMPLETE,
                this.kcg.getContext(chunks, type, start + 1, ci)));
          }
          // perform reduce
          int reduceStart = ci;
          while (reduceStart >= 0
              && chunks[reduceStart].getParent() == parent) {
            reduceStart--;
          }
          reduceStart++;
          chunks = reduceChunks(chunks, ci, parent);
          ci = reduceStart - 1; // ci will be incremented at end of loop
        } else {
          if (this.etype == ParserEventTypeEnum.CHECK) {
            parseEvents.add(new Event(ShiftReduceParser.INCOMPLETE,
                this.kcg.getContext(chunks, type, start + 1, ci)));
          }
        }
      }
      ci++;
    }
  }

  public static Parse[] getInitialChunks(final Parse p) {
    final List<Parse> chunks = new ArrayList<Parse>();
    getInitialChunks(p, chunks);
    return chunks.toArray(new Parse[chunks.size()]);
  }

  private static void getInitialChunks(final Parse p,
      final List<Parse> ichunks) {
    if (p.isPosTag()) {
      ichunks.add(p);
    } else {
      final Parse[] kids = p.getChildren();
      boolean allKidsAreTags = true;
      for (int ci = 0, cl = kids.length; ci < cl; ci++) {
        if (!kids[ci].isPosTag()) {
          allKidsAreTags = false;
          break;
        }
      }
      if (allKidsAreTags) {
        ichunks.add(p);
      } else {
        for (final Parse kid : kids) {
          getInitialChunks(kid, ichunks);
        }
      }
    }
  }

  /**
   * Returns true if the specified child is the first child of the specified
   * parent.
   *
   * @param child
   *          The child parse.
   * @param parent
   *          The parent parse.
   * @return true if the specified child is the first child of the specified
   *         parent; false otherwise.
   */
  protected boolean firstChild(final Parse child, final Parse parent) {
    return ShiftReduceParser.collapsePunctuation(parent.getChildren(),
        this.punctSet)[0] == child;
  }

  public static Parse[] reduceChunks(final Parse[] chunks, int ci,
      final Parse parent) {
    final String type = parent.getType();
    // perform reduce
    int reduceStart = ci;
    final int reduceEnd = ci;
    while (reduceStart >= 0 && chunks[reduceStart].getParent() == parent) {
      reduceStart--;
    }
    reduceStart++;
    Parse[] reducedChunks;
    if (!type.equals(ShiftReduceParser.TOP_NODE)) {
      reducedChunks = new Parse[chunks.length - (reduceEnd - reduceStart + 1)
          + 1]; // total - num_removed + 1 (for new node)
      // insert nodes before reduction
      for (int ri = 0, rn = reduceStart; ri < rn; ri++) {
        reducedChunks[ri] = chunks[ri];
      }
      // insert reduced node
      reducedChunks[reduceStart] = parent;
      // propagate punctuation sets
      parent
          .setPrevPunctuation(chunks[reduceStart].getPreviousPunctuationSet());
      parent.setNextPunctuation(chunks[reduceEnd].getNextPunctuationSet());
      // insert nodes after reduction
      int ri = reduceStart + 1;
      for (int rci = reduceEnd + 1; rci < chunks.length; rci++) {
        reducedChunks[ri] = chunks[rci];
        ri++;
      }
      ci = reduceStart - 1; // ci will be incremented at end of loop
    } else {
      reducedChunks = new Parse[0];
    }
    return reducedChunks;
  }

  /**
   * Returns true if the specified child is the last child of the specified
   * parent.
   * 
   * @param child
   *          The child parse.
   * @param parent
   *          The parent parse.
   * @return true if the specified child is the last child of the specified
   *         parent; false otherwise.
   */
  protected boolean lastChild(final Parse child, final Parse parent) {
    final Parse[] kids = ShiftReduceParser
        .collapsePunctuation(parent.getChildren(), this.punctSet);
    return kids[kids.length - 1] == child;
  }

}
