package host.anzo.anticheat.server.api;

import host.anzo.anticheat.config.AntiCheatClientConfig;
import host.anzo.anticheat.proxy.ProxyLibrary;
import host.anzo.anticheat.server.api.enums.EAntiClientRegisterResult;
import host.anzo.anticheat.server.api.interfaces.IAntiCheatClientAPI;
import host.anzo.anticheat.server.api.interfaces.IAntiCheatServerAPI;
import host.anzo.anticheat.server.api.model.AntiCheatClientInfo;
import host.anzo.anticheat.server.api.model.AntiCheatTokenInfo;
import host.anzo.anticheat.utils.IpUtils;
import host.anzo.simon.ClosedListener;
import host.anzo.simon.Lookup;
import host.anzo.simon.Simon;
import host.anzo.simon.exceptions.EstablishConnectionFailed;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

/**
 * @author ANZO
 * @since 9/17/2024
 */
@Slf4j
public abstract class AAntiCheatClientAPIConnection {
	private Lookup nameLookup;

	private IAntiCheatServerAPI serverAPI;

	private transient ScheduledFuture<?> reconnectTask;
	private transient ScheduledFuture<?> updateTask;

	private final ApiDisconnectedListener disconnectedListener = new ApiDisconnectedListener();

	private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);

	public abstract long getPlayersOnline();
	public abstract IAntiCheatClientAPI getClientAPI();

	@SuppressWarnings("unused")
	protected void initialize() {
		if (!AntiCheatClientConfig.ENABLED) {
			return;
		}

		final int proxyStartResult = ProxyLibrary.instance.StartTunnelFor("rmi.anzo.host", "127.0.0.1:" + AntiCheatClientConfig.RMI_PROXY_PORT);
		if (proxyStartResult != 0) {
			throw new RuntimeException("Proxy startup failed. Code: " + proxyStartResult);
		}

		connectToAntiCheatServer();
	}

	private void connectToAntiCheatServer() {
		try {
			if (nameLookup == null) {
				nameLookup = Simon.createNameLookup("127.0.0.1", AntiCheatClientConfig.RMI_PROXY_PORT);
			}

			serverAPI = (IAntiCheatServerAPI) nameLookup.lookup("master_server");
			nameLookup.addClosedListener(serverAPI, disconnectedListener);

			final EAntiClientRegisterResult registerResult = serverAPI.registerClient(getClientAPI(), getClientInfo());
			if (registerResult == EAntiClientRegisterResult.SUCCESS) {
				log.info("Anti-Cheat client registered on master API successfully.");
			}
			else {
				log.warn("Anti-Cheat client failed to connect to master API. Reason: {}", registerResult.toString());
			}
		}
		catch (EstablishConnectionFailed e) {
			log.warn("Master anti-cheat server isn't available. Make sure it's up and running.");
		}
		catch (Exception e) {
			log.error("Error while connecting to Login API.", e);
		}
		finally {
			if (updateTask == null) {
				updateTask = scheduler.scheduleAtFixedRate(this::updateClientInfo, 10, 10, TimeUnit.SECONDS);
			}

			reconnectTask = null;
			if (serverAPI == null) {
				onConnectionLost();
			}
		}
	}

	public void updateClientInfo() {
		if (reconnectTask != null || serverAPI == null) {
			return;
		}

		final AntiCheatClientInfo clientInfo = getClientInfo();
		try {
			serverAPI.updateClient(getClientAPI(), clientInfo);
		}
		catch (Exception e) {
			log.error("Error while updateClientInfo for client [{}]", clientInfo, e);
		}
	}

	private @NotNull AntiCheatClientInfo getClientInfo() {
		final AntiCheatClientInfo clientInfo = new AntiCheatClientInfo(AntiCheatClientConfig.CLIENT_ID, AntiCheatClientConfig.CLIENT_NAME, IpUtils.getExternalIP(), AntiCheatClientConfig.CLIENT_PASSWORD);
		clientInfo.setPlayersOnline(getPlayersOnline());
		return clientInfo;
	}

	/**
	 * @param token token to validate
	 * @return possible information about specified token
	 */
	public AntiCheatTokenInfo validateToken(String token) {
		return getClientAPI().validateToken(token);
	}

	/**
	 * Event happens when connection with login API lost
	 */
	void onConnectionLost() {
		if (reconnectTask != null) {
			return;
		}

		log.info("Connection with anti-cheat master server API lost.");
		if (serverAPI != null) {
			nameLookup.release(serverAPI);
			serverAPI = null;
		}

		reconnectTask = scheduler.schedule(() -> {
			log.info("Reconnecting to anti-cheat master server API...");
			connectToAntiCheatServer();
		}, 2, TimeUnit.SECONDS);
	}

	private class ApiDisconnectedListener implements ClosedListener {
		@Override
		public void closed() {
			onConnectionLost();
		}
	}
}