package host.anzo.anticheat.server.api;

import host.anzo.anticheat.commons.RunnableWrapper;
import host.anzo.anticheat.config.AntiCheatServerConfig;
import host.anzo.anticheat.server.api.interfaces.IAntiCheatNetworkClient;
import host.anzo.commons.interfaces.startup.IShutdownable;
import host.anzo.eossdk.eos.sdk.EOS;
import host.anzo.eossdk.eos.sdk.EOS_AntiCheatServer_Interface;
import host.anzo.eossdk.eos.sdk.EOS_Logging_Interface;
import host.anzo.eossdk.eos.sdk.EOS_Platform_Interface;
import host.anzo.eossdk.eos.sdk.anticheat.common.callbackresults.EOS_AntiCheatCommon_OnClientActionRequiredCallbackInfo;
import host.anzo.eossdk.eos.sdk.anticheat.common.callbackresults.EOS_AntiCheatCommon_OnClientAuthStatusChangedCallbackInfo;
import host.anzo.eossdk.eos.sdk.anticheat.common.callbackresults.EOS_AntiCheatCommon_OnMessageToClientCallbackInfo;
import host.anzo.eossdk.eos.sdk.anticheat.common.enums.EOS_EAntiCheatCommonClientAction;
import host.anzo.eossdk.eos.sdk.anticheat.common.enums.EOS_EAntiCheatCommonClientFlags;
import host.anzo.eossdk.eos.sdk.anticheat.common.enums.EOS_EAntiCheatCommonClientPlatform;
import host.anzo.eossdk.eos.sdk.anticheat.common.enums.EOS_EAntiCheatCommonClientType;
import host.anzo.eossdk.eos.sdk.anticheat.common.options.EOS_AntiCheatCommon_SetClientDetailsOptions;
import host.anzo.eossdk.eos.sdk.anticheat.server.options.EOS_AntiCheatServer_BeginSessionOptions;
import host.anzo.eossdk.eos.sdk.anticheat.server.options.EOS_AntiCheatServer_RegisterClientOptions;
import host.anzo.eossdk.eos.sdk.common.EOS_Bool;
import host.anzo.eossdk.eos.sdk.common.EOS_NotificationId;
import host.anzo.eossdk.eos.sdk.common.EOS_ProductUserId;
import host.anzo.eossdk.eos.sdk.common.enums.EOS_EResult;
import host.anzo.eossdk.eos.sdk.init.options.EOS_InitializeOptions;
import host.anzo.eossdk.eos.sdk.logging.EOS_LogMessage;
import host.anzo.eossdk.eos.sdk.logging.enums.EOS_ELogCategory;
import host.anzo.eossdk.eos.sdk.logging.enums.EOS_ELogLevel;
import host.anzo.eossdk.eos.sdk.platform.EOS_Platform_ClientCredentials;
import host.anzo.eossdk.eos.sdk.platform.enums.EOS_Platform_Create_Flag;
import host.anzo.eossdk.eos.sdk.platform.options.EOS_Platform_Options;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author ANZO
 * @since 9/20/2023
 */
@Slf4j(topic = "ANTICHEAT")
public abstract class AEpicOnlineService implements IShutdownable {
	private final static ScheduledExecutorService taskExecutor = Executors.newScheduledThreadPool(1);

	// Interfaces
	private EOS_Platform_Interface platform;
	private EOS_AntiCheatServer_Interface antiCheatServer;

	// Notifications
	private EOS_NotificationId messageToClientNotificationId;
	private EOS_NotificationId clientActionRequiredNotificationId;
	private EOS_NotificationId clientAuthStatusChangedNotificationId;

	@SuppressWarnings("unused")
	protected void initialize() {
		if (!AntiCheatServerConfig.ENABLE) {
			log.info("Connection to the Epic Online Services disabled by configuration");
			return;
		}

		callApi(() -> {
			final EOS_InitializeOptions initializeOptions = new EOS_InitializeOptions();
			initializeOptions.ProductName = AntiCheatServerConfig.PRODUCT_NAME;
			initializeOptions.ProductVersion = AntiCheatServerConfig.PRODUCT_VERSION;
			final EOS_EResult initResult = EOS.EOS_Initialize(initializeOptions);
			if (!initResult.isSuccess()) {
				throw new RuntimeException("Failed to EOS_Initialize with message: " + initResult);
			}

			final EOS_EResult setLogCallbackResult = EOS_Logging_Interface.setCallback(this::onLogMessage);
			if (!setLogCallbackResult.isSuccess()) {
				throw new RuntimeException("Failed to EOS_Logging_SetCallback with message: " + setLogCallbackResult);
			}

			final EOS_EResult setLogLevelResult = EOS_Logging_Interface.setLogLevel(EOS_ELogCategory.EOS_LC_ALL_CATEGORIES, AntiCheatServerConfig.DEBUG ? EOS_ELogLevel.EOS_LOG_VeryVerbose : EOS_ELogLevel.EOS_LOG_Info);
			if (!setLogLevelResult.isSuccess()) {
				throw new RuntimeException("Failed to EOS_Logging_SetLogLevel with message: " + setLogLevelResult);
			}

			final EOS_Platform_Options platformOptions = new EOS_Platform_Options();
			platformOptions.ProductId = AntiCheatServerConfig.PRODUCT_ID;
			platformOptions.SandboxId = AntiCheatServerConfig.SANDBOX_ID;
			platformOptions.DeploymentId = AntiCheatServerConfig.DEPLOYMENT_ID;
			platformOptions.ClientCredentials = new EOS_Platform_ClientCredentials(AntiCheatServerConfig.CLIENT_ID, AntiCheatServerConfig.CLIENT_SECRET);
			platformOptions.IsServer = EOS_Bool.EOS_TRUE;
			platformOptions.EncryptionKey = "1111111111111111111111111111111111111111111111111111111111111111";
			platformOptions.Flags = EOS_Platform_Create_Flag.of(EOS_Platform_Create_Flag.EOS_PF_DISABLE_OVERLAY);
			platformOptions.CacheDirectory = System.getProperty("java.io.tmpdir");

			platform = EOS.EOS_Platform_Create(platformOptions);
			if (platform == null) {
				throw new RuntimeException("Failed to EOS_Platform_Create: pointer is null");
			}

			antiCheatServer = platform.getAntiCheatServerInterface();
			if (antiCheatServer == null) {
				throw new RuntimeException("Failed to getAntiCheatServerInterface");
			}

			messageToClientNotificationId = antiCheatServer.addNotifyMessageToClient(null, this::onMessageToClient);
			if (!messageToClientNotificationId.isValid()) {
				throw new RuntimeException("Failed to addNotifyMessageToClient");
			}

			clientActionRequiredNotificationId = antiCheatServer.addNotifyClientActionRequired(null, this::onClientActionRequired);
			if (!clientActionRequiredNotificationId.isValid()) {
				throw new RuntimeException("Failed to addNotifyClientActionRequired");
			}

			clientAuthStatusChangedNotificationId = antiCheatServer.addNotifyClientAuthStatusChanged(null, this::onClientAuthStatusChanged);
			if (!clientAuthStatusChangedNotificationId.isValid()) {
				throw new RuntimeException("Failed to addNotifyClientAuthStatusChanged");
			}

			final EOS_AntiCheatServer_BeginSessionOptions beginSessionOptions = new EOS_AntiCheatServer_BeginSessionOptions();
			beginSessionOptions.RegisterTimeoutSeconds = AntiCheatServerConfig.ANTI_CHEAT_CLIENT_REGISTER_TIMEOUT;
			beginSessionOptions.ServerName = AntiCheatServerConfig.ANTI_CHEAT_SERVER_NAME;
			beginSessionOptions.EnableGameplayData = EOS_Bool.of(AntiCheatServerConfig.ANTI_CHEAT_ENABLE_GAMEPLAY_DATA);
			beginSessionOptions.LocalUserId = null;

			final EOS_EResult beginSessionResult = antiCheatServer.beginSession(beginSessionOptions);
			if (!beginSessionResult.isSuccess()) {
				throw new RuntimeException("Failed to anti-cheat beginSession: " + beginSessionResult);
			}
		});

		taskExecutor.scheduleAtFixedRate(() -> {
			if (platform != null) {
				platform.tick();
			}
		}, 0, AntiCheatServerConfig.PLATFORM_TICK_PERIOD, TimeUnit.MILLISECONDS);
	}

	/**
	 * @param connectionId network client connection ID
	 * @return network client by specified connection ID
	 */
	protected abstract IAntiCheatNetworkClient getClient(int connectionId);


	/**
	 * Event must be called when a new client connected to server
	 * @param client game client
	 */
	public void onClientConnected(IAntiCheatNetworkClient client) {
		callApi(() -> {
			final var registerClientOptions = new EOS_AntiCheatServer_RegisterClientOptions();
			registerClientOptions.ClientHandle = client.getEOSHandle();
			registerClientOptions.UserId = EOS_ProductUserId.fromString(client.getEosLocalUserID());
			registerClientOptions.IpAddress = client.getIpAddress();
			registerClientOptions.ClientType = EOS_EAntiCheatCommonClientType.EOS_ACCCT_ProtectedClient;
			registerClientOptions.ClientPlatform = EOS_EAntiCheatCommonClientPlatform.EOS_ACCCP_Windows;

			final var registerClientResult = antiCheatServer.registerClient(registerClientOptions);
			if (!registerClientResult.isSuccess()) {
				log.error("RegisterClient failed for clientId=[{}] with error=[{}]", client.getConnectionId(), registerClientResult);
				client.sendMessageToClient("Failed to register client");
				return;
			}

			var setClientDetailsOptions = new EOS_AntiCheatCommon_SetClientDetailsOptions();
			setClientDetailsOptions.ClientHandle = client.getEOSHandle();
			setClientDetailsOptions.ClientFlags = client.isAdmin() ? EOS_EAntiCheatCommonClientFlags.EOS_ACCCF_Admin : EOS_EAntiCheatCommonClientFlags.EOS_ACCCF_None;

			var setClientDetailsResult = antiCheatServer.setClientDetails(setClientDetailsOptions);
			if (!setClientDetailsResult.isSuccess()) {
				log.error("SetClientDetails failed for clientId=[{}] with error=[{}]", client.getConnectionId(), setClientDetailsResult);
				client.sendMessageToClient("Client details set failed");
			}
		});
	}

	/**
	 * Event must be called when need to send an EAC message from the client to server
	 * @param client game client
	 * @param data received bytes
	 */
	public void onClientMessage(IAntiCheatNetworkClient client, byte[] data) {
		callApi(() -> {
			final var receiveMessageFromClientResult = antiCheatServer.receiveMessageFromClient(client.getEOSHandle(), data);
			if (!receiveMessageFromClientResult.isSuccess()) {
				log.error("ReceiveMessageFromClient failed for clientId=[{}] with result=[{}]", client.getConnectionId(), receiveMessageFromClientResult);
				client.sendMessageToClient("Failed to receive security data");
			}
		});
	}

	/**
	 * Event must be called when a client disconnected from game server
	 * @param client WebSocket session
	 */
	public void onClientDisconnected(@NotNull IAntiCheatNetworkClient client) {
		log.info("Client id=[{}] disconnected", client.getConnectionId());
		callApi(() -> antiCheatServer.unregisterClient(client.getEOSHandle()));
	}

	/**
	 * Event happens when Anti-Cheat interface wants to apply specified action to the game client
	 * @param callbackInfo info about a client and action to apply
	 */
	private void onClientActionRequired(@NotNull EOS_AntiCheatCommon_OnClientActionRequiredCallbackInfo callbackInfo) {
		log.warn("onClientActionRequired: action={} reasonCode={}", callbackInfo.ClientAction, callbackInfo.ActionReasonCode);

		final IAntiCheatNetworkClient client = getClient((int)callbackInfo.ClientHandle.getValue());
		if (client == null) {
			log.warn("Received OnClientActionRequiredCallback for unknown client clientId=[{}]", callbackInfo.ClientHandle.getValue());
			return;
		}

		if (callbackInfo.ClientAction == EOS_EAntiCheatCommonClientAction.EOS_ACCCA_RemovePlayer) {
			log.warn("Kicking client with account=[{}] productUserId=[{}]: reasonCode=[{}] reasonDetails=[{}]",
					client.getAccountName(),
					client.getEosLocalUserID(),
					callbackInfo.ActionReasonCode,
					callbackInfo.ActionReasonDetailsString);
			client.sendMessageToClient(callbackInfo.ActionReasonDetailsString);
		}
	}

	/**
	 * Notify the Anti-Cheat client with message server want to send
	 * @param callbackInfo info about a client and data to send
	 */
	private void onMessageToClient(@NotNull EOS_AntiCheatCommon_OnMessageToClientCallbackInfo callbackInfo) {
		final IAntiCheatNetworkClient client = getClient((int)callbackInfo.ClientHandle.getValue());
		if (client != null) {
			client.sendEacMessageToClient(callbackInfo);
		}
	}

	/**
	 * Event happens when Anti-Cheat changed auth status for the connected client
	 * @param callbackInfo info about client and auth status change data
	 */
	private void onClientAuthStatusChanged(@NotNull EOS_AntiCheatCommon_OnClientAuthStatusChangedCallbackInfo callbackInfo) {
		log.info("onClientAuthStatusChanged: clientHandle={} authStatus={}", callbackInfo.ClientHandle.getValue(), callbackInfo.ClientAuthStatus);
		final IAntiCheatNetworkClient client = getClient((int)callbackInfo.ClientHandle.getValue());
		if (client != null) {
			var setClientNetworkStateResult = antiCheatServer.setClientNetworkState(client.getEOSHandle(), true);
			if (!setClientNetworkStateResult.isSuccess()) {
				log.error("SetClientNetworkState failed for clientId=[{}] result=[{}]", client.getConnectionId(), setClientNetworkStateResult);
				client.sendMessageToClient("Client set network state failed");
			}
		}
	}

	@Override
	public void onShutdown() {
		if (antiCheatServer != null) {
			if (messageToClientNotificationId.isValid()) {
				antiCheatServer.removeNotifyMessageToClient(messageToClientNotificationId);
			}
			if (clientActionRequiredNotificationId.isValid()) {
				antiCheatServer.removeNotifyClientActionRequired(clientActionRequiredNotificationId);
			}
			if (clientAuthStatusChangedNotificationId.isValid()) {
				antiCheatServer.removeNotifyClientAuthStatusChanged(clientAuthStatusChangedNotificationId);
			}
		}

		if (platform != null) {
			platform.release();
		}

		EOS.EOS_Shutdown();
	}

	/**
	 * Call EOS SDK methods in single thread
	 * @param methodToCall method to call
	 */
	private void callApi(Runnable methodToCall) {
		taskExecutor.execute(new RunnableWrapper(methodToCall));
	}

	/**
	 * Event happens when EOS logging engine sends the message
	 * @param logMessage object with message data
	 */
	private void onLogMessage(@NotNull EOS_LogMessage logMessage) {
		switch (logMessage.Level) {
			case EOS_LOG_Fatal, EOS_LOG_Error -> log.error(logMessage.Message);
			case EOS_LOG_Warning -> log.warn(logMessage.Message);
			case EOS_LOG_Info -> log.info(logMessage.Message);
			case EOS_LOG_Verbose, EOS_LOG_VeryVerbose -> log.debug(logMessage.Message);
		}
	}
}
