package host.anzo.anticheat.server.api;

import host.anzo.anticheat.config.AntiCheatClientConfig;
import host.anzo.anticheat.proxy.AntiCheatProxyLibrary;
import host.anzo.anticheat.server.api.enums.EAntiClientRegisterResult;
import host.anzo.anticheat.server.api.interfaces.rmi.IRMIAntiCheatClientAPI;
import host.anzo.anticheat.server.api.interfaces.rmi.IRMIAntiCheatServerAPI;
import host.anzo.anticheat.server.api.model.AntiCheatClientInfo;
import host.anzo.anticheat.server.api.model.AntiCheatTokenInfo;
import host.anzo.anticheat.commons.utils.IpUtils;
import host.anzo.simon.ClosedListener;
import host.anzo.simon.Lookup;
import host.anzo.simon.Simon;
import host.anzo.simon.exceptions.EstablishConnectionFailed;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

/**
 * @author ANZO
 * @since 9/17/2024
 */
@Slf4j
@SuppressWarnings("unused")
public abstract class AAntiCheatClientAPIConnection {
	private Lookup nameLookup;

	private IRMIAntiCheatServerAPI serverAPI;

	private transient ScheduledFuture<?> reconnectTask;
	private transient ScheduledFuture<?> updateTask;

	private final ApiDisconnectedListener disconnectedListener = new ApiDisconnectedListener();

	private final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);

	private int rmiLocalProxyPort = -1;

	public abstract long getPlayersOnline();
	public abstract IRMIAntiCheatClientAPI getClientAPI();

	@SuppressWarnings("unused")
	protected void initialize() {
		if (!AntiCheatClientConfig.RMI_ENABLED) {
			return;
		}

		rmiLocalProxyPort = IpUtils.getFreePort();
		if (rmiLocalProxyPort == -1) {
			throw new RuntimeException("Failed to find free port for RMI proxy");
		}

		final int proxyStartResult = AntiCheatProxyLibrary.instance.StartTunnelFor("rmi.anzo.host", "127.0.0.1:" + rmiLocalProxyPort);
		if (proxyStartResult != 0) {
			throw new RuntimeException("RMI proxy startup failed. Code: " + proxyStartResult);
		}

		connectToAntiCheatServer();
	}

	/**
	 * Connect to Anti-Cheat server API
	 */
	private void connectToAntiCheatServer() {
		try {
			if (nameLookup == null) {
				nameLookup = Simon.createNameLookup("127.0.0.1", rmiLocalProxyPort);
			}

			serverAPI = (IRMIAntiCheatServerAPI) nameLookup.lookup("master_server");
			nameLookup.addClosedListener(serverAPI, disconnectedListener);

			final EAntiClientRegisterResult registerResult = serverAPI.registerClient(getClientAPI(), getClientInfo());
			if (registerResult == EAntiClientRegisterResult.SUCCESS) {
				onConnected();
				log.info("Anti-cheat client registered on master API successfully.");
			}
			else {
				nameLookup.release(serverAPI);
				log.warn("Anti-cheat client failed to connect to master API. Reason: {}", registerResult.toString());
			}
		}
		catch (EstablishConnectionFailed e) {
			log.warn("Master anti-cheat server isn't available. Make sure it's up and running.");
		}
		catch (Exception e) {
			log.error("Error while connecting to Login API.", e);
		}
	}

	public void updateClientInfo() {
		if (reconnectTask != null || serverAPI == null) {
			return;
		}

		final AntiCheatClientInfo clientInfo = getClientInfo();
		try {
			serverAPI.updateClient(getClientAPI(), clientInfo);
		}
		catch (Exception e) {
			log.error("Error while updateClientInfo for client [{}]", clientInfo, e);
		}
	}

	private @NotNull AntiCheatClientInfo getClientInfo() {
		final AntiCheatClientInfo clientInfo = new AntiCheatClientInfo(AntiCheatClientConfig.RMI_CLIENT_ID, AntiCheatClientConfig.EOS_ANTI_CHEAT_SERVER_NAME, IpUtils.getExternalIP(), AntiCheatClientConfig.RMI_CLIENT_PASSWORD);
		clientInfo.setPlayersOnline(getPlayersOnline());
		return clientInfo;
	}

	/**
	 * @param token token to validate
	 * @return possible information about specified token
	 */
	public AntiCheatTokenInfo validateToken(String token) {
		return getClientAPI().validateToken(token);
	}

	/**
	 * Event happens when connection with Anti-Cheat server API established successfully
	 */
	void onConnected() {
		if (updateTask != null) {
			updateTask.cancel(true);
			updateTask = null;
		}
		updateTask = scheduler.scheduleAtFixedRate(this::updateClientInfo, 1, 10, TimeUnit.SECONDS);
	}

	private class ApiDisconnectedListener implements ClosedListener {
		@Override
		public void closed() {
			log.info("Connection with anti-cheat master server API lost.");
			serverAPI = null;

			try {
				if (updateTask != null) {
					updateTask.cancel(true);
					updateTask = null;
				}
				if (reconnectTask != null) {
					reconnectTask.cancel(true);
					reconnectTask = null;
				}
			}
			catch (Exception ignored) {
			}

			reconnectTask = scheduler.schedule(() -> {
				log.info("Reconnecting to anti-cheat master server API...");
				connectToAntiCheatServer();
			}, 2, TimeUnit.SECONDS);
		}
	}
}