package host.anzo.commons.annotations.processors;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * The `ExtendedEnum` annotation is used to mark an enum class for enhanced functionality.
 * When applied, this annotation instructs the annotation processor to generate additional methods
 * that provide efficient ways to retrieve enum constants based on specified fields.
 * <pre>{@code
 * @ExtendedEnum(cacheKeys = {"name", "code"})
 * public enum MyEnum {
 *     VALUE_ONE("Name1", "Code1"),
 *     VALUE_TWO("Name2", "Code2");
 *
 *     private final String name;
 *     private final String code;
 *
 *     MyEnum(String name, String code) {
 *         this.name = name;
 *         this.code = code;
 *     }
 * }
 * }</pre>
 * @author ANZO
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.SOURCE)
public @interface ExtendedEnum {
	/**
	 * An array of field names by which the enum constants should be cached and retrieved.
	 * If no values are provided, a default cache map based on the enum's name will be generated.
	 *
	 * @return an array of field names to use for caching
	 */
	String[] cacheKeys() default {};
}