package host.anzo.commons.processors;

import com.sun.source.util.Trees;
import com.sun.tools.javac.processing.JavacProcessingEnvironment;
import com.sun.tools.javac.tree.JCTree;
import com.sun.tools.javac.tree.TreeMaker;
import com.sun.tools.javac.util.Names;
import org.jetbrains.annotations.NotNull;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.Messager;
import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.SourceVersion;
import javax.tools.Diagnostic;
import java.lang.reflect.Method;

/**
 * Abstract base class for annotation processors providing common utilities and initialization
 * logic, especially for accessing Javac-specific APIs like {@link TreeMaker}, {@link Trees},
 * and {@link Names}. It also includes handling for potential JetBrains IDE-specific wrappers
 * around the {@link ProcessingEnvironment}.
 *
 * @author ANZO
 */
public abstract class CommonProcessor extends AbstractProcessor {
	/**
	 * Factory for creating Javac AST nodes ({@link JCTree}). Initialized in {@link #init(ProcessingEnvironment)}.
	 */
	protected TreeMaker maker;
	/**
	 * Utility for working with Javac ASTs ({@link JCTree}). Initialized in {@link #init(ProcessingEnvironment)}.
	 * Provides access to source tree structures.
	 */
	protected Trees trees;
	/**
	 * Interface for reporting errors, warnings, and other notices during annotation processing.
	 * Obtained from the {@link ProcessingEnvironment}.
	 */
	protected Messager messager;
	/**
	 * Factory for creating Javac {@link com.sun.tools.javac.util.Name} objects (identifiers).
	 * Initialized in {@link #init(ProcessingEnvironment)}.
	 */
	protected Names names;

	/**
	 * Specifies the latest supported Java source version.
	 *
	 * @return The latest {@link SourceVersion}.
	 */
	@Override
	public SourceVersion getSupportedSourceVersion() {
		return SourceVersion.latest();
	}

	/**
	 * Initializes the processor, setting up common utilities like {@link Messager},
	 * {@link TreeMaker}, {@link Trees}, and {@link Names}.
	 * <p>
	 * This implementation attempts to unwrap the {@link ProcessingEnvironment} if it's potentially
	 * wrapped by JetBrains tools (like in IntelliJ IDEA) to ensure access to the underlying
	 * {@link JavacProcessingEnvironment}. If the unwrapping fails or the environment is not
	 * a {@code JavacProcessingEnvironment}, an error is logged, and the Javac-specific tools
	 * ({@code maker}, {@code trees}, {@code names}) will remain null.
	 *
	 * @param procEnv Environment providing access to facilities like {@link Messager},
	 *                {@link javax.annotation.processing.Filer}, and {@link javax.lang.model.util.Elements}.
	 * @see #tryUnwrapJetBrainsProcEnv(Object)
	 */
	@Override
	public synchronized void init(ProcessingEnvironment procEnv) {
		super.init(procEnv);
		this.messager = procEnv.getMessager();
		final ProcessingEnvironment unwrappedProcEnv = tryUnwrapJetBrainsProcEnv(procEnv);
		if (unwrappedProcEnv instanceof JavacProcessingEnvironment javacEnv) {
			this.maker = TreeMaker.instance(javacEnv.getContext());
			this.trees = Trees.instance(javacEnv);
			this.names = Names.instance(javacEnv.getContext());
			log("[ExtendedEnumProcessor] Successfully initialized using JavacProcessingEnvironment.");
		}
		else {
			logError("Could not obtain JavacProcessingEnvironment. Annotation processing might not work correctly. Original env: " + procEnv.getClass().getName());
		}
	}

	/**
	 * Checks if a field with the specified name already exists within the given class declaration.
	 *
	 * @param classDecl The Javac AST node representing the class declaration.
	 * @param fieldName The name of the field to check for.
	 * @return {@code true} if a field with the given name exists, {@code false} otherwise.
	 */
	protected boolean fieldExists(JCTree.@NotNull JCClassDecl classDecl, String fieldName) {
		return classDecl.defs.stream()
				.anyMatch(def -> def instanceof JCTree.JCVariableDecl
						&& ((JCTree.JCVariableDecl) def).getName().contentEquals(fieldName));
	}

	/**
	 * Checks if a method with the specified name already exists within the given class declaration.
	 * Note: This performs a simple name check and does not consider method parameters (overloads).
	 *
	 * @param classDecl  The Javac AST node representing the class declaration.
	 * @param methodName The name of the method to check for.
	 * @return {@code true} if a method with the given name exists, {@code false} otherwise.
	 */
	protected boolean methodExists(JCTree.@NotNull JCClassDecl classDecl, String methodName) {
		return classDecl.defs.stream()
				.anyMatch(def -> def instanceof JCTree.JCMethodDecl
						&& ((JCTree.JCMethodDecl) def).getName().contentEquals(methodName));
	}

	/**
	 * Creates a Javac AST expression ({@link JCTree.JCExpression}) representing a fully qualified name
	 * (e.g., {@code java.util.List}).
	 *
	 * @param fullName The fully qualified name as a String (e.g., "java.lang.String").
	 * @return A {@link JCTree.JCExpression} representing the qualified name, suitable for use in AST generation.
	 *         This will typically be a {@link JCTree.JCIdent} for single-part names or a
	 *         {@link JCTree.JCFieldAccess} (Select) for multi-part names.
	 */
	protected JCTree.JCExpression createQualifiedName(@NotNull String fullName) {
		String[] parts = fullName.split("\\.");
		JCTree.JCExpression expr = maker.Ident(names.fromString(parts[0]));
		for (int i = 1; i < parts.length; i++) {
			expr = maker.Select(expr, names.fromString(parts[i]));
		}
		return expr;
	}

	/**
	 * Attempts to unwrap a {@link ProcessingEnvironment} instance that might be wrapped by
	 * JetBrains' build tools (e.g., within IntelliJ IDEA's build process).
	 * <p>
	 * This uses reflection to call the static {@code org.jetbrains.jps.javac.APIWrappers.unwrap} method.
	 * If the unwrapping fails (e.g., the class or method doesn't exist, or an exception occurs),
	 * the original wrapper object is returned.
	 *
	 * @param wrapper The potentially wrapped {@link ProcessingEnvironment} instance.
	 * @param <T>     The type of the object being unwrapped (expected to be {@code ProcessingEnvironment}).
	 * @return The unwrapped {@link ProcessingEnvironment} if successful, or the original {@code wrapper} otherwise.
	 */
	private static <T> T tryUnwrapJetBrainsProcEnv(T wrapper) {
		T unwrapped = null;
		try {
			// Use reflection to find and invoke the JetBrains unwrapper method
			final Class<?> apiWrappers = wrapper.getClass().getClassLoader().loadClass("org.jetbrains.jps.javac.APIWrappers");
			final Method unwrapMethod = apiWrappers.getDeclaredMethod("unwrap", Class.class, Object.class);
			// Cast the result back to the expected type (ProcessingEnvironment)
			unwrapped = ((Class<? extends T>) ProcessingEnvironment.class).cast(unwrapMethod.invoke(null, ProcessingEnvironment.class, wrapper));
		} catch (Throwable ignored) {
			// Ignore exceptions (ClassNotFound, NoSuchMethod, InvocationTargetException, etc.)
			// If unwrapping fails, we'll proceed with the original environment.
		}
		// Return the unwrapped object if successful, otherwise return the original wrapper
		return unwrapped != null ? unwrapped : wrapper;
	}

	/**
	 * Logs an informational message using the processing environment's {@link Messager}.
	 * The message is prefixed with the simple name of this processor class.
	 *
	 * @param msg The message to log.
	 */
	protected void log(String msg) {
		if (messager != null) {
			messager.printMessage(Diagnostic.Kind.NOTE, "[" + getClass().getSimpleName() + "] " + msg);
		} else {
			System.out.println("NOTE: [" + getClass().getSimpleName() + "] " + msg + " (Messager not initialized)");
		}
	}

	/**
	 * Logs an error message using the processing environment's {@link Messager}.
	 * The message is prefixed with the simple name of this processor class.
	 * Note that reporting an error typically halts the compilation process.
	 *
	 * @param msg The error message to log.
	 */
	protected void logError(String msg) {
		if (messager != null) {
			messager.printMessage(Diagnostic.Kind.ERROR, "[" + getClass().getSimpleName() + "] " + msg);
		} else {
			System.err.println("ERROR: [" + getClass().getSimpleName() + "] " + msg + " (Messager not initialized)");
		}
	}

	/**
	 * Logs a warning message using the processing environment's {@link Messager}.
	 * The message is prefixed with the simple name of this processor class.
	 *
	 * @param msg The warning message to log.
	 */
	protected void logWarn(String msg) {
		if (messager != null) {
			messager.printMessage(Diagnostic.Kind.WARNING, "[" + getClass().getSimpleName() + "] " + msg);
		} else {
			System.out.println("WARN: [" + getClass().getSimpleName() + "] " + msg + " (Messager not initialized)");
		}
	}
}