package host.anzo.commons.processors;

import com.google.auto.service.AutoService;
import com.sun.source.util.TreePath;
import com.sun.tools.javac.code.Flags;
import com.sun.tools.javac.code.TypeTag;
import com.sun.tools.javac.tree.JCTree.JCClassDecl;
import com.sun.tools.javac.tree.JCTree.JCExpression;
import com.sun.tools.javac.tree.JCTree.JCMethodDecl;
import com.sun.tools.javac.tree.JCTree.JCVariableDecl;
import com.sun.tools.javac.util.List;

import javax.annotation.processing.Processor;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.TypeElement;
import java.util.Set;

/**
 * An annotation processor that enhances enums annotated with {@code @ExtendedEnum}.
 * <p>
 * This processor automatically adds the following members to the annotated enum:
 * <ul>
 *     <li>A {@code private static final List<EnumType> cacheList} field, initialized with {@code List.of(values())}.</li>
 *     <li>A {@code public static EnumType getValue(int ordinal)} method to retrieve an enum constant by its ordinal using the cache.</li>
 *     <li>A {@code public static List<EnumType> getValues()} method that returns the cached list of enum constants.</li>
 * </ul>
 * This avoids repeated array allocation from the implicit {@code values()} method and provides convenient access methods.
 * It relies on the Javac AST manipulation utilities provided by {@link CommonProcessor}.
 *
 * @author ANZO
 * @see CommonProcessor
 * @see Statics#EXTENDED_ENUM_ANNOTATION_PATH
 */
@AutoService(Processor.class)
@SupportedAnnotationTypes(Statics.EXTENDED_ENUM_ANNOTATION_PATH)
public class ExtendedEnumProcessor extends CommonProcessor {

	/**
	 * Processes annotations for a processing round.
	 * <p>
	 * It finds all enums annotated with {@code @ExtendedEnum}, verifies processor initialization,
	 * and then modifies the AST of each found enum to add the caching field and utility methods.
	 *
	 * @param annotations The annotation types requested to be processed.
	 * @param roundEnv    Environment for information about the current and prior round.
	 * @return {@code false} always, as this processor does not claim the annotations.
	 *         This allows other processors to potentially process the same annotations, though typically
	 *         only one processor modifies a given element's structure.
	 */
	@Override
	public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
		if (this.maker == null || this.trees == null || this.names == null) {
			logError("Processor essentials (maker, trees, names) not initialized. Skipping processing. Check init logs.");
			return false;
		}

		if (roundEnv.processingOver() || annotations.isEmpty()) {
			return false;
		}

		final TypeElement extendedEnumAnnotation = processingEnv.getElementUtils().getTypeElement(Statics.EXTENDED_ENUM_ANNOTATION_PATH);
		if (extendedEnumAnnotation == null) {
			logError("Could not resolve " + Statics.EXTENDED_ENUM_ANNOTATION_PATH + " annotation class. Is it on the classpath?");
			return false;
		}

		for (Element element : roundEnv.getElementsAnnotatedWith(extendedEnumAnnotation)) {
			if (element.getKind() == ElementKind.ENUM) {
				final TreePath path = trees.getPath(element);
				if (path == null) {
					logWarn("Could not get TreePath for element: " + element.getSimpleName());
					continue;
				}
				final JCClassDecl classDecl = (JCClassDecl) path.getLeaf();

				addCacheListField(classDecl);
				addGetValueMethod(classDecl);
				addGetValuesMethod(classDecl);
			} else {
				logWarn("@" + extendedEnumAnnotation.getSimpleName() + " should only be applied to enums, but found on: " + element.getSimpleName() + " (" + element.getKind() + ")");
			}
		}

		return false;
	}

	/**
	 * Adds a {@code private static final List<E> cacheList} field to the enum's AST.
	 * The field is initialized with {@code java.util.List.of(values())}.
	 * If a field named "cacheList" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 */
	private void addCacheListField(JCClassDecl classDecl) {
		final String fieldName = "cacheList";
		if (fieldExists(classDecl, fieldName))
			return;

		final JCExpression listType = maker.TypeApply(
				createQualifiedName("java.util.List"),
				List.of(maker.Ident(classDecl.name))
		);

		final JCExpression valuesCall = maker.Apply(
				List.nil(),
				maker.Select(
						maker.Ident(classDecl.name),
						names.fromString("values")),
				List.nil()
		);

		final JCExpression init = maker.Apply(
				List.nil(),
				maker.Select(
						createQualifiedName("java.util.List"),
						names.fromString("of")),
				List.of(valuesCall)
		);

		final JCVariableDecl cacheListField = maker.VarDef(
				maker.Modifiers(Flags.PRIVATE | Flags.STATIC | Flags.FINAL),
				names.fromString(fieldName),
				listType,
				init
		);

		classDecl.defs = classDecl.defs.append(cacheListField);
	}

	/**
	 * Adds a {@code public static E getValue(int ordinal)} method to the enum's AST.
	 * The method body returns {@code cacheList.get(ordinal)}.
	 * If a method named "getValue" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 */
	private void addGetValueMethod(JCClassDecl classDecl) {
		final String methodName = "getValue";
		if (methodExists(classDecl, methodName))
			return;

		final JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				maker.Ident(classDecl.name),
				List.nil(),
				List.of(maker.VarDef(
						maker.Modifiers(Flags.PARAMETER),
						names.fromString("ordinal"),
						maker.TypeIdent(TypeTag.INT),
						null
				)),
				List.nil(),
				maker.Block(0, List.of(
						maker.Return(
								maker.Apply(
										List.nil(),
										maker.Select(
												maker.Ident(names.fromString("cacheList")),
												names.fromString("get")),
										List.of(maker.Ident(names.fromString("ordinal")))
								)
						)
				)),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds a {@code public static List<E> getValues()} method to the enum's AST.
	 * The method body returns the {@code cacheList} field.
	 * If a method named "getValues" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 */
	private void addGetValuesMethod(JCClassDecl classDecl) {
		final String methodName = "getValues";
		if (methodExists(classDecl, methodName))
			return;

		final JCExpression returnType = maker.TypeApply(
				createQualifiedName("java.util.List"),
				List.of(maker.Ident(classDecl.name))
		);

		final JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				returnType,
				List.nil(),
				List.nil(),
				List.nil(),
				maker.Block(0, List.of(
						maker.Return(maker.Ident(names.fromString("cacheList")))
				)),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}
}