package host.anzo.commons.processors;

import com.google.auto.service.AutoService;
import com.sun.source.util.TreePath;
import com.sun.tools.javac.code.Flags;
import com.sun.tools.javac.code.TypeTag;
import com.sun.tools.javac.tree.JCTree.*;
import com.sun.tools.javac.util.List;

import javax.annotation.processing.Processor;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.TypeElement;
import java.util.Set;

/**
 * An annotation processor that enhances enums annotated with {@code @ExtendedEnum}.
 * <p>
 * This processor automatically adds the following members to the annotated enum:
 * <ul>
 *     <li>A {@code private static final List<EnumType> cacheList} field, initialized with {@code List.of(values())}</li>
 *     <li>A {@code public static EnumType getValue(int ordinal)} method to retrieve an enum constant by its ordinal</li>
 *     <li>A {@code public static List<EnumType> getValues()} method that returns the cached list</li>
 *     <li>A {@code private static final Map<String, EnumType> cacheMap} field with case-insensitive name mapping</li>
 *     <li>Overloaded {@code public static EnumType getValueOf(String)} methods for name lookup</li>
 * </ul>
 * This avoids repeated array allocation and provides convenient access methods.
 * It relies on the Javac AST manipulation utilities provided by {@link CommonProcessor}.
 *
 * @author ANZO
 * @see CommonProcessor
 * @see Statics#EXTENDED_ENUM_ANNOTATION_PATH
 */
@AutoService(Processor.class)
@SupportedAnnotationTypes(Statics.EXTENDED_ENUM_ANNOTATION_PATH)
public class ExtendedEnumProcessor extends CommonProcessor {
	/**
	 * Processes annotations for a processing round.
	 * <p>
	 * It finds all enums annotated with {@code @ExtendedEnum}, verifies processor initialization,
	 * and then modifies the AST of each found enum to add the caching field and utility methods.
	 *
	 * @param annotations The annotation types requested to be processed.
	 * @param roundEnv    Environment for information about the current and prior round.
	 * @return {@code false} always, as this processor does not claim the annotations.
	 *         This allows other processors to potentially process the same annotations, though typically
	 *         only one processor modifies a given element's structure.
	 */
	@Override
	public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
		if (this.maker == null || this.trees == null || this.names == null) {
			logError("Processor essentials not initialized. Skipping processing.");
			return false;
		}

		if (roundEnv.processingOver() || annotations.isEmpty()) {
			return false;
		}

		final TypeElement extendedEnumAnnotation = processingEnv.getElementUtils().getTypeElement(Statics.EXTENDED_ENUM_ANNOTATION_PATH);
		if (extendedEnumAnnotation == null) {
			logError("Could not resolve " + Statics.EXTENDED_ENUM_ANNOTATION_PATH + " annotation.");
			return false;
		}

		for (Element element : roundEnv.getElementsAnnotatedWith(extendedEnumAnnotation)) {
			if (element.getKind() == ElementKind.ENUM) {
				final TreePath path = trees.getPath(element);
				if (path == null) {
					logWarn("Could not get TreePath for element: " + element.getSimpleName());
					continue;
				}
				final JCClassDecl classDecl = (JCClassDecl) path.getLeaf();

				addCacheListField(classDecl);
				addGetValueMethod(classDecl);
				addGetValuesMethod(classDecl);
				addCacheMapField(classDecl);
				addGetValueOfMethodWithTwoParams(classDecl);
				addGetValueOfMethodWithOneParam(classDecl);
			} else {
				logWarn("@" + extendedEnumAnnotation.getSimpleName() + " applied to non-enum: " + element.getSimpleName());
			}
		}

		return false;
	}

	/**
	 * Adds a {@code private static final List<E> cacheList} field to the enum's AST.
	 * The field is initialized with {@code java.util.List.of(values())}.
	 * If a field named "cacheList" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 */
	private void addCacheListField(JCClassDecl classDecl) {
		final String fieldName = "cacheList";
		if (fieldExists(classDecl, fieldName))
			return;

		final JCExpression listType = maker.TypeApply(
				createQualifiedName("java.util.List"),
				List.of(maker.Ident(classDecl.name))
		);

		final JCExpression valuesCall = maker.Apply(
				List.nil(),
				maker.Select(
						maker.Ident(classDecl.name),
						names.fromString("values")),
				List.nil()
		);

		final JCExpression init = maker.Apply(
				List.nil(),
				maker.Select(
						createQualifiedName("java.util.List"),
						names.fromString("of")),
				List.of(valuesCall)
		);

		final JCVariableDecl cacheListField = maker.VarDef(
				maker.Modifiers(Flags.PRIVATE | Flags.STATIC | Flags.FINAL),
				names.fromString(fieldName),
				listType,
				init
		);

		classDecl.defs = classDecl.defs.append(cacheListField);
	}

	/**
	 * Adds a {@code public static E getValue(int ordinal)} method to the enum's AST.
	 * The method body returns {@code cacheList.get(ordinal)}.
	 * If a method named "getValue" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 */
	private void addGetValueMethod(JCClassDecl classDecl) {
		final String methodName = "getValue";
		if (methodExists(classDecl, methodName))
			return;

		final JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				maker.Ident(classDecl.name),
				List.nil(),
				List.of(maker.VarDef(
						maker.Modifiers(Flags.PARAMETER),
						names.fromString("ordinal"),
						maker.TypeIdent(TypeTag.INT),
						null
				)),
				List.nil(),
				maker.Block(0, List.of(
						maker.Return(
								maker.Apply(
										List.nil(),
										maker.Select(
												maker.Ident(names.fromString("cacheList")),
												names.fromString("get")),
										List.of(maker.Ident(names.fromString("ordinal")))
								)
						)
				)),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds a {@code public static List<E> getValues()} method to the enum's AST.
	 * The method body returns the {@code cacheList} field.
	 * If a method named "getValues" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 */
	private void addGetValuesMethod(JCClassDecl classDecl) {
		final String methodName = "getValues";
		if (methodExists(classDecl, methodName))
			return;

		final JCExpression returnType = maker.TypeApply(
				createQualifiedName("java.util.List"),
				List.of(maker.Ident(classDecl.name))
		);

		final JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				returnType,
				List.nil(),
				List.nil(),
				List.nil(),
				maker.Block(0, List.of(
						maker.Return(maker.Ident(names.fromString("cacheList")))
				)),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds a static {@code Map<String, EnumType> cacheMap} field to the enum.
	 * The map is initialized with enum names in lowercase as keys.
	 * Uses {@code Collectors.toMap} with name.toLowerCase() mapping.
	 *
	 * @param classDecl The enum class declaration to modify
	 */
	private void addCacheMapField(JCClassDecl classDecl) {
		final String fieldName = "cacheMap";
		if (fieldExists(classDecl, fieldName)) return;

		JCExpression stringType = createQualifiedName("java.lang.String");
		JCExpression enumType = maker.Ident(classDecl.name);
		JCExpression mapType = maker.TypeApply(
				createQualifiedName("java.util.Map"),
				List.of(stringType, enumType)
		);

		JCExpression cacheListExpr = maker.Ident(names.fromString("cacheList"));
		JCExpression streamCall = maker.Select(cacheListExpr, names.fromString("stream"));
		JCExpression streamApply = maker.Apply(List.nil(), streamCall, List.nil());

		// Lambda e -> e.name().toLowerCase()
		JCVariableDecl paramE1 = maker.VarDef(maker.Modifiers(Flags.PARAMETER), names.fromString("e"), null, null);
		JCExpression eName = maker.Select(maker.Ident(names.fromString("e")), names.fromString("name"));
		JCExpression nameCall = maker.Apply(List.nil(), eName, List.nil());
		JCExpression toLowerCaseCall = maker.Apply(List.nil(), maker.Select(nameCall, names.fromString("toLowerCase")), List.nil());
		JCLambda lambda1 = maker.Lambda(List.of(paramE1), toLowerCaseCall);

		// Lambda e -> e
		JCVariableDecl paramE2 = maker.VarDef(maker.Modifiers(Flags.PARAMETER), names.fromString("e"), null, null);
		JCExpression eRef = maker.Ident(names.fromString("e"));
		JCLambda lambda2 = maker.Lambda(List.of(paramE2), eRef);

		// Collectors.toMap(...)
		JCExpression collectorsToMap = maker.Select(createQualifiedName("java.util.stream.Collectors"), names.fromString("toMap"));
		JCExpression toMapApply = maker.Apply(List.nil(), collectorsToMap, List.of(lambda1, lambda2));

		// collect(...)
		JCExpression collectCall = maker.Apply(
				List.nil(),
				maker.Select(streamApply, names.fromString("collect")),
				List.of(toMapApply)
		);

		JCVariableDecl cacheMapField = maker.VarDef(
				maker.Modifiers(Flags.PRIVATE | Flags.STATIC | Flags.FINAL),
				names.fromString(fieldName),
				mapType,
				collectCall
		);

		classDecl.defs = classDecl.defs.append(cacheMapField);
	}

	/**
	 * Adds the {@code getValueOf(String name, EnumType defaultValue)} method.
	 * The method performs case-insensitive lookup in cacheMap.
	 *
	 * @param classDecl The enum class declaration to modify
	 */
	private void addGetValueOfMethodWithTwoParams(JCClassDecl classDecl) {
		String methodName = "getValueOf";
		if (methodExists(classDecl, methodName, 2)) return;

		JCVariableDecl paramName = maker.VarDef(
				maker.Modifiers(Flags.PARAMETER),
				names.fromString("name"),
				createQualifiedName("java.lang.String"),
				null
		);
		JCVariableDecl paramDefault = maker.VarDef(
				maker.Modifiers(Flags.PARAMETER),
				names.fromString("defaultValue"),
				maker.Ident(classDecl.name),
				null
		);
		List<JCVariableDecl> params = List.of(paramName, paramDefault);

		JCExpression nameToLower = maker.Apply(
				List.nil(),
				maker.Select(maker.Ident(names.fromString("name")), names.fromString("toLowerCase")),
				List.nil()
		);
		JCExpression getOrDefault = maker.Apply(
				List.nil(),
				maker.Select(maker.Ident(names.fromString("cacheMap")), names.fromString("getOrDefault")),
				List.of(nameToLower, maker.Ident(names.fromString("defaultValue")))
		);

		JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				maker.Ident(classDecl.name),
				List.nil(),
				params,
				List.nil(),
				maker.Block(0, List.of(maker.Return(getOrDefault))),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds the overloaded {@code getValueOf(String name)} method.
	 * Delegates to the two-parameter version with null default value.
	 *
	 * @param classDecl The enum class declaration to modify
	 */
	private void addGetValueOfMethodWithOneParam(JCClassDecl classDecl) {
		String methodName = "getValueOf";
		if (methodExists(classDecl, methodName, 1)) return;

		JCVariableDecl paramName = maker.VarDef(
				maker.Modifiers(Flags.PARAMETER),
				names.fromString("name"),
				createQualifiedName("java.lang.String"),
				null
		);
		List<JCVariableDecl> params = List.of(paramName);

		JCExpression call = maker.Apply(
				List.nil(),
				maker.Ident(names.fromString(methodName)),
				List.of(
						maker.Ident(names.fromString("name")),
						maker.Literal(TypeTag.BOT, null)
				)
		);

		JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				maker.Ident(classDecl.name),
				List.nil(),
				params,
				List.nil(),
				maker.Block(0, List.of(maker.Return(call))),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}
}