package host.anzo.commons.processors;

import com.google.auto.service.AutoService;
import com.sun.source.util.TreePath;
import com.sun.tools.javac.code.Flags;
import com.sun.tools.javac.code.TypeTag;
import com.sun.tools.javac.tree.JCTree.*;
import com.sun.tools.javac.util.List;
import org.jetbrains.annotations.NotNull;

import javax.annotation.processing.Processor;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.lang.model.element.*;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * An annotation processor that enhances enums annotated with {@code @ExtendedEnum}.
 * <p>
 * This processor automatically adds the following members to the annotated enum:
 * <ul>
 *     <li>A {@code private static final java.util.List<EnumType> cacheList} field, initialized with {@code java.util.List.of(values())}</li>
 *     <li>A {@code public static EnumType getValue(int ordinal)} method to retrieve an enum constant by its ordinal</li>
 *     <li>A {@code public static java.util.List<EnumType> getValues()} method that returns the cached list</li>
 * </ul>
 * If the {@code cacheKeys} attribute of {@code @ExtendedEnum} is used:
 * <ul>
 *     <li>For each specified key (field name):
 *         <ul>
 *             <li>A {@code private static final java.util.Map<FieldType, EnumType> cacheMapBy<FieldName>} field.</li>
 *             <li>A static initializer block to populate this map.</li>
 *             <li>A {@code public static EnumType getValueOf<FieldName>(FieldType key)} method.</li>
 *         </ul>
 *     </li>
 * </ul>
 * If {@code cacheKeys} is not used or is empty:
 * <ul>
 *     <li>A {@code private static final java.util.Map<String, EnumType> cacheMap} field (mapping enum constant names).</li>
 *     <li>Overloaded {@code public static EnumType getValueOf(String name)} methods for name lookup.</li>
 * </ul>
 * This avoids repeated array allocation and provides convenient access methods.
 * It relies on the Javac AST manipulation utilities provided by {@link CommonProcessor}.
 *
 * @author ANZO
 * @see CommonProcessor
 * @see Statics#EXTENDED_ENUM_ANNOTATION_PATH
 */
@AutoService(Processor.class)
@SupportedAnnotationTypes(Statics.EXTENDED_ENUM_ANNOTATION_PATH)
public class ExtendedEnumProcessor extends CommonProcessor {
	/**
	 * Processes annotations for a processing round.
	 * <p>
	 * It finds all enums annotated with {@code @ExtendedEnum}, verifies processor initialization,
	 * and then modifies the AST of each found enum to add the caching field and utility methods.
	 *
	 * @param annotations The annotation types requested to be processed.
	 * @param roundEnv    Environment for information about the current and prior round.
	 * @return {@code false} always, as this processor does not claim the annotations.
	 */
	@Override
	public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
		if (this.maker == null || this.trees == null || this.names == null) {
			logError("Processor essentials not initialized. Skipping processing.");
			return false;
		}

		if (roundEnv.processingOver() || annotations.isEmpty()) {
			return false;
		}

		final TypeElement extendedEnumAnnotation = processingEnv.getElementUtils().getTypeElement(Statics.EXTENDED_ENUM_ANNOTATION_PATH);
		if (extendedEnumAnnotation == null) {
			logError("Could not resolve " + Statics.EXTENDED_ENUM_ANNOTATION_PATH + " annotation.");
			return false;
		}

		for (Element element : roundEnv.getElementsAnnotatedWith(extendedEnumAnnotation)) {
			if (element.getKind() == ElementKind.ENUM) {
				final TreePath path = trees.getPath(element);
				if (path == null) {
					logWarn("Could not get TreePath for element: " + element.getSimpleName());
					continue;
				}
				final JCClassDecl classDecl = (JCClassDecl) path.getLeaf();

				// List cache by ordinal
				addCacheListField(classDecl);
				addGetValueByOrdinalMethod(classDecl);
				addGetValuesMethod(classDecl);

				// Map cache by enum name
				addDefaultNameCacheMapField(classDecl);
				addDefaultGetValueOfMethodWithTwoParams(classDecl);
				addDefaultGetValueOfMethodWithOneParam(classDecl);

				// Map cache by specified keys
				final Set<String> cacheKeyFieldNames = getCacheKeyFieldNames(element);
				List<JCStatement> allCustomMapPutStatements = List.nil();
				if (!cacheKeyFieldNames.isEmpty()) {
					for (String fieldName : cacheKeyFieldNames) {
						final JCExpression fieldTypeNode = findFieldType(classDecl, fieldName);
						if (fieldTypeNode == null) {
							logError("Field '" + fieldName + "' specified in @ExtendedEnum cacheKeys not found in enum " + classDecl.name.toString() + ".");
							continue;
						}

						String capitalizedFieldName = capitalize(fieldName);
						String mapFieldName = "cacheMapBy" + capitalizedFieldName;
						String getterMethodName = "getValueOf" + capitalizedFieldName;

						if (fieldExists(classDecl, mapFieldName)) {
							continue;
						}
						if (methodExists(classDecl, getterMethodName, 1)) {
							continue;
						}

						addCustomCacheMapFieldDeclaration(classDecl, fieldTypeNode, mapFieldName);

						final JCStatement putStatement = createCustomCacheMapPutStatement(
								names.fromString("enumConstant"),
								fieldName,
								mapFieldName
						);
						allCustomMapPutStatements = allCustomMapPutStatements.append(putStatement);

						addCustomValueGetterMethod(classDecl, fieldName, fieldTypeNode, getterMethodName, mapFieldName);
					}

					if (allCustomMapPutStatements.nonEmpty()) {
						JCIdent enumIdent = maker.Ident(classDecl.name);
						com.sun.tools.javac.util.Name loopVarName = names.fromString("enumConstant");
						final JCVariableDecl loopVar = maker.VarDef(maker.Modifiers(0), loopVarName, enumIdent, null);

						JCExpression valuesCall = maker.Apply(
								List.nil(), maker.Select(enumIdent, names.fromString("values")), List.nil()
						);

						final JCBlock forLoopBody = maker.Block(0, allCustomMapPutStatements);
						final JCEnhancedForLoop singleForLoop = maker.ForeachLoop(loopVar, valuesCall, forLoopBody);
						final JCBlock combinedStaticInitializer = maker.Block(Flags.STATIC, List.of(singleForLoop));
						classDecl.defs = classDecl.defs.append(combinedStaticInitializer);
					}
				}
			} else {
				logWarn("@" + extendedEnumAnnotation.getSimpleName() + " applied to non-enum: " + element.getSimpleName());
			}
		}

		return false;
	}

	/**
	 * Retrieves the cache key field names from the given element, if the element is annotated with the extended enum annotation.
	 *
	 * @param element the element to retrieve cache key field names from
	 * @return a set of string representing the cache key field names
	 */
	private @NotNull Set<String> getCacheKeyFieldNames(@NotNull Element element) {
		Set<String> keys = new HashSet<>();
		AnnotationMirror extendedEnumAnnotationMirror = null;

		for (AnnotationMirror am : element.getAnnotationMirrors()) {
			if (am.getAnnotationType().toString().equals(Statics.EXTENDED_ENUM_ANNOTATION_PATH)) {
				extendedEnumAnnotationMirror = am;
				break;
			}
		}

		if (extendedEnumAnnotationMirror != null) {
			final Map<? extends ExecutableElement, ? extends AnnotationValue> elementValues =
					processingEnv.getElementUtils().getElementValuesWithDefaults(extendedEnumAnnotationMirror);

			for (Map.Entry<? extends ExecutableElement, ? extends AnnotationValue> entry : elementValues.entrySet()) {
				if (entry.getKey().getSimpleName().toString().equals("cacheKeys")) {
					Object value = entry.getValue().getValue();
					if (value instanceof java.util.List) {
						for (Object item : (java.util.List<?>) value) {
							if (item instanceof AnnotationValue) {
								Object innerValue = ((AnnotationValue) item).getValue();
								if (innerValue instanceof String) {
									keys.add((String) innerValue);
								}
							}
						}
					}
					break;
				}
			}
		}
		return keys;
	}

	/**
	 * Adds a {@code private static final List<E> cacheList} field to the enum's AST.
	 * The field is initialized with {@code java.util.List.of(values())}.
	 * If a field named "cacheList" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 */
	private void addCacheListField(JCClassDecl classDecl) {
		final String fieldName = "cacheList";
		if (fieldExists(classDecl, fieldName))
			return;

		final JCExpression listType = maker.TypeApply(
				createQualifiedName("java.util.List"),
				List.of(maker.Ident(classDecl.name))
		);

		final JCExpression valuesCall = maker.Apply(
				List.nil(),
				maker.Select(
						maker.Ident(classDecl.name),
						names.fromString("values")),
				List.nil()
		);

		final JCExpression init = maker.Apply(
				List.nil(),
				maker.Select(
						createQualifiedName("java.util.List"),
						names.fromString("of")),
				List.of(valuesCall)
		);

		final JCVariableDecl cacheListField = maker.VarDef(
				maker.Modifiers(Flags.PRIVATE | Flags.STATIC | Flags.FINAL),
				names.fromString(fieldName),
				listType,
				init
		);

		classDecl.defs = classDecl.defs.append(cacheListField);
	}

	/**
	 * Adds a {@code public static E getValue(int ordinal)} method to the enum's AST.
	 * The method body returns {@code cacheList.get(ordinal)}.
	 * If a method named "getValue" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 */
	private void addGetValueByOrdinalMethod(JCClassDecl classDecl) {
		final String methodName = "getValue";
		if (methodExists(classDecl, methodName, 1))
			return;

		final JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				maker.Ident(classDecl.name),
				List.nil(),
				List.of(maker.VarDef(
						maker.Modifiers(Flags.PARAMETER),
						names.fromString("ordinal"),
						maker.TypeIdent(TypeTag.INT),
						null
				)),
				List.nil(),
				maker.Block(0, List.of(
						maker.Return(
								maker.Apply(
										List.nil(),
										maker.Select(
												maker.Ident(names.fromString("cacheList")),
												names.fromString("get")),
										List.of(maker.Ident(names.fromString("ordinal")))
								)
						)
				)),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds a {@code public static List<E> getValues()} method to the enum's AST.
	 * The method body returns the {@code cacheList} field.
	 * If a method named "getValues" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 */
	private void addGetValuesMethod(JCClassDecl classDecl) {
		final String methodName = "getValues";
		if (methodExists(classDecl, methodName, 0))
			return;

		final JCExpression returnType = maker.TypeApply(
				createQualifiedName("java.util.List"),
				List.of(maker.Ident(classDecl.name))
		);

		final JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				returnType,
				List.nil(), // type params
				List.nil(), // params
				List.nil(), // thrown
				maker.Block(0, List.of(
						maker.Return(maker.Ident(names.fromString("cacheList")))
				)),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds a static {@code Map<String, EnumType> cacheMap} field to the enum.
	 * The map is initialized with enum names as keys.
	 * Uses {@code Collectors.toMap} with name mapping.
	 *
	 * @param classDecl The enum class declaration to modify
	 */
	private void addDefaultNameCacheMapField(JCClassDecl classDecl) {
		final String fieldName = "cacheMap";
		if (fieldExists(classDecl, fieldName)) return;

		JCExpression stringType = createQualifiedName("java.lang.String");
		JCIdent enumTypeIdent = maker.Ident(classDecl.name);
		JCExpression mapType = maker.TypeApply(
				createQualifiedName("java.util.Map"),
				List.of(stringType, enumTypeIdent)
		);

		JCExpression cacheListExpr = maker.Ident(names.fromString("cacheList"));
		JCExpression streamCall = maker.Select(cacheListExpr, names.fromString("stream"));
		JCExpression streamApply = maker.Apply(List.nil(), streamCall, List.nil());

		com.sun.tools.javac.util.Name lambdaParamName = names.fromString("e");

		// Lambda e -> e.name()
		JCVariableDecl paramE1 = maker.VarDef(maker.Modifiers(Flags.PARAMETER), lambdaParamName, enumTypeIdent, null);
		JCExpression eNameCall = maker.Apply(List.nil(), maker.Select(maker.Ident(lambdaParamName), names.fromString("name")), List.nil());
		JCLambda lambda1 = maker.Lambda(List.of(paramE1), eNameCall);

		// Lambda e -> e
		JCVariableDecl paramE2 = maker.VarDef(maker.Modifiers(Flags.PARAMETER), lambdaParamName, enumTypeIdent, null);
		JCLambda lambda2 = maker.Lambda(List.of(paramE2), maker.Ident(lambdaParamName));

		JCExpression collectorsToMap = maker.Select(createQualifiedName("java.util.stream.Collectors"), names.fromString("toMap"));
		JCExpression toMapApply = maker.Apply(List.nil(), collectorsToMap, List.of(lambda1, lambda2));

		JCExpression collectCall = maker.Apply(
				List.nil(),
				maker.Select(streamApply, names.fromString("collect")),
				List.of(toMapApply)
		);

		JCVariableDecl cacheMapField = maker.VarDef(
				maker.Modifiers(Flags.PRIVATE | Flags.STATIC | Flags.FINAL),
				names.fromString(fieldName),
				mapType,
				collectCall
		);
		classDecl.defs = classDecl.defs.append(cacheMapField);
	}

	/**
	 * Adds the {@code getValueOf(String name, EnumType defaultValue)} method.
	 * The method performs case-insensitive lookup in cacheMap.
	 *
	 * @param classDecl The enum class declaration to modify
	 */
	private void addDefaultGetValueOfMethodWithTwoParams(JCClassDecl classDecl) {
		String methodName = "getValueOf";
		if (methodExists(classDecl, methodName, 2)) return;

		JCVariableDecl paramName = maker.VarDef(
				maker.Modifiers(Flags.PARAMETER),
				names.fromString("name"),
				createQualifiedName("java.lang.String"),
				null
		);
		JCVariableDecl paramDefault = maker.VarDef(
				maker.Modifiers(Flags.PARAMETER),
				names.fromString("defaultValue"),
				maker.Ident(classDecl.name),
				null
		);
		List<JCVariableDecl> params = List.of(paramName, paramDefault);

		JCExpression getOrDefault = maker.Apply(
				List.nil(),
				maker.Select(maker.Ident(names.fromString("cacheMap")), names.fromString("getOrDefault")),
				List.of(maker.Ident(names.fromString("name")), maker.Ident(names.fromString("defaultValue")))
		);

		JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				maker.Ident(classDecl.name),
				List.nil(),
				params,
				List.nil(),
				maker.Block(0, List.of(maker.Return(getOrDefault))),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds the overloaded {@code getValueOf(String name)} method.
	 * Delegates to the two-parameter version with null default value.
	 *
	 * @param classDecl The enum class declaration to modify
	 */
	private void addDefaultGetValueOfMethodWithOneParam(JCClassDecl classDecl) {
		String methodName = "getValueOf";
		if (methodExists(classDecl, methodName, 1)) return;

		JCVariableDecl paramName = maker.VarDef(
				maker.Modifiers(Flags.PARAMETER),
				names.fromString("name"),
				createQualifiedName("java.lang.String"),
				null
		);
		List<JCVariableDecl> params = List.of(paramName);

		JCExpression call = maker.Apply(
				List.nil(),
				maker.Ident(names.fromString(methodName)),
				List.of(
						maker.Ident(names.fromString("name")),
						maker.Literal(TypeTag.BOT, null)
				)
		);

		JCMethodDecl method = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				maker.Ident(classDecl.name),
				List.nil(),
				params,
				List.nil(),
				maker.Block(0, List.of(maker.Return(call))),
				null
		);

		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds a custom cache map field declaration (e.g., cacheMapBy<FieldName> = new HashMap<>).
	 * This method ONLY adds the field definition. Population is handled separately.
	 */
	private void addCustomCacheMapFieldDeclaration(@NotNull JCClassDecl classDecl, JCExpression fieldType, String mapFieldName) {
		JCIdent enumIdent = maker.Ident(classDecl.name);
		JCExpression mapKeyType = getBoxedType(fieldType);

		JCExpression mapGenericType = maker.TypeApply(
				createQualifiedName("java.util.Map"),
				List.of(mapKeyType, enumIdent)
		);

		JCExpression newHashMapInstance = maker.NewClass(
				null,
				List.nil(),
				maker.TypeApply(createQualifiedName("java.util.HashMap"), List.nil()),
				List.nil(),
				null
		);

		JCVariableDecl cacheMapVDef = maker.VarDef(
				maker.Modifiers(Flags.PRIVATE | Flags.STATIC | Flags.FINAL),
				names.fromString(mapFieldName),
				mapGenericType,
				newHashMapInstance
		);
		classDecl.defs = classDecl.defs.append(cacheMapVDef);
	}

	/**
	 * Creates a single Javac "put" statement for populating a custom cache map
	 * within a loop.
	 * (e.g., mapFieldName.put(loopVarAccess.originalFieldName, loopVarAccess);)
	 *
	 * @param loopVarNameIdentifier The Javac Name of the loop variable (e.g., "enumConstant").
	 * @param originalFieldName     The name of the enum field to use as the key.
	 * @param mapFieldName          The name of the map to put into.
	 * @return A JCStatement representing the "put" call.
	 */
	private JCStatement createCustomCacheMapPutStatement(
			com.sun.tools.javac.util.Name loopVarNameIdentifier,
			String originalFieldName,
			String mapFieldName) {

		final JCIdent loopVarAccess = maker.Ident(loopVarNameIdentifier);
		final JCExpression fieldAccessInLoop = maker.Select(loopVarAccess, names.fromString(originalFieldName));
		final JCExpression putCall = maker.Apply(
				List.nil(),
				maker.Select(maker.Ident(names.fromString(mapFieldName)), names.fromString("put")),
				List.of(fieldAccessInLoop, loopVarAccess)
		);

		return maker.Exec(putCall);
	}

	/**
	 * Adds a custom getter method (e.g., getValueOf<FieldName>) for the custom cache map.
	 *
	 * @param classDecl        The enum class declaration.
	 * @param originalFieldName The name of the enum field (used as parameter name).
	 * @param fieldType        The JCTree type of the enum field (used as parameter type).
	 * @param methodName       The generated name for the getter method (e.g., "getValueOfType").
	 * @param mapFieldName     The name of the map field to get from (e.g., "cacheMapByType").
	 */
	private void addCustomValueGetterMethod(@NotNull JCClassDecl classDecl, String originalFieldName, JCExpression fieldType, String methodName, String mapFieldName) {
		JCIdent enumIdent = maker.Ident(classDecl.name);

		// Parameter: FieldType fieldName
		JCVariableDecl methodParam = maker.VarDef(
				maker.Modifiers(Flags.PARAMETER),
				names.fromString(originalFieldName), // Parameter name matches field name
				fieldType,                           // Parameter type is the original field type
				null
		);

		// cacheMapByFieldName.get(fieldName)
		JCExpression mapGetCall = maker.Apply(
				List.nil(),
				maker.Select(maker.Ident(names.fromString(mapFieldName)), names.fromString("get")),
				List.of(maker.Ident(methodParam.name))
		);

		// public static EnumType getValueOfFieldName(FieldType fieldName) { return map.get(fieldName); }
		JCMethodDecl getterMethod = maker.MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				enumIdent, // return type
				List.nil(), // type parameters
				List.of(methodParam), // parameters
				List.nil(), // thrown exceptions
				maker.Block(0, List.of(maker.Return(mapGetCall))),
				null
		);
		classDecl.defs = classDecl.defs.append(getterMethod);
	}
}