package host.anzo.commons.processors;

import com.sun.source.util.Trees;
import com.sun.tools.javac.code.Flags;
import com.sun.tools.javac.code.TypeTag;
import com.sun.tools.javac.processing.JavacProcessingEnvironment;
import com.sun.tools.javac.tree.JCTree;
import com.sun.tools.javac.tree.Pretty;
import com.sun.tools.javac.tree.TreeMaker;
import com.sun.tools.javac.util.List;
import com.sun.tools.javac.util.Names;
import host.anzo.commons.processors.utils.Permit;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.Messager;
import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.tools.Diagnostic;
import java.io.StringWriter;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Proxy;

import static com.sun.tools.javac.tree.JCTree.*;

/**
 * Base processor class providing common AST manipulation utilities.
 * <p>
 * Handles low-level Javac API access and provides:
 * <ul>
 *     <li>Field/Method existence checks</li>
 *     <li>Qualified type name construction</li>
 *     <li>Custom javac environment unwrapping</li>
 *     <li>Logging facilities</li>
 * </ul>
 *
 * @author ANZO
 */
public abstract class CommonProcessor extends AbstractProcessor {
	/**
	 * Factory for creating Javac AST nodes ({@link JCTree}). Initialized in {@link #init(ProcessingEnvironment)}.
	 */
	protected TreeMaker maker;

	/**
	 * Utility for working with Javac ASTs ({@link JCTree}). Initialized in {@link #init(ProcessingEnvironment)}.
	 * Provides access to source tree structures.
	 */
	protected Trees trees;

	/**
	 * Interface for reporting errors, warnings, and other notices during annotation processing.
	 * Obtained from the {@link ProcessingEnvironment}.
	 */
	protected Messager messager;

	/**
	 * Factory for creating Javac {@link com.sun.tools.javac.util.Name} objects (identifiers).
	 * Initialized in {@link #init(ProcessingEnvironment)}.
	 */
	protected Names names;

	protected boolean isVerbose;

	/**
	 * Specifies the latest supported Java source version.
	 *
	 * @return The latest {@link SourceVersion}.
	 */
	@Override
	public SourceVersion getSupportedSourceVersion() {
		return SourceVersion.latest();
	}

	/**
	 * Initializes the processor, setting up common utilities like {@link Messager},
	 * {@link TreeMaker}, {@link Trees}, and {@link Names}.
	 * <p>
	 * This implementation attempts to unwrap the {@link ProcessingEnvironment} if it's potentially
	 * wrapped by JetBrains tools (like in IntelliJ IDEA) to ensure access to the underlying
	 * {@link JavacProcessingEnvironment}. If the unwrapping fails or the environment is not
	 * a {@code JavacProcessingEnvironment}, an error is logged, and the Javac-specific tools
	 * ({@code maker}, {@code trees}, {@code names}) will remain null.
	 *
	 * @param procEnv Environment providing access to facilities like {@link Messager},
	 *                {@link javax.annotation.processing.Filer}, and {@link javax.lang.model.util.Elements}.
	 * @see #getJavacProcessingEnvironment(Object)
	 */
	@Override
	public synchronized void init(ProcessingEnvironment procEnv) {
		super.init(procEnv);
		this.messager = procEnv.getMessager();
		final ProcessingEnvironment unwrappedProcEnv = getJavacProcessingEnvironment(procEnv);
		if (unwrappedProcEnv instanceof JavacProcessingEnvironment javacEnv) {
			this.maker = TreeMaker.instance(javacEnv.getContext());
			this.trees = Trees.instance(javacEnv);
			this.names = Names.instance(javacEnv.getContext());
		}
		else {
			logError("Could not obtain JavacProcessingEnvironment. Annotation processing might not work correctly. Original env: " + procEnv.getClass().getName());
		}
	}

	/**
	 * Checks if a field with the specified name already exists within the given class declaration.
	 *
	 * @param classDecl The Javac AST node representing the class declaration.
	 * @param fieldName The name of the field to check for.
	 * @return {@code true} if a field with the given name exists, {@code false} otherwise.
	 */
	protected boolean fieldExists(@NotNull JCClassDecl classDecl, String fieldName) {
		return classDecl.defs.stream()
				.anyMatch(def -> def instanceof JCVariableDecl
						&& ((JCVariableDecl) def).getName().contentEquals(fieldName));
	}

	/**
	 * Checks method existence with parameter count constraint
	 *
	 * @param classDecl   The class declaration to check
	 * @param methodName  Target method name
	 * @param paramCount  Expected number of parameters
	 * @return {@code true} if method exists with specified name and parameter count
	 */
	protected boolean methodExists(@NotNull JCClassDecl classDecl, String methodName, int paramCount) {
		return classDecl.defs.stream()
				.anyMatch(def -> def instanceof JCMethodDecl
						&& ((JCMethodDecl) def).getName().contentEquals(methodName)
						&& ((JCMethodDecl) def).getParameters().size() == paramCount);
	}

	/**
	 * Creates a Javac AST expression ({@link JCExpression}) representing a fully qualified name
	 * (e.g., {@code java.util.List}).
	 *
	 * @param fullTypeName The fully qualified name as a String (e.g., "java.lang.String").
	 * @return A {@link JCExpression} representing the qualified name, suitable for use in AST generation.
	 *         This will typically be a {@link JCIdent} for single-part names or a
	 *         {@link JCFieldAccess} (Select) for multi-part names.
	 */
	protected JCExpression createQualifiedName(int pos, @NotNull String fullTypeName) {
		maker.at(pos);
		String[] parts = fullTypeName.split("\\.");
		JCExpression expr = maker.Ident(names.fromString(parts[0]));
		for (int i = 1; i < parts.length; i++) {
			expr = maker.Select(expr, names.fromString(parts[i]));
		}
		return expr;
	}

	/**
	 * Finds the field type of a given field name in a class declaration.
	 *
	 * @param classDecl The class declaration in which to search for the field.
	 * @param fieldName The name of the field whose type is to be found.
	 * @return The type of the field if found, otherwise null.
	 */
	protected @Nullable JCExpression findFieldType(@NotNull JCClassDecl classDecl, String fieldName) {
		for (JCTree def : classDecl.defs) {
			if (def instanceof JCVariableDecl field) {
				if (field.name.contentEquals(fieldName)) {
					return field.vartype;
				}
			}
		}
		return null;
	}

	/**
	 * Finds the return type of a given method name in the given class declaration.
	 *
	 * @param element       The class declaration in which to search for the method.
	 * @param methodName    The name of the method whose return type is to be found.
	 * @param pos           The maker position
	 * @return The type of the method return if found, otherwise null.
	 */
	protected JCExpression findMethodReturnType(@NotNull Element element, String methodName, int pos) {
		final TypeElement elementType = (TypeElement) element;
		for (Element member : processingEnv.getElementUtils().getAllMembers(elementType)) {
			if (member.getKind() == ElementKind.METHOD &&
					member.getSimpleName().contentEquals(methodName)) {
				final ExecutableElement method = (ExecutableElement) member;
				return createQualifiedName(pos, method.getReturnType().toString());
			}
		}
		return null;
	}

	protected void addToStaticInitializer(@NotNull JCClassDecl classDecl, JCStatement stmt) {
		final JCBlock staticBlock = maker.Block(Flags.STATIC, List.of(stmt));
		classDecl.defs = classDecl.defs.append(staticBlock);
	}

	/**
	 * Capitalizes the first character of the input string.
	 *
	 * @param str The string to be capitalized.
	 * @return The capitalized string.
	 */
	protected @NotNull String capitalize(@NotNull String str) {
		return str.substring(0, 1).toUpperCase() + str.substring(1);
	}

	/**
	 * Returns the boxed type expression for a given primitive type expression.
	 * If the input type is not a primitive type, it is returned unchanged.
	 *
	 * @param primitiveOrRefType The {@link JCExpression} representing the type.
	 * @return The {@link JCExpression} for the boxed type or the original type if not primitive.
	 */
	protected @NotNull JCExpression getBoxedType(@NotNull JCExpression primitiveOrRefType, int pos) {
		if (primitiveOrRefType instanceof JCPrimitiveTypeTree primitiveTypeTree) {
			TypeTag tag = primitiveTypeTree.typetag;
			return switch (tag) {
				case BYTE -> createQualifiedName(pos, "java.lang.Byte");
				case SHORT -> createQualifiedName(pos,"java.lang.Short");
				case INT -> createQualifiedName(pos,"java.lang.Integer");
				case LONG -> createQualifiedName(pos,"java.lang.Long");
				case FLOAT -> createQualifiedName(pos,"java.lang.Float");
				case DOUBLE -> createQualifiedName(pos,"java.lang.Double");
				case CHAR -> createQualifiedName(pos,"java.lang.Character");
				case BOOLEAN -> createQualifiedName(pos,"java.lang.Boolean");
				default -> primitiveOrRefType; // Should not happen for recognized primitive types
			};
		}
		return primitiveOrRefType;
	}

	/**
	 * Attempts to retrieve the underlying {@link JavacProcessingEnvironment}
	 * from a potentially wrapped processing environment object. This method
	 * handles various wrapping strategies used by different tools and
	 * environments, such as Gradle and IntelliJ IDEA.
	 *
	 * <p>If the input object is already an instance of
	 * {@code JavacProcessingEnvironment}, it is returned directly. Otherwise,
	 * the method attempts to unwrap it by searching for delegate fields
	 * recursively until the desired environment is found.</p>
	 *
	 * @param procEnv The processing environment object, potentially wrapped.
	 * @return The unwrapped {@link JavacProcessingEnvironment} if found,
	 *         otherwise {@code null}.
	 */
	protected JavacProcessingEnvironment getJavacProcessingEnvironment(Object procEnv) {
		if (procEnv instanceof JavacProcessingEnvironment)
			return (JavacProcessingEnvironment) procEnv;

		// try to find a "delegate" field in the object, and use this to try to obtain a JavacProcessingEnvironment
		for (Class<?> procEnvClass = procEnv.getClass(); procEnvClass != null; procEnvClass = procEnvClass.getSuperclass()) {
			Object delegate = tryGetDelegateField(procEnvClass, procEnv);
			if (delegate == null)
				delegate = tryGetProxyDelegateToField(procEnvClass, procEnv);
			if (delegate == null)
				delegate = tryGetProcessingEnvField(procEnvClass, procEnv);

			if (delegate != null)
				return getJavacProcessingEnvironment(delegate);
		}

		return null;
	}

	/**
	 * Tries to find a "processingEnv" field in the given object, as is used by
	 * Kotlin incremental processing. This is a fallback approach used when the
	 * given object is not a {@code JavacProcessingEnvironment} directly, but rather
	 * wraps one.
	 *
	 * @param delegateClass The class of the object in which to search for the
	 *                       "processingEnv" field.
	 * @param instance The object in which to search for the "processingEnv" field.
	 * @return The delegate object if found, otherwise {@code null}.
	 */
	private @Nullable Object tryGetProcessingEnvField(Class<?> delegateClass, Object instance) {
		try {
			return Permit.getField(delegateClass, "processingEnv").get(instance);
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * Attempts to find a "delegate" field in the given object, as is used by Gradle
	 * incremental processing. This is a fallback approach used when the given object
	 * is not a {@code JavacProcessingEnvironment} directly, but rather wraps one.
	 *
	 * @param delegateClass The class of the object in which to search for the delegate.
	 * @param instance The object in which to search for the delegate.
	 * @return The delegate object if found, otherwise {@code null}.
	 */
	private @Nullable Object tryGetDelegateField(Class<?> delegateClass, Object instance) {
		try {
			return Permit.getField(delegateClass, "delegate").get(instance);
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * Attempts to retrieve the "delegateTo" field from a proxy instance used in
	 * IntelliJ IDEA versions 2020.3 and above. This method extracts the delegate
	 * by accessing the invocation handler of the proxy and fetching the specified
	 * field.
	 *
	 * @param delegateClass The class type from which the proxy delegate field is to be extracted.
	 * @param instance The proxy instance whose "delegateTo" field is to be accessed.
	 * @return The value of the "delegateTo" field if found, otherwise {@code null}.
	 */
	private @Nullable Object tryGetProxyDelegateToField(Class<?> delegateClass, Object instance) {
		try {
			InvocationHandler handler = Proxy.getInvocationHandler(instance);
			return Permit.getField(handler.getClass(), "val$delegateTo").get(handler);
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * Prints the Abstract Syntax Tree (AST) of the given compilation unit to the log.
	 *
	 * <p>This method uses the {@link Pretty} printer to generate a textual representation
	 * of the AST for debugging or analysis purposes. If an error occurs during printing,
	 * an error message is logged along with the stack trace.</p>
	 *
	 * @param compilationUnit the {@link JCCompilationUnit} representing the compilation unit
	 *                        whose AST is to be printed; may be {@code null}
	 *
	 * @see Pretty
	 * @see JCCompilationUnit
	 */
	protected void printCompilationUnitAst(JCCompilationUnit compilationUnit) {
		if (compilationUnit == null) {
			logWarn("Cannot print AST, compilation unit is null.");
			return;
		}
		try {
			StringWriter stringWriter = new StringWriter();
			Pretty printer = new Pretty(stringWriter, true);
			compilationUnit.accept(printer);
			log("AST for " + compilationUnit.getSourceFile().getName() + ":\n" + stringWriter);
		} catch (Exception e) {
			logError("Error printing AST: " + e.getMessage());
			e.printStackTrace(System.err);
		}
	}


	/**
	 * Logs an informational message using the processing environment's {@link Messager}.
	 * The message is prefixed with the simple name of this processor class.
	 *
	 * @param msg The message to log.
	 */
	protected void log(String msg) {
		if (messager != null) {
			messager.printMessage(Diagnostic.Kind.NOTE, "[" + getClass().getSimpleName() + "] " + msg);
		} else {
			System.out.println("NOTE: [" + getClass().getSimpleName() + "] " + msg + " (Messager not initialized)");
		}
	}

	/**
	 * Logs an error message using the processing environment's {@link Messager}.
	 * The message is prefixed with the simple name of this processor class.
	 * Note that reporting an error typically halts the compilation process.
	 *
	 * @param msg The error message to log.
	 */
	protected void logError(String msg) {
		if (messager != null) {
			messager.printMessage(Diagnostic.Kind.ERROR, "[" + getClass().getSimpleName() + "] " + msg);
		} else {
			System.err.println("ERROR: [" + getClass().getSimpleName() + "] " + msg + " (Messager not initialized)");
		}
	}

	/**
	 * Logs a warning message using the processing environment's {@link Messager}.
	 * The message is prefixed with the simple name of this processor class.
	 *
	 * @param msg The warning message to log.
	 */
	protected void logWarn(String msg) {
		if (messager != null) {
			messager.printMessage(Diagnostic.Kind.WARNING, "[" + getClass().getSimpleName() + "] " + msg);
		} else {
			System.out.println("WARN: [" + getClass().getSimpleName() + "] " + msg + " (Messager not initialized)");
		}
	}
}