package host.anzo.commons.processors.utils;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

// sunapi suppresses javac's warning about using Unsafe; 'all' suppresses eclipse's warning about the unspecified 'sunapi' key. Leave them both.
// Yes, javac's definition of the word 'all' is quite contrary to what the dictionary says it means. 'all' does NOT include 'sunapi' according to javac.
@SuppressWarnings({"sunapi", "all"})
public class Permit {
	private Permit() {}

	private static final long ACCESSIBLE_OVERRIDE_FIELD_OFFSET;
	private static final IllegalAccessException INIT_ERROR;
	private static final sun.misc.Unsafe UNSAFE = (sun.misc.Unsafe) reflectiveStaticFieldAccess(sun.misc.Unsafe.class, "theUnsafe");

	static {
		Field f;
		long g;
		Throwable ex;

		try {
			g = getOverrideFieldOffset();
			ex = null;
		} catch (Throwable t) {
			f = null;
			g = -1L;
			ex = t;
		}

		ACCESSIBLE_OVERRIDE_FIELD_OFFSET = g;
		if (ex == null) INIT_ERROR = null;
		else if (ex instanceof IllegalAccessException) INIT_ERROR = (IllegalAccessException) ex;
		else {
			INIT_ERROR = new IllegalAccessException("Cannot initialize Unsafe-based permit");
			INIT_ERROR.initCause(ex);
		}
	}

	/**
	 * Set the accessible flag of the given accessible object to true.
	 * <p>
	 * This method is used to set the accessible flag of a field, method, or
	 * constructor without having to declare the checked exception
	 * IllegalAccessException.
	 * <p>
	 * @param accessor the accessible object to set the accessible flag of
	 * @return the given accessible object
	 */
	public static <T extends AccessibleObject> T setAccessible(T accessor) {
		if (INIT_ERROR == null) {
			UNSAFE.putBoolean(accessor, ACCESSIBLE_OVERRIDE_FIELD_OFFSET, true);
		} else {
			accessor.setAccessible(true);
		}

		return accessor;
	}

	/**
	 * Attempts to retrieve the offset of the "override" field in the AccessibleObject class.
	 * <p>
	 * This method first tries to locate the "override" field directly within the AccessibleObject class.
	 * If unsuccessful, it saves the encountered exception and attempts to retrieve the field offset
	 * from a fake class. This approach is risky but currently works for all AccessibleObjects in Java.
	 * If neither approach succeeds, it throws the initially saved exception.
	 *
	 * @return the offset of the "override" field
	 * @throws Throwable if the field cannot be accessed
	 */
	private static long getOverrideFieldOffset() throws Throwable {
		Field f = null;
		Throwable saved = null;
		try {
			f = AccessibleObject.class.getDeclaredField("override");
		} catch (Throwable t) {
			saved = t;
		}

		if (f != null) {
			return UNSAFE.objectFieldOffset(f);
		}
		// The below seems very risky, but for all AccessibleObjects in java today it does work, and starting with JDK12, making the field accessible is no longer possible.
		try {
			return UNSAFE.objectFieldOffset(Fake.class.getDeclaredField("override"));
		} catch (Throwable t) {
			throw saved;
		}
	}

	static class Fake {
		boolean override;
		Object accessCheckCache;
	}

	/**
	 * Gets a method declared in the given class or any of its superclasses.
	 * <p>
	 * This method will traverse the class hierarchy and find the given method, regardless of
	 * whether it is declared as public, private, protected, or default (package-private).
	 * <p>
	 * @param c the class to search for the method
	 * @param mName the name of the method to search for
	 * @param parameterTypes the parameter types of the method to search for
	 * @return the method found, or {@code null} if no such method exists
	 * @throws NoSuchMethodException if the method does not exist
	 */
	public static Method getMethod(Class<?> c, String mName, Class<?>... parameterTypes) throws NoSuchMethodException {
		Method m = null;
		Class<?> oc = c;
		while (c != null) {
			try {
				m = c.getDeclaredMethod(mName, parameterTypes);
				break;
			} catch (NoSuchMethodException e) {}
			c = c.getSuperclass();
		}

		if (m == null) throw new NoSuchMethodException(oc.getName() + " :: " + mName + "(args)");
		return setAccessible(m);
	}

	/**
	 * Like {@link #getMethod(Class, String, Class[])}, but won't throw an exception if the method doesn't exist.
	 * <p>
	 * Instead, this method will return {@code null} if the method doesn't exist.
	 * <p>
	 * @param c the class to search for the method
	 * @param mName the name of the method to search for
	 * @param parameterTypes the parameter types of the method to search for
	 * @return the method found, or {@code null} if no such method exists
	 */
	public static @Nullable Method permissiveGetMethod(Class<?> c, String mName, Class<?>... parameterTypes) {
		try {
			return getMethod(c, mName, parameterTypes);
		} catch (Exception ignore) {
			return null;
		}
	}

	/**
	 * Retrieves a field with the given name from the specified class or its superclasses.
	 * <p>
	 * This method searches for a field with the specified name in the given class and, if not found,
	 * traverses the superclass hierarchy to find the field. Once found, it sets the field
	 * to be accessible and returns it.
	 * <p>
	 * @param c the class to search for the field
	 * @param fName the name of the field to search for
	 * @return the found field with accessible set to true
	 * @throws NoSuchFieldException if the field does not exist in the given class or its superclasses
	 */
	public static Field getField(Class<?> c, String fName) throws NoSuchFieldException {
		Field f = null;
		Class<?> d = c;
		while (d != null) {
			try {
				f = d.getDeclaredField(fName);
				break;
			} catch (NoSuchFieldException e) {}
			d = d.getSuperclass();
		}
		if (f == null) throw new NoSuchFieldException(c.getName() + " :: " + fName);

		return setAccessible(f);
	}

	/**
	 * Retrieves a field with the given name from the specified class or its superclasses.
	 * <p>
	 * This method searches for a field with the specified name in the given class and, if not found,
	 * traverses the superclass hierarchy to find the field. Once found, it sets the field
	 * to be accessible and returns it. If no field is found, it returns null.
	 * <p>
	 * @param c the class to search for the field
	 * @param fName the name of the field to search for
	 * @return the found field with accessible set to true, or null if the field does not exist
	 */
	public static @Nullable Field permissiveGetField(Class<?> c, String fName) {
		try {
			return getField(c, fName);
		} catch (Exception ignore) {
			return null;
		}
	}

	/**
	 * Like {@link #readField(Class, Field, Object)}, but won't throw an exception if the field
	 * doesn't exist or is not accessible.
	 * <p>
	 * Instead, this method will return {@code null} if the field doesn't exist or if any exception
	 * occurs while trying to read the field.
	 * <p>
	 * @param type the type of the object to be returned
	 * @param f the field to read
	 * @param instance the object containing the field
	 * @return the value of the field, or {@code null} if the field doesn't exist or any exception
	 *         occurs
	 */
	public static <T> @Nullable T permissiveReadField(Class<T> type, Field f, Object instance) {
		try {
			return type.cast(f.get(instance));
		} catch (Exception ignore) {
			return null;
		}
	}

	/**
	 * Finds a constructor with the given parameter types and makes it accessible.
	 * <p>
	 * This method is like {@link Class#getDeclaredConstructor(Class[])}, but sets the
	 * accessible flag of the constructor to true before returning it.
	 * <p>
	 * @param c the class to search for the constructor
	 * @param parameterTypes the parameter types of the constructor to search for
	 * @return the found constructor with accessible set to true
	 * @throws NoSuchMethodException if no such constructor exists
	 */
	public static <T> Constructor<T> getConstructor(@NotNull Class<T> c, Class<?>... parameterTypes) throws NoSuchMethodException {
		return setAccessible(c.getDeclaredConstructor(parameterTypes));
	}

	/**
	 * Retrieves the value of a static field with the specified name from the given class.
	 * <p>
	 * This method attempts to access a static field in the specified class by name,
	 * setting the field to be accessible if necessary. If successful, it returns the
	 * value of the field. If any exception occurs during this process, it returns {@code null}.
	 * <p>
	 * @param c the class to search for the static field
	 * @param fName the name of the static field to retrieve
	 * @return the value of the static field, or {@code null} if the field cannot be accessed
	 */
	private static @Nullable Object reflectiveStaticFieldAccess(Class<?> c, String fName) {
		try {
			Field f = c.getDeclaredField(fName);
			f.setAccessible(true);
			return f.get(null);
		} catch (Exception e) {
			return null;
		}
	}


	/**
	 * Handles a reflection-related exception by printing out a helpful message
	 * about the exception and, if given, an exception that occurred while setting
	 * up reflection.
	 *
	 * @param t the exception to handle
	 * @param initError the exception that occurred while setting up reflection, or
	 *        null if no such exception occurred
	 */
	public static void handleReflectionDebug(@NotNull Throwable t, Throwable initError) {
		System.err.println("** COMMONS CORE REFLECTION exception: " + t.getClass() + ": " + (t.getMessage() == null ? "(no message)" : t.getMessage()));
		t.printStackTrace(System.err);
		if (initError != null) {
			System.err.println("*** ADDITIONALLY, exception occurred setting up reflection: ");
			initError.printStackTrace(System.err);
		}
	}

	public static Object invoke(Method m, Object receiver, Object... args) throws IllegalAccessException, InvocationTargetException {
		return invoke(null, m, receiver, args);
	}

	/**
	 * Invokes the specified method on the given receiver with the provided arguments.
	 * <p>
	 * This method attempts to call the specified method using reflection. If any
	 * exceptions are thrown during the invocation, they are caught and handled
	 * by logging the relevant debug information, and then re-thrown.
	 * <p>
	 * @param initError the optional exception that occurred while setting up reflection
	 * @param m the method to invoke
	 * @param receiver the object on which to invoke the method
	 * @param args the arguments to pass to the method
	 * @return the result of the method invocation
	 * @throws IllegalAccessException if the method is not accessible
	 * @throws InvocationTargetException if the underlying method throws an exception
	 */
	public static Object invoke(Throwable initError, Method m, Object receiver, Object... args) throws IllegalAccessException, InvocationTargetException {
		try {
			return m.invoke(receiver, args);
		} catch (IllegalAccessException e) {
			handleReflectionDebug(e, initError);
			throw e;
		} catch (RuntimeException e) {
			handleReflectionDebug(e, initError);
			throw e;
		} catch (Error e) {
			handleReflectionDebug(e, initError);
			throw e;
		}
	}

	public static Object invokeSneaky(Method m, Object receiver, Object... args) {
		return invokeSneaky(null, m, receiver, args);
	}

	/**
	 * Invokes the specified method on the given receiver with the provided arguments.
	 * <p>
	 * This method attempts to call the specified method using reflection. If any
	 * exceptions are thrown during the invocation, they are caught and handled
	 * by logging the relevant debug information, and then re-thrown. If the
	 * exception is a {@code NoClassDefFoundError} or a {@code NullPointerException},
	 * the exception is ignored and {@code null} is returned.
	 * <p>
	 * @param initError the optional exception that occurred while setting up reflection
	 * @param m the method to invoke
	 * @param receiver the object on which to invoke the method
	 * @param args the arguments to pass to the method
	 * @return the result of the method invocation, or {@code null} if a
	 *         {@code NoClassDefFoundError} or a {@code NullPointerException} was thrown
	 * @throws IllegalAccessException if the method is not accessible
	 * @throws InvocationTargetException if the underlying method throws an exception
	 */
	public static @Nullable Object invokeSneaky(Throwable initError, Method m, Object receiver, Object... args) {
		try {
			return m.invoke(receiver, args);
		} catch (NoClassDefFoundError e) {
			handleReflectionDebug(e, initError);
			//ignore, we don't have access to the correct ECJ classes, so can't possibly
			//do anything useful here.
			return null;
		} catch (NullPointerException e) {
			handleReflectionDebug(e, initError);
			//ignore, we don't have access to the correct ECJ classes, so can't possibly
			//do anything useful here.
			return null;
		} catch (IllegalAccessException e) {
			handleReflectionDebug(e, initError);
			throw sneakyThrow(e);
		} catch (InvocationTargetException e) {
			throw sneakyThrow(e.getCause());
		} catch (RuntimeException e) {
			handleReflectionDebug(e, initError);
			throw e;
		} catch (Error e) {
			handleReflectionDebug(e, initError);
			throw e;
		}
	}

	/**
	 * Invokes the specified constructor with the given arguments.
	 * <p>
	 * This method attempts to call the specified constructor using reflection. If any
	 * exceptions are thrown during the invocation, they are caught and handled
	 * by logging the relevant debug information, and then re-thrown.
	 * <p>
	 * @param c the constructor to invoke
	 * @param args the arguments to pass to the constructor
	 * @return the result of the constructor invocation
	 * @throws IllegalAccessException if the constructor is not accessible
	 * @throws InvocationTargetException if the underlying constructor throws an exception
	 * @throws InstantiationException if the underlying constructor throws an exception
	 */
	public static <T> @NotNull T newInstance(Constructor<T> c, Object... args) throws IllegalAccessException, InvocationTargetException, InstantiationException {
		return newInstance(null, c, args);
	}

	/**
	 * Invokes the specified constructor with the given arguments.
	 * <p>
	 * This method attempts to call the specified constructor using reflection. If any
	 * exceptions are thrown during the invocation, they are caught and handled
	 * by logging the relevant debug information, and then re-thrown.
	 * <p>
	 * @param initError the optional exception that occurred while setting up reflection
	 * @param c the constructor to invoke
	 * @param args the arguments to pass to the constructor
	 * @return the result of the constructor invocation
	 * @throws IllegalAccessException if the constructor is not accessible
	 * @throws InvocationTargetException if the underlying constructor throws an exception
	 * @throws InstantiationException if the underlying constructor throws an exception
	 */
	public static <T> @NotNull T newInstance(Throwable initError, Constructor<T> c, Object... args) throws IllegalAccessException, InvocationTargetException, InstantiationException {
		try {
			return c.newInstance(args);
		} catch (IllegalAccessException e) {
			handleReflectionDebug(e, initError);
			throw e;
		} catch (InstantiationException e) {
			handleReflectionDebug(e, initError);
			throw e;
		} catch (RuntimeException e) {
			handleReflectionDebug(e, initError);
			throw e;
		} catch (Error e) {
			handleReflectionDebug(e, initError);
			throw e;
		}
	}

	/**
	 * Invokes the specified constructor with the given arguments.
	 * <p>
	 * This method attempts to call the specified constructor using reflection. If any
	 * exceptions are thrown during the invocation, they are caught and handled
	 * by logging the relevant debug information, and then re-thrown. If the
	 * exception is a {@code NoClassDefFoundError} or a {@code NullPointerException},
	 * the exception is ignored and {@code null} is returned.
	 * <p>
	 * @param c the constructor to invoke
	 * @param args the arguments to pass to the constructor
	 * @return the result of the constructor invocation, or {@code null} if a
	 *         {@code NoClassDefFoundError} or a {@code NullPointerException} was thrown
	 */
	public static <T> T newInstanceSneaky(Constructor<T> c, Object... args) {
		return newInstanceSneaky(null, c, args);
	}

	/**
	 * Invokes the specified constructor with the given arguments.
	 * <p>
	 * This method attempts to call the specified constructor using reflection. If any
	 * exceptions are thrown during the invocation, they are caught and handled
	 * by logging the relevant debug information, and then re-thrown. If the
	 * exception is a {@code NoClassDefFoundError} or a {@code NullPointerException},
	 * the exception is ignored and {@code null} is returned.
	 * <p>
	 * @param initError the optional exception that occurred while setting up reflection
	 * @param c the constructor to invoke
	 * @param args the arguments to pass to the constructor
	 * @return the result of the constructor invocation, or {@code null} if a
	 *         {@code NoClassDefFoundError} or a {@code NullPointerException} was thrown
	 */
	public static <T> @Nullable T newInstanceSneaky(Throwable initError, Constructor<T> c, Object... args) {
		try {
			return c.newInstance(args);
		} catch (NoClassDefFoundError e) {
			handleReflectionDebug(e, initError);
			//ignore, we don't have access to the correct ECJ classes, so can't possibly
			//do anything useful here.
			return null;
		} catch (NullPointerException e) {
			handleReflectionDebug(e, initError);
			//ignore, we don't have access to the correct ECJ classes, so can't possibly
			//do anything useful here.
			return null;
		} catch (IllegalAccessException e) {
			handleReflectionDebug(e, initError);
			throw sneakyThrow(e);
		} catch (InstantiationException e) {
			handleReflectionDebug(e, initError);
			throw sneakyThrow(e);
		} catch (InvocationTargetException e) {
			throw sneakyThrow(e.getCause());
		} catch (RuntimeException e) {
			handleReflectionDebug(e, initError);
			throw e;
		} catch (Error e) {
			handleReflectionDebug(e, initError);
			throw e;
		}
	}

	/**
	 * Retrieves the value of the given field from the given receiver.
	 * <p>
	 * This method attempts to call the specified field using reflection. If any
	 * exceptions are thrown during the invocation, they are caught and handled
	 * by logging the relevant debug information, and then re-thrown.
	 * <p>
	 * @param f the field to retrieve
	 * @param receiver the object from which to retrieve the field
	 * @return the value of the field
	 * @throws IllegalAccessException if the field is not accessible
	 */
	public static <T> T get(Field f, Object receiver) throws IllegalAccessException {
		try {
			return (T) f.get(receiver);
		} catch (IllegalAccessException e) {
			handleReflectionDebug(e, null);
			throw e;
		} catch (RuntimeException e) {
			handleReflectionDebug(e, null);
			throw e;
		} catch (Error e) {
			handleReflectionDebug(e, null);
			throw e;
		}
	}

	/**
	 * Set the value of the field to {@code newValue}.
	 * @param f the field to set
	 * @param receiver the object in which to set the field
	 * @param newValue the value to set
	 * @throws IllegalAccessException if the field is not accessible
	 */
	public static void set(Field f, Object receiver, Object newValue) throws IllegalAccessException {
		try {
			f.set(receiver, newValue);
		} catch (IllegalAccessException e) {
			handleReflectionDebug(e, null);
			throw e;
		} catch (RuntimeException e) {
			handleReflectionDebug(e, null);
			throw e;
		} catch (Error e) {
			handleReflectionDebug(e, null);
			throw e;
		}
	}

	/**
	 * Reports a reflection-related problem by printing out a message to the error stream.
	 * <p>
	 * This method logs a specified message indicating a reflection issue and, if provided,
	 * also logs an exception that occurred during the setup of reflection.
	 * <p>
	 * @param initError the exception that occurred while setting up reflection, or
	 *        null if no such exception occurred
	 * @param msg the message describing the reflection problem
	 */
	public static void reportReflectionProblem(Throwable initError, String msg) {
		System.err.println("** COMMONS CORE REFLECTION issue: " + msg);
		if (initError != null) {
			System.err.println("*** ADDITIONALLY, exception occurred setting up reflection: ");
			initError.printStackTrace(System.err);
		}
	}

	/**
	 * Wraps a {@code Throwable} in a {@code RuntimeException}, but only if it isn't
	 * already a {@code RuntimeException}. This is useful for wrapping exceptions
	 * that you want to propagate up the call stack without having to declare them
	 * in the method signature.
	 * <p>
	 * If the given {@code Throwable} is already a {@code RuntimeException}, this
	 * method simply returns it as is. Otherwise, it wraps it in a
	 * {@code RuntimeException} and returns that.
	 * <p>
	 * @param t the exception to wrap
	 * @return the wrapped exception
	 * @throws NullPointerException if the given exception is null
	 */
	public static RuntimeException sneakyThrow(Throwable t) {
		if (t == null) throw new NullPointerException("t");
		return Permit.<RuntimeException>sneakyThrow0(t);
	}

	/**
	 * Throws a given throwable as an unchecked exception.
	 * <p>
	 * This method uses type erasure to throw any checked exception without
	 * declaring it in the method signature. It effectively bypasses the
	 * Java checked exception mechanism by casting the throwable to a generic
	 * type and immediately throwing it.
	 * <p>
	 * Note that this method should be used with caution as it breaks
	 * the contract of checked exceptions and may lead to unexpected behavior
	 * if not properly handled.
	 * <p>
	 * @param t the throwable to be thrown
	 * @throws T the throwable to be thrown
	 * @param <T> a generic type that extends Throwable
	 */
	@SuppressWarnings("unchecked")
	private static <T extends Throwable> T sneakyThrow0(Throwable t) throws T {
		throw (T)t;
	}
}