package host.anzo.commons.processors;

import com.google.auto.service.AutoService;
import com.sun.source.tree.CompilationUnitTree;
import com.sun.source.util.TreePath;
import com.sun.tools.javac.code.Flags;
import com.sun.tools.javac.code.TypeTag;
import com.sun.tools.javac.tree.JCTree;
import com.sun.tools.javac.tree.JCTree.*;
import com.sun.tools.javac.util.List;
import com.sun.tools.javac.util.Name;
import host.anzo.commons.processors.utils.Statics;
import org.jetbrains.annotations.NotNull;

import javax.annotation.processing.Processor;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.annotation.processing.SupportedOptions;
import javax.lang.model.element.*;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * @author ANZO
 */
@AutoService(Processor.class)
@SupportedOptions({"CommonsCoreVerbose"})
@SupportedAnnotationTypes(Statics.EXTENDED_ENUM_ANNOTATION_PATH)
public class ExtendedEnumProcessor extends CommonProcessor {
	/**
	 * Process the given annotations and round environment.
	 *
	 * @param annotations The annotation types requested to be processed.
	 * @param roundEnv    The environment for information about the current and previous round.
	 * @return {@code true} if the annotation processing was successful, {@code false} otherwise.
	 * @see javax.annotation.processing.AbstractProcessor#process(Set, RoundEnvironment)
	 */
	@Override
	public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
		if (this.maker == null || this.trees == null || this.names == null) {
			logError("Processor essentials not initialized. Skipping processing.");
			return false;
		}

		if (roundEnv.processingOver() || annotations.isEmpty()) {
			return false;
		}

		final TypeElement extendedEnumAnnotation = processingEnv.getElementUtils().getTypeElement(Statics.EXTENDED_ENUM_ANNOTATION_PATH);
		if (extendedEnumAnnotation == null) {
			logError("Could not resolve " + Statics.EXTENDED_ENUM_ANNOTATION_PATH + " annotation.");
			return false;
		}

		isVerbose = processingEnv.getOptions().containsKey("CommonsCoreVerbose");

		for (Element classElement : roundEnv.getElementsAnnotatedWith(extendedEnumAnnotation)) {
			if (classElement.getKind() == ElementKind.ENUM) {
				final TreePath path = trees.getPath(classElement);
				if (path == null) {
					logWarn("Could not get TreePath for element: " + classElement.getSimpleName());
					continue;
				}

				if (isVerbose)
					log("Processing " + classElement.getSimpleName());

				final CompilationUnitTree cuTree = path.getCompilationUnit();
				if (cuTree == null) {
					logWarn("Could not get CompilationUnitTree for element: " + classElement.getSimpleName());
					continue;
				}

				final JCCompilationUnit compilationUnit = (JCCompilationUnit) cuTree;
				final JCClassDecl classDecl = (JCClassDecl) path.getLeaf();
				final int currentPos = classDecl.pos;

				// cacheList
				addCacheListField(classDecl, currentPos);
				addGetValueByOrdinalMethod(classDecl, currentPos);
				addGetValueByOrdinalInternalMethod(classDecl, currentPos);
				addGetValuesMethod(classDecl, currentPos);
				addGetValuesInternalMethod(classDecl, currentPos);

				java.util.List<JCStatement> collectedPutStatements = new ArrayList<>();

				// cacheMap's
				addNameCacheAndCollectPuts(classDecl, collectedPutStatements, currentPos);
				processCacheKeysAndCollectPuts(classElement, classDecl, collectedPutStatements, currentPos);

				// If there are 'put' statements, create the loop and add it to the static initializer
				if (!collectedPutStatements.isEmpty()) {
					final JCIdent enumType = maker.Ident(classDecl.name);
					final Name loopVarName = names.fromString("enumConstant");
					final JCVariableDecl loopVar = maker.at(currentPos).VarDef(
							maker.Modifiers(0),
							loopVarName,
							enumType,
							null
					);

					final JCExpression iterable = maker.at(currentPos).Apply(
							List.nil(),
							maker.at(currentPos).Select(enumType, names.fromString("values")),
							List.nil()
					);

					// Create loop body with all collected statements at once
					final JCBlock loopBody = maker.at(currentPos).Block(0, List.from(collectedPutStatements));
					final JCEnhancedForLoop foreachLoop = maker.at(currentPos).ForeachLoop(loopVar, iterable, loopBody);
					addToStaticInitializer(classDecl, foreachLoop);
				}

				if (isVerbose) {
					printCompilationUnitAst(compilationUnit);
				}
			} else {
				logWarn("@" + extendedEnumAnnotation.getSimpleName() + " applied to non-enum: " + classElement.getSimpleName());
			}
		}
		return false;
	}

	/**
	 * Adds support for caching enum constants by their name and collects the "put" statements for the static initializer.
	 * <p>
	 * This method checks if a cache map field and corresponding getter methods already exist in the given class declaration.
	 * If not, it adds a new map field for caching enum constants by their name (using a String key) and defines getter methods
	 * for accessing these cached constants. It also creates a "put" statement for populating the cache map and adds it to
	 * the collection of statements to be included in the static initializer.
	 *
	 * @param classDecl                 The {@link JCClassDecl} AST node of the enum to modify.
	 * @param collectedPutStatements    A list to collect "put" statements for populating the cache map.
	 * @param currentPos                The maker position for AST node creation.
	 */
	private void addNameCacheAndCollectPuts(JCClassDecl classDecl, java.util.List<JCStatement> collectedPutStatements, int currentPos) {
		final String mapFieldName = "cacheMap";
		final JCExpression keyType = createQualifiedName(currentPos,"java.lang.String");

		if (!fieldExists(classDecl, mapFieldName)) {
			addCacheMapField(classDecl, keyType, mapFieldName, currentPos);
		}

		final String getterName = "getValueOf";
		if (!methodExists(classDecl, getterName, 1)) {
			addCacheMapValueGetterMethod(classDecl, "name", keyType, getterName, mapFieldName, false, currentPos);
			addCacheMapValueGetterInternalMethod(classDecl, "name", keyType, getterName, false, currentPos);
		}
		if (!methodExists(classDecl, getterName, 2)) {
			addCacheMapValueGetterMethod(classDecl, "name", keyType, getterName, mapFieldName, true, currentPos);
			addCacheMapValueGetterInternalMethod(classDecl, "name", keyType, getterName, true, currentPos);
		}

		final JCStatement putStmt = createCacheMapPutStatement(
				names.fromString("enumConstant"),
				"name",
				mapFieldName,
				true,
				currentPos
		);
		collectedPutStatements.add(putStmt);
	}


	/**
	 * Adds support for caching enum constants using a specified field or method as the cache key
	 * and collects the "put" statements for the static initializer block.
	 * <p>
	 * This method checks if a cache map field and corresponding getter methods already exist
	 * in the given class declaration. If not, it adds a new map field for caching enum constants
	 * using the specified key type and defines getter methods for accessing these cached constants.
	 * It also creates a "put" statement for populating the cache map and adds it to the collection
	 * of statements to be included in the static initializer.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 * @param accessorName The name of the field or method to use as the cache key.
	 * @param keyType The type of the key used in the cache map.
	 * @param mapFieldName The name of the cache map field.
	 * @param getterMethodName The name of the getter method for the cache map.
	 * @param isMethod Indicates whether the accessorName refers to a method call.
	 * @param collectedPutStatements A list to collect "put" statements for populating the cache map.
	 * @param currentPos The maker position for AST node creation.
	 */
	private void addKeysCacheAndCollectPuts(JCClassDecl classDecl,
	                                        String accessorName,
	                                        JCExpression keyType,
	                                        String mapFieldName,
	                                        String getterMethodName,
	                                        boolean isMethod,
	                                        java.util.List<JCStatement> collectedPutStatements,
	                                        int currentPos) {
		if (!fieldExists(classDecl, mapFieldName)) {
			addCacheMapField(classDecl, keyType, mapFieldName, currentPos);
		}

		final JCStatement putStmt = createCacheMapPutStatement(
				names.fromString("enumConstant"),
				accessorName,
				mapFieldName,
				isMethod,
				currentPos
		);
		collectedPutStatements.add(putStmt);

		if (!methodExists(classDecl, getterMethodName, 1)) {
			addCacheMapValueGetterMethod(classDecl, accessorName, keyType,
					getterMethodName, mapFieldName, false, currentPos);
			addCacheMapValueGetterInternalMethod(classDecl, accessorName, keyType, getterMethodName, false, currentPos);
		}
		if (!methodExists(classDecl, getterMethodName, 2)) {
			addCacheMapValueGetterMethod(classDecl, accessorName, keyType,
					getterMethodName, mapFieldName, true, currentPos);
			addCacheMapValueGetterInternalMethod(classDecl, accessorName, keyType, getterMethodName, true, currentPos);
		}
	}

	/**
	 * Process the cache keys annotated on the given element.
	 * This method assumes that the given element is an enum.
	 *
	 * @param classElement The element (enum) to process cache keys for.
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 * @param collectedPutStatements A list of collected put statements.
	 * @param currentPos The maker position for AST node creation.
	 */
	private void processCacheKeysAndCollectPuts(Element classElement, JCClassDecl classDecl, java.util.List<JCStatement> collectedPutStatements, int currentPos) {
		final Set<String> cacheKeys = getCacheKeyFieldNames(classElement);
		for (String cacheKeyName : cacheKeys) {
			boolean isMethod = cacheKeyName.endsWith("()");
			String cleanName = isMethod ?
					cacheKeyName.substring(0, cacheKeyName.length() - 2) :
					cacheKeyName;

			JCExpression keyType;
			String mapFieldName = "cacheMapBy" + capitalize(cleanName);
			String getterMethodName = "getValueOf" + capitalize(cleanName);

			try {
				if (isMethod) {
					keyType = findMethodReturnType(classElement, cleanName, currentPos);
				} else {
					keyType = findFieldType(classDecl, cleanName);
				}

				if (keyType == null) {
					logError("In enum " + classDecl.name.toString() + ", " + (isMethod ? "method '" : "field '") + cleanName + "' not found for @ExtendedEnum cacheKey.");
					continue;
				}

				final JCExpression mapKeyActualType = getBoxedType(keyType, currentPos);
				addKeysCacheAndCollectPuts(
						classDecl,
						cleanName,
						mapKeyActualType,
						mapFieldName,
						getterMethodName,
						isMethod,
						collectedPutStatements,
						currentPos
				);

			} catch (Exception e) {
				logError("Error processing accessor '" + cacheKeyName + "' for enum " + classDecl.name.toString() + ": " + e.getMessage());
			}
		}
	}

	/**
	 * Retrieves the cache key field names from the given element, if the element is annotated with the extended enum annotation.
	 *
	 * @param element the element to retrieve cache key field names from
	 * @return a set of string representing the cache key field names
	 */
	private @NotNull Set<String> getCacheKeyFieldNames(@NotNull Element element) {
		final Set<String> keys = new HashSet<>();
		AnnotationMirror extendedEnumAnnotationMirror = null;

		for (AnnotationMirror am : element.getAnnotationMirrors()) {
			if (am.getAnnotationType().toString().equals(Statics.EXTENDED_ENUM_ANNOTATION_PATH)) {
				extendedEnumAnnotationMirror = am;
				break;
			}
		}

		if (extendedEnumAnnotationMirror != null) {
			final Map<? extends ExecutableElement, ? extends AnnotationValue> elementValues =
					processingEnv.getElementUtils().getElementValuesWithDefaults(extendedEnumAnnotationMirror);

			for (Map.Entry<? extends ExecutableElement, ? extends AnnotationValue> entry : elementValues.entrySet()) {
				if (entry.getKey().getSimpleName().toString().equals("cacheKeys")) {
					final Object value = entry.getValue().getValue();
					if (value instanceof java.util.List) {
						for (Object item : (java.util.List<?>) value) {
							if (item instanceof AnnotationValue) {
								final Object innerValue = ((AnnotationValue) item).getValue();
								if (innerValue instanceof String) {
									keys.add((String) innerValue);
								}
							}
						}
					}
					break;
				}
			}
		}
		return keys;
	}

	/**
	 * Adds a {@code private static final List<E> cacheList} field to the enum's AST.
	 * The field is initialized with {@code java.util.List.of(values())}.
	 * If a field named "cacheList" already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 * @param currentPos The maker position for AST node creation.
	 */
	private void addCacheListField(JCClassDecl classDecl, int currentPos) {
		final String fieldName = "cacheList";
		if (fieldExists(classDecl, fieldName))
			return;

		final JCExpression listType = maker.at(currentPos).TypeApply(
				createQualifiedName(currentPos, "java.util.List"),
				List.of(maker.at(currentPos).Ident(classDecl.name))
		);

		final JCExpression valuesCall = maker.at(currentPos).Apply(
				List.nil(),
				maker.at(currentPos).Select(
						maker.at(currentPos).Ident(classDecl.name),
						names.fromString("values")),
				List.nil()
		);

		final JCExpression init = maker.at(currentPos).Apply(
				List.nil(),
				maker.at(currentPos).Select(
						createQualifiedName(currentPos, "java.util.List"),
						names.fromString("of")),
				List.of(valuesCall)
		);

		final JCVariableDecl cacheListField = maker.at(currentPos).VarDef(
				maker.Modifiers(Flags.PRIVATE | Flags.STATIC | Flags.FINAL),
				names.fromString(fieldName),
				listType,
				init
		);
		classDecl.defs = classDecl.defs.append(cacheListField);
	}

	/**
	 * Adds a {@code public static E getValue(int ordinal)} method to the enum's AST.
	 * The method body returns {@code cacheList.get(ordinal)}.
	 * If a method named "getValue" already exists with one int parameter, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 * @param currentPos The maker position for AST node creation.
	 */
	private void addGetValueByOrdinalMethod(JCClassDecl classDecl, int currentPos) {
		final String methodName = "getValue";
		if (methodExists(classDecl, methodName, 1))
			return;

		final JCMethodDecl method = maker.at(currentPos).MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				maker.at(currentPos).Ident(classDecl.name),
				List.nil(),
				List.of(maker.at(currentPos).VarDef(
						maker.Modifiers(Flags.PARAMETER),
						names.fromString("ordinal"),
						maker.at(currentPos).TypeIdent(TypeTag.INT),
						null
				)),
				List.nil(),
				maker.at(currentPos).Block(0, List.of(
						maker.at(currentPos).Return(
								maker.at(currentPos).Apply(
										List.nil(),
										maker.at(currentPos).Select(
												maker.at(currentPos).Ident(names.fromString("cacheList")),
												names.fromString("get")
										),
										List.of(maker.at(currentPos).Ident(names.fromString("ordinal")))
								)
						)
				)),
				null
		);
		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds an internal instance method {@code getValueInternal(int ordinal)} to the enum's AST.
	 * This method calls the public static {@code getValue(int ordinal)} method and returns its result.
	 * If a method named "getValueInternal" already exists with one int parameter, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 * @param currentPos The maker position for AST node creation.
	 */
	private void addGetValueByOrdinalInternalMethod(JCClassDecl classDecl, int currentPos) {
		final String staticMethodName = "getValue";
		final String internalMethodName = staticMethodName + "Internal";
		final String paramName = "ordinal";

		if (methodExists(classDecl, internalMethodName, 1))
			return;

		JCVariableDecl ordinalParam = maker.at(currentPos).VarDef(
				maker.Modifiers(Flags.PARAMETER),
				names.fromString(paramName),
				maker.TypeIdent(TypeTag.INT),
				null
		);

		JCExpression staticMethodCall = maker.at(currentPos).Apply(
				List.nil(),
				maker.Ident(names.fromString(staticMethodName)),
				List.of(maker.Ident(names.fromString(paramName)))
		);

		JCMethodDecl internalMethod = maker.at(currentPos).MethodDef(
				maker.Modifiers(Flags.PUBLIC),
				names.fromString(internalMethodName),
				maker.Ident(classDecl.name),
				List.nil(), List.of(ordinalParam), List.nil(),
				maker.Block(0, List.of(maker.Return(staticMethodCall))), null
		);
		classDecl.defs = classDecl.defs.append(internalMethod);
	}

	/**
	 * Adds a {@code public static List<E> getValues()} method to the enum's AST.
	 * The method body returns the {@code cacheList} field.
	 * If a method named "getValues" already exists with no parameters, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 * @param currentPos The maker position for AST node creation.
	 */
	private void addGetValuesMethod(JCClassDecl classDecl, int currentPos) {
		final String methodName = "getValues";
		if (methodExists(classDecl, methodName, 0))
			return;

		final JCExpression returnType = maker.at(currentPos).TypeApply(
				createQualifiedName(currentPos, "java.util.List"),
				List.of(maker.at(currentPos).Ident(classDecl.name))
		);

		final JCMethodDecl method = maker.at(currentPos).MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				returnType,
				List.nil(),
				List.nil(),
				List.nil(),
				maker.at(currentPos).Block(0, List.of(
						maker.at(currentPos).Return(maker.at(currentPos).Ident(names.fromString("cacheList")))
				)),
				null
		);
		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds an internal instance method {@code getValuesInternal()} to the enum's AST.
	 * This method calls the public static {@code getValues()} method and returns its result.
	 * If a method named "getValuesInternal" already exists with no parameters, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 * @param currentPos The maker position for AST node creation.
	 */
	private void addGetValuesInternalMethod(JCClassDecl classDecl, int currentPos) {
		final String staticMethodName = "getValues";
		final String internalMethodName = staticMethodName + "Internal";

		if (methodExists(classDecl, internalMethodName, 0))
			return;

		final JCExpression returnType = maker.at(currentPos).TypeApply(
				createQualifiedName(currentPos, "java.util.List"),
				List.of(maker.Ident(classDecl.name))
		);

		JCExpression staticMethodCall = maker.at(currentPos).Apply(
				List.nil(),
				maker.Ident(names.fromString(staticMethodName)),
				List.nil()
		);

		JCMethodDecl internalMethod = maker.at(currentPos).MethodDef(
				maker.Modifiers(Flags.PUBLIC), // Instance method
				names.fromString(internalMethodName), returnType,
				List.nil(), List.nil(), List.nil(),
				maker.Block(0, List.of(maker.Return(staticMethodCall))), null
		);
		classDecl.defs = classDecl.defs.append(internalMethod);
	}


	/**
	 * Adds a {@code private static final Map<K, E> mapFieldName} field to the enum's AST.
	 * The field is initialized with a new {@code HashMap} instance.
	 * If a field with the specified mapFieldName already exists, this method does nothing.
	 *
	 * @param classDecl The {@link JCClassDecl} AST node of the enum to modify.
	 * @param keyType The type of the keys used in the map.
	 * @param mapFieldName The name of the map field to be added.
	 * @param currentPos The maker position for AST node creation.
	 */
	private void addCacheMapField(@NotNull JCClassDecl classDecl, JCExpression keyType, String mapFieldName, int currentPos) {
		final JCIdent enumIdent = maker.at(currentPos).Ident(classDecl.name);
		final JCExpression mapGenericType = maker.at(currentPos).TypeApply(
				createQualifiedName(currentPos, "java.util.Map"),
				List.of(keyType, enumIdent)
		);

		final JCExpression newHashMapInstance = maker.at(currentPos).NewClass(
				null,
				List.nil(),
				maker.at(currentPos).TypeApply(createQualifiedName(currentPos, "java.util.HashMap"), List.nil()),
				List.nil(),
				null
		);

		final JCVariableDecl cacheMapVDef = maker.at(currentPos).VarDef(
				maker.Modifiers(Flags.PRIVATE | Flags.STATIC | Flags.FINAL),
				names.fromString(mapFieldName),
				mapGenericType,
				newHashMapInstance
		);
		classDecl.defs = classDecl.defs.append(cacheMapVDef);
	}

	/**
	 * Creates a single Javac "put" statement for populating a cache map within a loop.
	 * (e.g., mapFieldName.put(loopVarAccess.originalFieldNameOrMethod(), loopVarAccess);)
	 *
	 * @param loopVarNameIdentifier The Javac Name of the loop variable (e.g., "enumConstant").
	 * @param originalAccessorName  The name of the enum field or method to use as the key.
	 * @param mapFieldName          The name of the map to put into.
	 * @param isAccessorMethod      Whether the originalAccessorName refers to a method call.
	 * @param currentPos            The maker position for AST node creation.
	 * @return A JCStatement representing the "put" call.
	 */
	private JCStatement createCacheMapPutStatement(
			com.sun.tools.javac.util.Name loopVarNameIdentifier,
			String originalAccessorName,
			String mapFieldName,
			boolean isAccessorMethod,
			int currentPos) {

		if (isVerbose) {
			log("Creating put statement: " + mapFieldName + ".put(" + loopVarNameIdentifier.toString() + "." + originalAccessorName + (isAccessorMethod ? "()" : "") + ", " + loopVarNameIdentifier + ")");
		}

		final JCIdent loopVarAccess = maker.at(currentPos).Ident(loopVarNameIdentifier);
		final Name accessorAstName = names.fromString(originalAccessorName);

		final JCExpression keyAccessExpression = isAccessorMethod ?
				maker.at(currentPos).Apply(List.nil(), maker.at(currentPos).Select(loopVarAccess, accessorAstName), List.nil()) :
				maker.at(currentPos).Select(loopVarAccess, accessorAstName);

		JCExpression containsKeyCall = maker.at(currentPos).Apply(
				List.nil(),
				maker.at(currentPos).Select(maker.at(currentPos).Ident(names.fromString(mapFieldName)), names.fromString("containsKey")),
				List.of(keyAccessExpression)
		);

		JCExpression putCall = maker.at(currentPos).Apply(
				List.nil(),
				maker.at(currentPos).Select(maker.at(currentPos).Ident(names.fromString(mapFieldName)), names.fromString("put")),
				List.of(keyAccessExpression, loopVarAccess)
		);

		return maker.at(currentPos).If(
				maker.at(currentPos).Unary(Tag.NOT, containsKeyCall),
				maker.at(currentPos).Exec(putCall),
				null
		);
	}

	/**
	 * Adds a public static getter method to the enum's AST that returns an enum value
	 * from a cache map based on the given key parameter name, type, and map field name.
	 * The method is named by the given methodName.
	 * If withDefault is true, the method has an additional defaultValue parameter and
	 * uses the getOrDefault method on the cache map, otherwise it uses the get method.
	 *
	 * @param classDecl        The JCClassDecl AST node of the enum to modify.
	 * @param keyParameterName The name of the key parameter.
	 * @param keyType          The type of the key parameter.
	 * @param methodName       The name of the getter method.
	 * @param mapFieldName     The name of the cache map field.
	 * @param withDefault      Whether to use the getOrDefault method instead of get.
	 * @param currentPos       The maker position for AST node creation.
	 */
	private void addCacheMapValueGetterMethod(@NotNull JCClassDecl classDecl, String keyParameterName,
	                                          JCExpression keyType,
	                                          String methodName,
	                                          String mapFieldName, boolean withDefault, int currentPos) {
		final JCIdent enumIdent = maker.at(currentPos).Ident(classDecl.name);

		List<JCVariableDecl> params = List.of(
				maker.at(currentPos).VarDef(
						maker.Modifiers(Flags.PARAMETER),
						names.fromString(keyParameterName),
						keyType,
						null
				)
		);

		String mapAccessMethodName = "get";
		List<JCExpression> mapAccessArgs = List.of(maker.at(currentPos).Ident(names.fromString(keyParameterName)));

		if (withDefault) {
			params = params.append(
					maker.at(currentPos).VarDef(
							maker.Modifiers(Flags.PARAMETER),
							names.fromString("defaultValue"),
							enumIdent,
							null
					)
			);
			mapAccessMethodName = "getOrDefault";
			mapAccessArgs = mapAccessArgs.append(maker.at(currentPos).Ident(names.fromString("defaultValue")));
		}

		final JCExpression mapAccess = maker.at(currentPos).Apply(
				List.nil(),
				maker.at(currentPos).Select(maker.at(currentPos).Ident(names.fromString(mapFieldName)),
						names.fromString(mapAccessMethodName)),
				mapAccessArgs
		);

		final JCMethodDecl method = maker.at(currentPos).MethodDef(
				maker.Modifiers(Flags.PUBLIC | Flags.STATIC),
				names.fromString(methodName),
				enumIdent,
				List.nil(),
				params,
				List.nil(),
				maker.at(currentPos).Block(0, List.of(maker.at(currentPos).Return(mapAccess))),
				null
		);
		classDecl.defs = classDecl.defs.append(method);
	}

	/**
	 * Adds an internal instance method {@code internalGetterMethodName} to the enum's AST.
	 * This method calls the public static {@code staticGetterMethodName} method and returns its result.
	 * If a method named "internalGetterMethodName" already exists with the same parameter types, this method does nothing.
	 *
	 * @param classDecl                   The {@link JCClassDecl} AST node of the enum to modify.
	 * @param keyParameterName            The name of the parameter for the key in the cache map.
	 * @param keyType                     The type of the key in the cache map.
	 * @param staticGetterMethodName      The name of the public static method to call.
	 * @param withDefault                 Whether to add a second parameter "defaultValue" to the internal method.
	 * @param currentPos                  The maker position for AST node creation.
	 */
	private void addCacheMapValueGetterInternalMethod(@NotNull JCClassDecl classDecl,
	                                                  String keyParameterName,
	                                                  JCExpression keyType,
	                                                  String staticGetterMethodName,
	                                                  boolean withDefault, int currentPos) {

		final String internalGetterMethodName = staticGetterMethodName + "Internal";
		final int paramCount = withDefault ? 2 : 1;

		if (methodExists(classDecl, internalGetterMethodName, paramCount)) {
			return;
		}

		final JCIdent enumIdent = maker.at(currentPos).Ident(classDecl.name);

		List<JCVariableDecl> params = List.of(
				maker.at(currentPos).VarDef(
						maker.Modifiers(Flags.PARAMETER),
						names.fromString(keyParameterName),
						keyType,
						null
				)
		);

		List<JCExpression> callArgs = List.of(maker.at(currentPos).Ident(names.fromString(keyParameterName)));

		if (withDefault) {
			final String defaultValueParamName = "defaultValue";
			params = params.append(
					maker.at(currentPos).VarDef(
							maker.Modifiers(Flags.PARAMETER),
							names.fromString(defaultValueParamName),
							enumIdent,
							null
					)
			);
			callArgs = callArgs.append(maker.at(currentPos).Ident(names.fromString(defaultValueParamName)));
		}

		JCExpression staticMethodCall = maker.at(currentPos).Apply(List.nil(), maker.Ident(names.fromString(staticGetterMethodName)), callArgs);

		JCMethodDecl internalMethod = maker.at(currentPos).MethodDef(
				maker.Modifiers(Flags.PUBLIC), names.fromString(internalGetterMethodName), enumIdent, List.nil(), params, List.nil(), maker.Block(0, List.of(maker.Return(staticMethodCall))), null);
		classDecl.defs = classDecl.defs.append(internalMethod);
	}
}