package host.anzo.commons.threading;

import de.mxro.metrics.jre.Metrics;
import delight.async.properties.PropertyNode;
import host.anzo.commons.emergency.metric.Metric;
import host.anzo.commons.emergency.metric.MetricGroupType;
import host.anzo.commons.emergency.metric.MetricResult;
import host.anzo.core.config.EmergencyConfig;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

@Slf4j
@Metric
public class RunnableWrapper implements Runnable {
    private static final PropertyNode RUNNABLE_WRAPPER_METRICS = Metrics.create();
    private static final PropertyNode RUNNABLE_WRAPPER_METRICS_VIRTUAL = Metrics.create();

	private final Runnable _r;
	private final boolean _v;
    private final String _n;

	public RunnableWrapper(final Runnable r) {
		this(r, false);
	}

	public RunnableWrapper(final Runnable r, final boolean virtual) {
        this(r, virtual, r.getClass().getSimpleName());
	}

    public RunnableWrapper(final Runnable r, final String name) {
        this(r, false, name);
    }

    public RunnableWrapper(final Runnable r, final boolean virtual, final String name) {
        _r = r;
        _v = virtual;
        _n = name != null ? name : r.getClass().getSimpleName();
    }

	@Override
	public void run() {
		if (_r != null) {
			try {
                final long start = System.nanoTime();
				_r.run();
                final long end = System.nanoTime();
                if (EmergencyConfig.ENABLE_METRICS) {
                    (_v ? RUNNABLE_WRAPPER_METRICS_VIRTUAL : RUNNABLE_WRAPPER_METRICS).record(Metrics.value(_n + "_time_ns", end - start));
                }
			} catch (final Throwable e) {
                if (EmergencyConfig.ENABLE_METRICS) {
                    (_v ? RUNNABLE_WRAPPER_METRICS_VIRTUAL : RUNNABLE_WRAPPER_METRICS).record(Metrics.happened(_n + "_error"));
                }
				log.error("Error while running RunnableWrapper:", e);
				throw new RuntimeException(e);
			}
		}
	}

    public static @NotNull MetricResult getMetric() {
        final MetricResult rw = new MetricResult();
        rw.setMetricGroupType(MetricGroupType.THREADPOOL);
        rw.setName("RunnableWrapper");
        rw.setData(RUNNABLE_WRAPPER_METRICS.render().get());
        return rw;
    }

    public static MetricResult getMetricVirtual() {
        final MetricResult rwv = new MetricResult();
        rwv.setMetricGroupType(MetricGroupType.THREADPOOL);
        rwv.setName("RunnableWrapperVirtual");
        rwv.setData(RUNNABLE_WRAPPER_METRICS_VIRTUAL.render().get());
        return rwv;
    }
}
