package host.anzo.core.service;

import host.anzo.commons.io.xml.XmlParser;
import host.anzo.core.startup.StartupComponent;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;

import java.util.*;
import java.util.regex.Pattern;

/***
 * @author ANZO
 * @since 7/20/2021
 */
@Slf4j
@StartupComponent("Service")
public class ObsceneFilterService {
    @Getter(lazy = true)
    private final static ObsceneFilterService instance = new ObsceneFilterService();

    private static final List<ObsceneData> obsceneList = new ArrayList<>();
    private static final List<String> badWordsCache = new ArrayList<>();

    private ObsceneFilterService() {
        try {
            try (XmlParser parser = XmlParser.fromResource("obsence_filter/obscene_filter_en.xml", getClass().getClassLoader())) {
                obsceneList.add(new ObsceneData(parser));
            }
            try (XmlParser parser = XmlParser.fromResource("obsence_filter/obscene_filter_ru.xml", getClass().getClassLoader())) {
                obsceneList.add(new ObsceneData(parser));
            }
            log.info("Loaded obscene filters for [{}] locales.", obsceneList.size());
        }
        catch (Exception e) {
            log.error("Error while ObsceneFilterService initializing", e);
        }
    }

    /***
     * Replace all obscene words in text with censor symbols
     * @param inputString input text
     * @return censored text
     */
    public String filterWord(@NotNull String inputString) {
        String outputString = inputString;
        for (String word : inputString.split(" ")) {
            for (ObsceneData obscene : obsceneList) {
                if (obscene.isApplicable(word)) {
                    if (isObsceneWord(word)) {
                        outputString = outputString.replace(word, StringUtils.repeat("*", word.length()));
                    }
                }
            }
        }
        return outputString;
    }

    /**
     * @param word checked word
     * @return {@code true} if word contains obscene content, {@code false} otherwise
     */
    public boolean isObsceneWord(String word) {
        for (ObsceneData obscene : obsceneList) {
            if (isObsceneWord(word, obscene)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @param word checked word
     * @param obscene obscene data
     * @return {@code true} if word contains obscene content
     */
    private boolean isObsceneWord(String word, ObsceneData obscene) {
        if (badWordsCache.contains(word)) {
            return true;
        }
        if (obscene.getPattern().matcher(word).find() && obscene.isObsceneWord(word)) {
            badWordsCache.add(word);
            return true;
        }
        return false;
    }

    public enum EObsceneLocaleType {
        EN,
        RU
    }

    public record ObsceneBadWordSet(Pattern include, List<Pattern> excludes) {
    }

    public static class ObsceneData {
        private @Getter final EObsceneLocaleType locale;
        private @Getter final Pattern pattern;
        private final Map<String, String> equivalents = new HashMap<>();
        private final List<ObsceneBadWordSet> badWordSet = new ArrayList<>();

        public ObsceneData(@NotNull XmlParser node) {
            locale = EObsceneLocaleType.valueOf(node.readString("locale"));
            pattern = Pattern.compile(node.readString("matcher"));
            for (XmlParser childParser : node.children()) {
                switch (childParser.name()) {
                    case "equivalents":
                        for (XmlParser eqParser : childParser.children("equivalent")) {
                            final String find = eqParser.optChild("find").content().trim();
                            final String replace = eqParser.optChild("replace").content().trim();
                            equivalents.put(find, replace);
                        }
                        break;
                    case "badWords":
                        for (XmlParser bwParser : childParser.children("badWordSet")) {
                            if (bwParser.children("include").isEmpty() && bwParser.children("exclude").isEmpty()) {
                                final ObsceneBadWordSet badWord = new ObsceneBadWordSet(Pattern.compile(bwParser.content().trim()), Collections.emptyList());
                                badWordSet.add(badWord);
                            }
                            else {
                                Pattern include = null;
                                final List<Pattern> excludes = new ArrayList<>();
                                for (XmlParser inExParser : bwParser.children()) {
                                    switch (inExParser.name()) {
                                        case "include":
                                            include = Pattern.compile(inExParser.content().trim());
                                            break;
                                        case "exclude":
                                            excludes.add(Pattern.compile(inExParser.content().trim()));
                                            break;
                                    }
                                }
                                if (include != null) {
                                    final ObsceneBadWordSet badWord = new ObsceneBadWordSet(include, excludes);
                                    badWordSet.add(badWord);
                                }
                            }
                        }
                        break;
                }
            }
        }

        public boolean isApplicable(String input) {
            return pattern.matcher(input).find();
        }

        /**
         * @param word checked word set
         * @return word set with replaced equivalents chars
         */
        private String replaceWithEquivalents(String word) {
            for (Map.Entry<String, String> entry : equivalents.entrySet()) {
                if (word.contains(entry.getKey())) {
                    word = word.replace(entry.getKey(), entry.getValue());
                }
            }
            return word;
        }

        /**
         * @param word word for check
         * @return {@code true} if word contains obscene content
         */
        public boolean isObsceneWord(String word) {
            word = replaceWithEquivalents(word);

            for (ObsceneBadWordSet bw : badWordSet) {
                if (bw.include().matcher(word).matches()) {
                    final List<Pattern> excludes = bw.excludes();
                    if (!excludes.isEmpty()) {
                        for (Pattern exclude : excludes) {
                            if (!exclude.matcher(word).matches()) {
                                return true;
                            }
                        }
                    }
                }
            }
            return false;
        }
    }
}