package host.anzo.core.service;

import host.anzo.classindex.ClassIndex;
import host.anzo.commons.socials.telegram.ATelegramBot;
import host.anzo.commons.socials.telegram.TelegramBot;
import host.anzo.commons.utils.ClassUtils;
import host.anzo.core.startup.IShutdownable;
import host.anzo.core.startup.Scheduled;
import host.anzo.core.startup.StartupComponent;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.telegram.telegrambots.longpolling.TelegramBotsLongPollingApplication;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;

/**
 * @author ANZO
 */
@Slf4j
@StartupComponent("Service")
public class TelegramService implements IShutdownable {
    @Getter(lazy = true)
    private static final TelegramService instance = new TelegramService();
    private static final TelegramBotsLongPollingApplication botsApplication = new TelegramBotsLongPollingApplication();
    private static final Map<Class<?>, ATelegramBot> registeredBots = new HashMap<>();
    private static final Set<Long> banned_chats = new TreeSet<>();

    public TelegramService() {
        try {
            for (Class<?> clazz : ClassIndex.getAnnotated(TelegramBot.class)) {
                try {
                    final ATelegramBot bot = (ATelegramBot) ClassUtils.singletonInstance(clazz);
                    if (bot.isEnabled()) {
                        registeredBots.put(clazz, bot);
                        botsApplication.registerBot(bot.getToken(), bot);
                        log.info("Registered [{}] telegram bot.", bot.getBotName());
                    }
                    else {
                        log.info("Skipped [{}] telegram bot because disabled by configuration.", bot.getBotName());
                    }
                }
                catch (Exception e) {
                    log.error("Error while bot registration with class=[{}]", clazz.getSimpleName(), e);
                }
            }
        }
        catch (Exception e) {
            log.error("Error while initialize Telegram bot session", e);
        }
    }

    public void banChatId(long chatId) {
        banned_chats.add(chatId);
        log.warn("Banned chat id: {}", chatId);
    }

    public boolean isChatBanned(long chatId) {
        return banned_chats.contains(chatId);
    }

    @SuppressWarnings("unused")
    @Scheduled(period = 10, timeUnit = TimeUnit.SECONDS, runAfterServerStart = true)
    public void tickTask() {
        banned_chats.clear();
        registeredBots.values().forEach(ATelegramBot::onTick);
    }

    @SuppressWarnings("unchecked")
    public <T extends ATelegramBot> T getBot(Class<T> botClass) {
        return (T)registeredBots.get(botClass);
    }

    @Override
    public void onShutdown() {
        registeredBots.values().forEach(ATelegramBot::onServerShutdown);
	    try {
		    botsApplication.close();
	    } catch (Exception e) {
		    log.error("Error while closing Telegram bot session", e);
	    }
    }
}