package host.anzo.core.webserver.model;

import com.fasterxml.jackson.core.JsonProcessingException;
import host.anzo.commons.utils.IpUtils;
import host.anzo.commons.utils.JsonUtils;
import host.anzo.commons.utils.VMUtils;
import host.anzo.core.config.WebServerConfig;
import host.anzo.core.webserver.WebService;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import spark.ModelAndView;
import spark.Request;
import spark.Response;

import java.io.IOException;
import java.util.Map;

import static spark.Spark.halt;
import static spark.Spark.modelAndView;

/**
 * @author ANZO
 * @since 08.06.2017
 */
@Slf4j(topic = "WebServer")
public abstract class AWebArea {
	public abstract void registerPaths();

	protected <T> T jsonToObject(Request request, Class<T> clazz) throws JsonProcessingException {
		return JsonUtils.jsonToObject(request, clazz);
	}

	@SuppressWarnings("SameParameterValue")
	protected <T> T jsonToObject(String body, Class<T> clazz) throws JsonProcessingException {
		return JsonUtils.jsonToObject(body, clazz);
	}

	protected String dataToJson(Response response, Object data) {
		return JsonUtils.dataToJson(response, data);
	}

	protected String dataToJson(Object data) {
		return JsonUtils.dataToJson(data);
	}

	protected String error(@NotNull Response response) {
		response.redirect("/ErrorPage");
		halt(); // Need halt to prevent pass request to target controller
		return null;
	}

	protected String error(@NotNull Response response, String title, String text) {
		response.redirect("/ErrorPage?title=" + title + "&text=" + text);
		halt(); // Need halt to prevent pass request to target controller
		return null;
	}

	protected String errorDisabled(@NotNull Response response) {
		return error(response, "Error", "Area disabled by Administrator");
	}

	protected String maintenance(@NotNull Response response) {
		response.redirect("/Maintenance");
		halt(); // Need halt to prevent pass request to target controller
		return null;
	}

	protected String render(Map<String, Object> model, String templatePath) {
		return render(modelAndView(model, templatePath));
	}

	protected String render(Object model, String templatePath) {
		return render(modelAndView(model, templatePath));
	}

	private String render(ModelAndView modelAndView) {
		String renderResult = WebService.getInstance().getTemplateEngine().render(modelAndView);
		if (VMUtils.DEBUG) {
			// Don't use minified scripts/stylesheets for debug mode
			renderResult = renderResult.replace(".min.css", ".css");
			renderResult = renderResult.replace(".min.js", ".js");
			// Remove context/select event blockers for debug mode
			renderResult = renderResult.replaceAll("(?i)ondragstart=\"return false;\"", "");
			renderResult = renderResult.replaceAll("(?i)onselectstart=\"return false;\"", "");
			renderResult = renderResult.replaceAll("(?i)oncontextmenu=\"return false\"", "");
		}
		return renderResult;
	}

	protected void redirectKeepAlive(@NotNull Response response, String location, int httpStatusCode) {
		response.raw().setStatus(httpStatusCode);
		response.raw().setHeader("Connection", "keep-alive");
		response.raw().setHeader("Cache-Control", "private");
		response.raw().setHeader("Location", location);
		try {
			response.raw().sendError(httpStatusCode);
		} catch (IOException e) {
			log.warn("Exception when trying to redirect permanently", e);
		}
	}

	/**
	 * @param request HTTP request
	 * @return {@code true} if request is passed firewall checks, {@code false} otherwise
	 */
	protected boolean isAllowedAddress(Request request) {
		return WebService.getInstance().isAllowedAddress(getRealIp(request));
	}

	/**
	 * @param request request object
	 * @return real client IP (in case if using reverse-proxify like a CloudFlare)
	 */
	protected String getRealIp(@NotNull Request request) {
		return IpUtils.getRealIp(request);
	}

	/**
	 * @param request request object
	 * @return {@code true} if specified request sent from one of administrator IP, {@code false} otherwise
	 */
	@SuppressWarnings("BooleanMethodIsAlwaysInverted")
	protected boolean isAdministratorIP(@NotNull Request request) {
		return WebServerConfig.ADMINISTRATOR_IP_ADDRESSES.contains(getRealIp(request));
	}
}