package host.anzo.core.service;

import host.anzo.commons.network.iprange.IP4Range;
import host.anzo.commons.network.iprange.IP6Range;
import host.anzo.core.startup.IReloadable;
import host.anzo.core.startup.Reloadable;
import host.anzo.core.startup.StartupComponent;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author Aristo
 */
@Slf4j
@Reloadable(name = "cloudflare", group = "service")
@StartupComponent("Service")
public class CloudflareService implements IReloadable {
    @Getter(lazy = true)
    private static final CloudflareService instance = new CloudflareService();

    private static final String CF_IP4_RANGES_URL = "https://www.cloudflare.com/ips-v4";
    private static final String CF_IP6_RANGES_URL = "https://www.cloudflare.com/ips-v6";

    public static final String CF_RAY = "CF-RAY";
    public static final String CF_CONNECTING_IP = "CF-Connecting-IP";
    public static final String CF_IPCOUNTRY = "CF-IPCountry";

    private final List<IP4Range> CLOUDFLARE_IP4_RANGES = new ArrayList<>();
    private final List<IP6Range> CLOUDFLARE_IP6_RANGES = new ArrayList<>();

    public CloudflareService() {
        load();
    }

    private void load() {
        try {
            // Загружаем IP4 адреса Cloudflare
            for (String ip : getCloudflareIPs(CF_IP4_RANGES_URL)) {
                CLOUDFLARE_IP4_RANGES.add(new IP4Range(ip));
            }

            // Загружаем IP4 адреса Cloudflare
            for (String ip : getCloudflareIPs(CF_IP6_RANGES_URL)) {
                CLOUDFLARE_IP6_RANGES.add(new IP6Range(ip));
            }
	        log.info("Loaded {} Cloudflare IP4 ranges", CLOUDFLARE_IP4_RANGES.size());
	        log.info("Loaded {} Cloudflare IP6 ranges", CLOUDFLARE_IP6_RANGES.size());

        } catch (IOException e) {
            log.error("Error loading Cloudflare IP ranges", e);
        }
    }

    private @NotNull List<String> getCloudflareIPs(String url_string) throws IOException {
        try {
            List<String> cloudflareIPs = new ArrayList<>();

            URL url = new URL(url_string);
            BufferedReader reader = new BufferedReader(new InputStreamReader(url.openStream()));
            String line;

            while ((line = reader.readLine()) != null) {
                cloudflareIPs.add(line.trim());
            }

            reader.close();
            return cloudflareIPs;
        } catch (Exception e) {
            return Collections.emptyList();
        }
    }

    /**
     * @param ip IP address to check
     * @return {@code true} if specified IP address is part of CloudFlare network, {@code false} otherwise
     */
    public boolean isCloudflareIP(String ip) {
	    try {
		    return isCloudflareIP(InetAddress.getByName(ip));
	    } catch (Exception e) {
		    return false;
	    }
    }

    /**
     * @param inetAddress InetAddress IP representation to check
     * @return {@code true} if specified InetAddress address is part of CloudFlare network, {@code false} otherwise
     */
    public boolean isCloudflareIP(InetAddress inetAddress) {
        if (inetAddress == null || inetAddress.isLoopbackAddress()) {
            return false;
        }

        for (IP4Range range : CLOUDFLARE_IP4_RANGES) {
            if (range.contains(inetAddress)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @return {@code true} if Cloudflare ranges is loaded, {@code false} otherwise
     */
    public boolean isLoaded() {
        return !CLOUDFLARE_IP4_RANGES.isEmpty();
    }

    @Override
    public void reload() {
        CLOUDFLARE_IP4_RANGES.clear();
        CLOUDFLARE_IP6_RANGES.clear();
        load();
    }
}