/*
 * Copyright © 2016 BDO-Emu authors. All rights reserved.
 * Viewing, editing, running and distribution of this software strongly prohibited.
 * Author: xTz, Anton Lasevich, Tibald
 */

package host.anzo.commons.xml;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.text.TextStringBuilder;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.Map;

/***
 * @author ANZO
 * @since 5/24/2022
 */
@Slf4j
public class XmlBuilder {
    private final String name;
    private final Object value;
    private final XmlBuilder parent;
    private final String comment;
    private final LinkedHashMap<String, String> attributes = new LinkedHashMap<>();
    private final LinkedList<XmlBuilder> childNodes = new LinkedList<>();

    public XmlBuilder(String rootName) {
        this.name = rootName;
        this.value = null;
        this.parent = null;
        this.comment = null;
    }

    protected XmlBuilder(String name, Object value, XmlBuilder parent, String comment) {
        this.name = name;
        this.value = value;
        this.parent = parent;
        this.comment = comment;
    }

    public XmlBuilder e(String name) {
        return e(name, null, null);
    }

    public XmlBuilder e(String name, String comment) {
        return e(name, null, comment);
    }

    public XmlBuilder e(String name, Object value, String comment) {
        final XmlBuilder node = new XmlBuilder(name, value, this, comment);
        this.childNodes.add(node);
        return node;
    }

    public XmlBuilder a(String name, @NotNull Object value) {
        this.attributes.put(name, value.toString());
        return this;
    }

    @Override
    public String toString() {
        return getString(0);
    }

    protected String getString(int tabCount) {
        final TextStringBuilder builder = new TextStringBuilder();

        if (StringUtils.isNotEmpty(this.comment)) {
            builder.append(StringUtils.repeat("\t", tabCount)).append("<!-- ").append(this.comment).appendln(" -->");
        }

        builder.append(StringUtils.repeat("\t", tabCount)).append("<").append(this.name);
        if (!this.attributes.isEmpty()) {
            for (Map.Entry<String, String> entry : attributes.entrySet()) {
                builder.append(" ").append(entry.getKey()).append("=\"").append(entry.getValue()).append("\"");
            }
        }
        if (this.value == null && this.childNodes.isEmpty()) {
            builder.appendln("/>");
        }
        else {
            builder.appendln(">");
            tabCount++;
            if (this.value != null) {
                builder.append(StringUtils.repeat("\t", tabCount)).appendln(value);
            }
            else if (!this.childNodes.isEmpty()) {
                for (XmlBuilder node : this.childNodes) {
                    builder.append(node.getString(tabCount));
                }
            }
            tabCount--;
            builder.append(StringUtils.repeat("\t", tabCount)).appendln("</" + this.name + ">");
        }
        return builder.toString();
    }

    @SuppressWarnings("unused")
    public void writeToFile(File file) {
        try {
            FileUtils.writeStringToFile(file, toString(), StandardCharsets.UTF_8);
        } catch (IOException e) {
            log.error("Error writing XmlBuilder string result to file=[{}]", file, e);
        }
    }

    @SuppressWarnings("unused")
    public void writeToFile(String filePath) {
        try {
            FileUtils.writeStringToFile(new File(filePath), toString(), StandardCharsets.UTF_8);
        } catch (IOException e) {
            log.error("Error writing XmlBuilder string result to file=[{}]", filePath, e);
        }
    }
}