package host.anzo.core.startup;

import host.anzo.commons.utils.ClassUtils;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import host.anzo.classindex.ClassIndex;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * @since 10.01.2017
 * @author ANZO
 */
@Slf4j
@StartupComponent("Service")
public class ReloadService {
	@Getter(lazy=true)
	private final static ReloadService instance = new ReloadService();

	private final Map<String, Map<String, Class<?>>> classTree = new HashMap<>();

	private ReloadService() {
		ClassIndex.getAnnotated(Reloadable.class).forEach(item -> {
			final Reloadable annotation = item.getAnnotation(Reloadable.class);
			if (annotation != null) {
				if (IReloadable.class.isAssignableFrom(item)) {
					final String name = annotation.name();
					final String group = annotation.group();
					classTree.computeIfAbsent(group, m -> new HashMap<>()).putIfAbsent(name, item);
				}
				else {
					throw new RuntimeException("Found marked with Reloadable annotation class, but without IReloadable implementing: " + item.getSimpleName());
				}
			}
		});
		log.info("Loaded {} reloadable instances.", classTree.values().stream().mapToLong(item -> item.values().size()).sum());
	}

	/**
	 * Call reload method of IReloadable instance
	 * @param clazz class of IReloadable instance
	 */
	private void callReload(Class<?> clazz) {
		final Object object = ClassUtils.singletonInstance(clazz);
		if (object != null) {
			if (object instanceof IReloadable) {
				((IReloadable)object).reload();
			}
		}
		else {
			log.error("Can't find singleton for class=[{}]", clazz.getSimpleName());
		}
	}

	/**
	 * Reload all instances with specified reloadable name
	 * @param name reloadable name
	 * @return reloaded instance count
	 */
	public int reloadByName(String name) {
		int count = 0;
		for (Map<String, Class<?>> classMap : classTree.values()) {
			for (Map.Entry<String, Class<?>> classEntry : classMap.entrySet()) {
				if (classEntry.getKey().equalsIgnoreCase(name)) {
					callReload(classEntry.getValue());
					count++;
				}
			}
		}
		return count;
	}

	/**
	 * Reload all instances with a specified reloadable group
	 * @param group reloadable group
	 * @return reloaded instance count
	 */
	public int reloadByGroup(String group) {
		int count = 0;
		if (classTree.containsKey(group)) {
			final Collection<Class<?>> classes = classTree.get(group).values();
			for (Class<?> clazz : classes) {
				callReload(clazz);
				count++;
			}
		}
		return count;
	}
}