package host.anzo.commons.socials.telegram;

import host.anzo.classindex.ClassIndex;
import host.anzo.core.startup.IShutdownable;
import host.anzo.core.startup.Scheduled;
import host.anzo.core.startup.StartupComponent;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.telegram.telegrambots.longpolling.TelegramBotsLongPollingApplication;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @author ANZO
 */
@Slf4j
@StartupComponent("Service")
public class TelegramService implements IShutdownable {
    @Getter(lazy = true)
    private static final TelegramService instance = new TelegramService();

    private static final TelegramBotsLongPollingApplication botsApplication = new TelegramBotsLongPollingApplication();

    private static final Map<Class<?>, ATelegramBot> registeredBots = new HashMap<>();

    public TelegramService() {
        try {
            for (Class<?> clazz : ClassIndex.getAnnotated(TelegramBot.class)) {
                try {
                    final ATelegramBot bot = (ATelegramBot)clazz.getDeclaredConstructor().newInstance();
                    if (bot.isEnabled()) {
                        registeredBots.put(clazz, bot);
                        botsApplication.registerBot(bot.getToken(), bot);
                        log.info("Registered [{}] telegram bot", bot.getBotName());
                    }
                }
                catch (Exception e) {
                    log.error("Error while bot registration with class=[{}]", clazz.getSimpleName());
                }
            }
        }
        catch (Exception e) {
            log.error("Error while initialize Telegram bot session", e);
        }
    }

    @SuppressWarnings("unused")
    @Scheduled(period = 10, timeUnit = TimeUnit.SECONDS, runAfterServerStart = true)
    public void tickTask() {
        registeredBots.values().forEach(ATelegramBot::onTick);
    }

    @SuppressWarnings("unchecked")
    public <T extends ATelegramBot> T getBot(Class<T> botClass) {
        return (T)registeredBots.get(botClass);
    }

    @Override
    public void onShutdown() {
        registeredBots.values().forEach(ATelegramBot::onServerShutdown);
    }
}