package host.anzo.core.service;

import de.mxro.metrics.jre.Metrics;
import delight.async.properties.PropertyNode;
import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.emergency.metric.IMetric;
import host.anzo.commons.emergency.metric.MetricGroupType;
import host.anzo.commons.emergency.metric.MetricResult;
import host.anzo.core.config.EmergencyConfig;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.text.TextStringBuilder;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ForkJoinWorkerThread;

/**
 * @author ANZO
 */
@Slf4j
@StartupComponent("Service")
public class ForkJoinPoolService implements IMetric {
    @Getter(lazy = true)
    private static final ForkJoinPoolService instance = new ForkJoinPoolService();

    private final static PropertyNode packetSubmitsMetric = Metrics.create();
    private final static PropertyNode foreachMetric = Metrics.create();

    private final ForkJoinPool packetPool;
    private final ForkJoinPool foreachPool;

    protected ForkJoinPoolService() {
        this.packetPool = new ForkJoinPool(Math.max(1, Runtime.getRuntime().availableProcessors()), pool -> {
            final ForkJoinWorkerThread worker = ForkJoinPool.defaultForkJoinWorkerThreadFactory.newThread(pool);
            worker.setName("PacketPool-" + worker.getPoolIndex());
            return worker;
        }, null, true);
        this.foreachPool = new ForkJoinPool(Math.max(1, Runtime.getRuntime().availableProcessors()), pool -> {
            final ForkJoinWorkerThread worker = ForkJoinPool.defaultForkJoinWorkerThreadFactory.newThread(pool);
            worker.setName("ForeachPool-" + worker.getPoolIndex());
            return worker;
        }, null, true);
    }

    public void sendPackets(String name, Runnable runnable) {
        if (EmergencyConfig.ENABLE_METRICS) {
            packetSubmitsMetric.record(Metrics.happened(name));
        }
        try {
            packetPool.submit(runnable).get();
        } catch (Exception e) {
            log.error("Error while sendPackets name={}", name, e);
        }
    }

    public void forEach(String name, Runnable runnable) {
        if (EmergencyConfig.ENABLE_METRICS) {
            foreachMetric.record(Metrics.happened(name));
        }
        try {
            foreachPool.submit(runnable).get();
        } catch (Exception e) {
            log.error("Error while forEach name={}", name, e);
        }
    }

    public String getStats() {
        final TextStringBuilder builder = new TextStringBuilder();
        builder.appendln(" | -------");
        builder.appendln(" + PacketPool:");
        builder.appendln(" |- ActiveThreads:       " + packetPool.getActiveThreadCount());
        builder.appendln(" |- PoolSize:            " + packetPool.getPoolSize());
        builder.appendln(" |- QueuedTasks:         " + packetPool.getQueuedTaskCount());
        builder.appendln(" |- RunningThreadCount:  " + packetPool.getRunningThreadCount());
        builder.appendln(" | -------");
        builder.appendln(" + ForeachPool:");
        builder.appendln(" |- ActiveThreads:       " + foreachPool.getActiveThreadCount());
        builder.appendln(" |- PoolSize:            " + foreachPool.getPoolSize());
        builder.appendln(" |- QueuedTasks:         " + foreachPool.getQueuedTaskCount());
        builder.appendln(" |- RunningThreadCount:  " + foreachPool.getRunningThreadCount());
        builder.appendln(" | -------");
        return builder.toString();
    }

    @Override
    public List<MetricResult> getMetric() {
        final List<MetricResult> metricResults = new ArrayList<>();

        final MetricResult packetSubmitsResult = new MetricResult();
        packetSubmitsResult.setMetricGroupType(MetricGroupType.THREADPOOL);
        packetSubmitsResult.setName("PacketsSubmit");
        packetSubmitsResult.setData(packetSubmitsMetric.render().get());
        metricResults.add(packetSubmitsResult);

        final MetricResult foreachResult = new MetricResult();
        foreachResult.setMetricGroupType(MetricGroupType.THREADPOOL);
        foreachResult.setName("Foreach");
        foreachResult.setData(foreachMetric.render().get());
        metricResults.add(foreachResult);
        return metricResults;
    }
}