package host.anzo.commons.config;

import com.google.common.primitives.Ints;
import com.google.common.primitives.Shorts;
import host.anzo.commons.utils.MathUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.ParameterizedType;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;

/**
 * The Type Caster is small utility that helps put string values into object fields with different types.
 *
 * @author Yorie, ANZO
 * @since 02.04.2017
 */
@Slf4j
public class ConfigTypeCaster {
	/**
	 * Puts value to field.
	 *
	 * @param object Object, whom field value should be changed.
	 * @param field  Class field.
	 * @param value  Value to cast.
	 * @param splitter splitter symbol for collection type fields
	 */
	@SuppressWarnings("unchecked")
	public static void cast(Object object, @NotNull Field field, String value, String splitter, Long minValue, Long maxValue) throws IllegalAccessException, InvocationTargetException, InstantiationException, NoSuchMethodException {
		if (!field.trySetAccessible()) {
			throw new RuntimeException("Can't set accessible flag for field [" + field.getName() + "] at object [" + object + "]");
		}
		final Class<?> type = field.getType();
		if (type.isEnum()) {
			if (StringUtils.isNumeric(value)) {
				field.set(object, field.getType().getEnumConstants()[Integer.parseInt(value)]);
			}
			else {
				field.set(object, Enum.valueOf((Class<Enum>) type, value));
			}
		}
		else if (type == Integer.class || type == int.class)
			field.set(object, MathUtils.clamp(Integer.decode(value), Ints.saturatedCast(minValue), Ints.saturatedCast(maxValue)));
		else if (type == Short.class || type == short.class)
			field.set(object, MathUtils.clamp(Short.decode(value), Shorts.saturatedCast(minValue), Shorts.saturatedCast(maxValue)));
		else if (type == Float.class || type == float.class)
			field.set(object, MathUtils.clamp(Float.parseFloat(value), minValue.floatValue(), maxValue.floatValue()));
		else if (type == Double.class || type == double.class)
			field.set(object, MathUtils.clamp(Double.parseDouble(value), minValue.doubleValue(), maxValue.doubleValue()));
		else if (type == Long.class || type == long.class)
			field.set(object, MathUtils.clamp(Long.decode(value), minValue, maxValue));
		else if (type == Boolean.class || type == boolean.class) {
			if (NumberUtils.isDigits(value)) {
				field.set(object, value.equals("1"));
			}
			else {
				field.set(object, Boolean.parseBoolean(value));
			}
		}
		else if (type == String.class)
			field.set(object, value);
		else if (type == Character.class || type == char.class)
			field.set(object, value.charAt(0));
		else if (type == Byte.class || type == byte.class)
			field.set(object, Byte.parseByte(value));
		else if (type == AtomicInteger.class)
			field.set(object, new AtomicInteger(Integer.decode(value)));
		else if (type == AtomicBoolean.class)
			field.set(object, new AtomicBoolean(Boolean.parseBoolean(value)));
		else if (type == AtomicLong.class)
			field.set(object, new AtomicLong(Long.decode(value)));
		else if (type == BigInteger.class)
			field.set(object, new BigInteger(value));
		else if (type == BigDecimal.class)
			field.set(object, new BigDecimal(value));
		else if (type == Path.class)
			field.set(object, Paths.get(value));
		else if (type == Pattern.class)
			field.set(object, Pattern.compile(value));
		else if (type == Duration.class)
			field.set(object, Duration.parse(value));
		else if (type == UUID.class)
			field.set(object, UUID.fromString(value));
		else if (type.isArray()) {
			if (value != null) {
				final String[] values = value.split(splitter);
				final Class<?> baseType = field.getType().getComponentType();
				final Object array = Array.newInstance(baseType, values.length);
				field.set(object, array);

				int index = 0;
				for (String arrValue : values) {
					final Object objectValue = ConfigTypeCaster.cast(baseType, arrValue.trim());
					Array.set(array, index, objectValue);
					++index;
				}

				field.set(object, array);
			}
		}
		else if (type.isAssignableFrom(List.class)) {
			if (value != null) {
				final Class<?> genericType = (Class<?>)((ParameterizedType)field.getGenericType()).getActualTypeArguments()[0];
				final List<Object> list = ((List<Object>) field.get(object));
				list.clear();
				final String[] values = value.split(splitter);
				for (String listValue : values) {
					if (!listValue.trim().isEmpty()) {
						list.add(ConfigTypeCaster.cast(genericType, listValue.trim()));
					}
				}
			}
		}
		else if (type.isAssignableFrom(EnumSet.class)) {
			if (value != null) {
				final Class<?> genericType = (Class<?>) ((ParameterizedType) field.getGenericType()).getActualTypeArguments()[0];
				final EnumSet set = ((EnumSet) field.get(object));
				set.clear();
				final String[] values = value.split(splitter);
				for (String listValue : values) {
					set.add(ConfigTypeCaster.cast(genericType, listValue.trim()));
				}
			}
		}
		else if (type.isAssignableFrom(HashSet.class)) {
			if (value != null) {
				final Class<?> genericType = (Class<?>) ((ParameterizedType) field.getGenericType()).getActualTypeArguments()[0];
				final HashSet<Object> set = ((HashSet<Object>) field.get(object));
				set.clear();
				final String[] values = value.split(splitter);
				for (String listValue : values) {
					set.add(ConfigTypeCaster.cast(genericType, listValue.trim()));
				}
			}
		}
		else {
			try {
				// Using constructor with string argument as last resort
				field.set(object, type.getDeclaredConstructor(String.class).newInstance(value));
			}
			catch (Exception e) {
				throw new RuntimeException("Unsupported type [" + field.getType().getName() + "] for field [" + field.getName() + "]", e);
			}
		}
	}

	/**
	 * Changes targets' value to new given value with type casting.
	 *
	 * @param type  Cast type.
	 * @param value Value to cast.
	 */
	@SuppressWarnings({"unchecked", "rawtypes"})
	public static <T> @Nullable T cast(Class<T> type, String value) throws IllegalAccessException {
		String parseValue = value;
		if (StringUtils.isEmpty(value)
				&& (Number.class.isAssignableFrom(type)
					|| type == int.class
					|| type == short.class
					|| type == float.class
					|| type == double.class
					|| type == long.class)) {
			parseValue = "0";
		}
		if (type.isEnum()) {
			if (StringUtils.isNumeric(value)) {
				return type.getEnumConstants()[Integer.parseInt(value)];
			}
			else {
				return (T) Enum.valueOf((Class<Enum>) type, value);
			}
		}
		else if (type == Integer.class || type == int.class)
			return (T) Integer.decode(parseValue);
		else if (type == Short.class || type == short.class)
			return (T) Short.decode(parseValue);
		else if (type == Float.class || type == float.class)
			return (T) Float.valueOf(parseValue);
		else if (type == Double.class || type == double.class)
			return (T) Double.valueOf(parseValue);
		else if (type == Long.class || type == long.class)
			return (T) Long.decode(parseValue);
		else if (type == Boolean.class || type == boolean.class) {
			if (NumberUtils.isDigits(value)) {
				return (T) (Boolean) value.equals("1");
			}
			else {
				return (T) (Boolean) Boolean.parseBoolean(value);
			}
		}
		else if (type == String.class)
			return (T) value;
		else if (type == Character.class || type == char.class)
			return (T) ((Object) value.charAt(0));
		else if (type == Byte.class || type == byte.class)
			return (T) Byte.decode(value);
		else if (type == AtomicInteger.class)
			return (T) new AtomicInteger(Integer.decode(value));
		else if (type == AtomicBoolean.class)
			return (T) new AtomicBoolean(Boolean.parseBoolean(value));
		else if (type == AtomicLong.class)
			return (T) new AtomicLong(Long.decode(parseValue));
		else if (type == BigInteger.class)
			return (T) new BigInteger(parseValue);
		else if (type == BigDecimal.class)
			return (T) new BigDecimal(parseValue);
		else if (type == Path.class)
			return (T) Paths.get(value);
		else if (type == Pattern.class)
			return (T) Pattern.compile(value);
		else if (type == Duration.class)
			return (T) Duration.parse(value);
		else if (type == UUID.class)
			return (T) UUID.fromString(value);

		try {
			// Using constructor with string argument as last resort
			return type.getDeclaredConstructor(String.class).newInstance(value);
		}
		catch (Exception e) {
			throw new RuntimeException("Can't cast value [" + value + "] to type [" + type + "]", e);
		}
	}
}