package host.anzo.commons.utils;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.*;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import spark.Request;
import spark.Response;

import java.io.IOException;

/**
 * @author ANZO
 * @since 08.06.2017
 */
@Slf4j
public class JsonUtils {
    private static final ObjectMapper jsonMapper = new ObjectMapper();
    static {
        jsonMapper.setVisibility(jsonMapper.getSerializationConfig()
                .with(SerializationFeature.INDENT_OUTPUT)
                .with(SerializationFeature.WRITE_ENUMS_USING_INDEX)
                .with(MapperFeature.ACCEPT_CASE_INSENSITIVE_VALUES)
                .with(MapperFeature.ACCEPT_CASE_INSENSITIVE_ENUMS)
                .getDefaultVisibilityChecker()
                .withFieldVisibility(JsonAutoDetect.Visibility.ANY)
                .withGetterVisibility(JsonAutoDetect.Visibility.NONE)
                .withSetterVisibility(JsonAutoDetect.Visibility.NONE)
                .withCreatorVisibility(JsonAutoDetect.Visibility.NONE));
    }

    public static <T> T jsonToObject(@NotNull Request request, Class<T> clazz) throws JsonProcessingException {
        return jsonMapper.readValue(request.body(), clazz);
    }

    public static <T> T jsonToObject(String body, Class<T> clazz) throws JsonProcessingException {
        return jsonMapper.readValue(body, clazz);
    }

    public static String dataToJson(@NotNull Response response, Object data) {
        response.type("application/json");
        try {
            return jsonMapper.writeValueAsString(data);
        }
        catch (JsonProcessingException e) {
            log.error("Error while mapping object to JSON", e);
        }
        return "";
    }

    public static String dataToJson(Object data) {
        try {
            return jsonMapper.writeValueAsString(data);
        }
        catch (JsonProcessingException e){
            log.error("Error while mapping object to JSON", e);
        }
        return "";
    }

    public static class CustomBooleanDeserializer extends JsonDeserializer<Boolean> {
        @Override
        public Boolean deserialize(@NotNull JsonParser jp, DeserializationContext ctxt) throws IOException {
            final JsonToken currentToken = jp.getCurrentToken();

            if (currentToken.equals(JsonToken.VALUE_STRING)) {
                final String text = jp.getText().toLowerCase();
                if (text.equals("true") || text.equals("y") || text.equals("1")) {
                    return Boolean.TRUE;
                }
                return Boolean.FALSE;
            }
            else if (currentToken.equals(JsonToken.VALUE_NULL)) {
                return Boolean.FALSE;
            }
            ctxt.reportInputMismatch(this, "Can't parse boolean value: " + jp.getText());
            return false;
        }
    }
}