package host.anzo.core.service;

import de.mxro.metrics.jre.Metrics;
import delight.async.properties.PropertyNode;
import host.anzo.commons.annotations.startup.Scheduled;
import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.emergency.metric.IMetric;
import host.anzo.commons.emergency.metric.Metric;
import host.anzo.commons.emergency.metric.MetricGroupType;
import host.anzo.commons.emergency.metric.MetricResult;
import host.anzo.commons.enums.startup.EShutdownPriority;
import host.anzo.commons.interfaces.startup.IShutdownable;
import host.anzo.commons.threading.RunnableWrapper;
import host.anzo.commons.threading.ThreadPoolPriorityFactory;
import host.anzo.commons.utils.ConsoleUtils;
import host.anzo.core.config.EmergencyConfig;
import host.anzo.core.config.ThreadPoolConfig;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

/**
 * @author Aristo, ANZO
 * @since 8/19/2021
 */
@Slf4j
@Metric
@StartupComponent(value = "Threading", shutdownPriority = EShutdownPriority.MINOR)
public class ThreadPoolService implements IShutdownable, IMetric {
	@Getter(lazy = true)
	private static final ThreadPoolService instance = new ThreadPoolService();

	private final AtomicInteger threadNumber = new AtomicInteger(1);

	// System thread executors
	private final ScheduledThreadPoolExecutor SYSTEM_SCHEDULER;
	private final ThreadPoolExecutor SYSTEM_EXECUTOR;

	// Virtual thread executor
	private final ExecutorService VIRTUAL_EXECUTOR = Executors.newVirtualThreadPerTaskExecutor();

	// ForkJoin
	private final ForkJoinPool FORK_JOIN_POOL;

	// Metrics
	private final PropertyNode SYSTEM_SCHEDULER_METRICS;
	private final PropertyNode SYSTEM_EXECUTOR_METRICS;
	private final PropertyNode VIRTUAL_EXECUTOR_METRICS;
	private final PropertyNode FORK_JOIN_POOL_METRICS;

	private final AtomicBoolean _shutdown = new AtomicBoolean();

	ThreadPoolService() {
		ConsoleUtils.printSection("ThreadPoolManager Loading");

		SYSTEM_SCHEDULER = new ScheduledThreadPoolExecutor(ThreadPoolConfig.SCHEDULED_THREAD_POOL_CONFIG[0] > -1 ? ThreadPoolConfig.SCHEDULED_THREAD_POOL_CONFIG[0] : Runtime.getRuntime().availableProcessors(),
				new ThreadPoolPriorityFactory("[POOL]Scheduled", ThreadPoolConfig.SCHEDULED_THREAD_POOL_CONFIG[1]));
		SYSTEM_SCHEDULER.setRemoveOnCancelPolicy(true);
		SYSTEM_SCHEDULER.prestartAllCoreThreads();
		SYSTEM_SCHEDULER.setExecuteExistingDelayedTasksAfterShutdownPolicy(false);

		SYSTEM_EXECUTOR = new ThreadPoolExecutor(ThreadPoolConfig.THREAD_POOL_EXECUTOR_CONFIG[0] > -1 ? ThreadPoolConfig.THREAD_POOL_EXECUTOR_CONFIG[0] : Runtime.getRuntime().availableProcessors(),
				Integer.MAX_VALUE, 1, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(), new ThreadPoolPriorityFactory("[POOL]Execute", ThreadPoolConfig.THREAD_POOL_EXECUTOR_CONFIG[1]));
		SYSTEM_EXECUTOR.prestartAllCoreThreads();

		FORK_JOIN_POOL = new ForkJoinPool(ThreadPoolConfig.FORK_JOIN_POOL_CONFIG[0] > -1 ? ThreadPoolConfig.FORK_JOIN_POOL_CONFIG[0] : Runtime.getRuntime().availableProcessors(), pool -> {
			final ForkJoinWorkerThread worker = ForkJoinPool.defaultForkJoinWorkerThreadFactory.newThread(pool);
			worker.setName("[POOL]ForkJoin-" + threadNumber.getAndIncrement());
			worker.setPriority(ThreadPoolConfig.FORK_JOIN_POOL_CONFIG[1]);
			return worker;
		}, null, true);

		SYSTEM_SCHEDULER_METRICS = Metrics.create();
		SYSTEM_EXECUTOR_METRICS = Metrics.create();
		VIRTUAL_EXECUTOR_METRICS = Metrics.create();
		FORK_JOIN_POOL_METRICS = Metrics.create();

		printStats();
	}

	/**
	 * Executes an general task sometime in future in another thread.
	 * @param task the task to execute
	 */
	public void execute(Runnable task) {
		execute(task, false);
	}

	/**
	 * Executes an general task sometime in future in another thread.
	 * @param task the task to execute
	 * @param name optional task name for metrics
	 */
	public void execute(Runnable task, String name) {
		execute(task, false, name);
	}

	/**
	 * Executes an general task sometime in future in another thread.
	 * @param task      the task to execute
	 * @param isVirtual {@code true} if it's must be virtual thread, {@code false} otherwise
	 */
	public void execute(Runnable task, boolean isVirtual) {
		execute(task, isVirtual, null);
	}

	/**
	 * Executes an general task sometime in future in another thread.
	 * @param task      the task to execute
	 * @param isVirtual {@code true} if it's must be virtual thread, {@code false} otherwise
	 * @param name      optional task name for metrics
	 */
	public void execute(Runnable task, boolean isVirtual, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				(isVirtual ? VIRTUAL_EXECUTOR_METRICS : SYSTEM_EXECUTOR_METRICS).record(Metrics.happened(name != null ? name : task.getClass().getName()));
			}
			(isVirtual ? VIRTUAL_EXECUTOR : SYSTEM_EXECUTOR).execute(new RunnableWrapper(task, isVirtual, name));
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed execute task!", e);
				Thread.dumpStack();
			}
		}
	}

	/**
	 * Submits a Runnable task for execution and returns a Future
	 * representing that task. The Future's {@code get} method will
	 * return {@code null} upon <em>successful</em> completion.
	 *
	 * @param task the task to submit
	 * @return a Future representing pending completion of the task
	 */
	public Future<?> submit(Runnable task) {
		return submit(task, false);
	}

	/**
	 * Submits a Runnable task for execution and returns a Future
	 * representing that task. The Future's {@code get} method will
	 * return {@code null} upon <em>successful</em> completion.
	 *
	 * @param task the task to submit
	 * @param name optional task name for metrics
	 * @return a Future representing pending completion of the task
	 */
	public Future<?> submit(Runnable task, String name) {
		return submit(task, false, name);
	}

	/**
	 * Submits a Runnable task for execution and returns a Future
	 * representing that task. The Future's {@code get} method will
	 * return {@code null} upon <em>successful</em> completion.
	 *
	 * @param task      the task to submit
	 * @param isVirtual {@code true} if it's must be virtual thread, {@code false} otherwise
	 * @return a Future representing pending completion of the task
	 */
	public @Nullable Future<?> submit(Runnable task, boolean isVirtual) {
		return submit(task, isVirtual, null);
	}

	/**
	 * Submits a Runnable task for execution and returns a Future
	 * representing that task. The Future's {@code get} method will
	 * return {@code null} upon <em>successful</em> completion.
	 *
	 * @param task      the task to submit
	 * @param name      optional task name for metrics
	 * @param isVirtual {@code true} if it's must be virtual thread, {@code false} otherwise
	 * @return a Future representing pending completion of the task
	 */
	public @Nullable Future<?> submit(Runnable task, boolean isVirtual, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				(isVirtual ? VIRTUAL_EXECUTOR_METRICS : SYSTEM_EXECUTOR_METRICS).record(Metrics.happened(name != null ? name : task.getClass().getSimpleName()));
			}
			return (isVirtual ? VIRTUAL_EXECUTOR : SYSTEM_EXECUTOR).submit(new RunnableWrapper(task, isVirtual, name));
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed submit task!", e);
				Thread.dumpStack();
			}
		}
		return null;
	}

	/**
	 * Submits a one-shot task that becomes enabled after the given delay.
	 *
	 * @param task the task to execute
	 * @param delay the time from now to delay execution
	 * @return a ScheduledFuture representing pending completion of
	 *         the task and whose {@code get()} method will return
	 *         {@code null} upon completion
	 */
	public ScheduledFuture<?> schedule(Runnable task, long delay) {
		return schedule(task, delay, TimeUnit.MILLISECONDS);
	}

	/**
	 * Submits a one-shot task that becomes enabled after the given delay.
	 *
	 * @param task the task to execute
	 * @param delay the time from now to delay execution
	 * @param unit the time unit of the delay parameter
	 * @return a ScheduledFuture representing pending completion of
	 *         the task and whose {@code get()} method will return
	 *         {@code null} upon completion
	 */
	public @Nullable ScheduledFuture<?> schedule(Runnable task, long delay, TimeUnit unit) {
		return schedule(task, delay, unit, null);
	}

	/**
	 * Submits a one-shot task that becomes enabled after the given delay.
	 *
	 * @param task the task to execute
	 * @param delay the time from now to delay execution
	 * @param unit the time unit of the delay parameter
	 * @param name optional task name for metrics
	 * @return a ScheduledFuture representing pending completion of
	 *         the task and whose {@code get()} method will return
	 *         {@code null} upon completion
	 */
	public @Nullable ScheduledFuture<?> schedule(Runnable task, long delay, TimeUnit unit, String name) {
		try {
			if (unit == null) {
				unit = TimeUnit.MILLISECONDS;
			}

			if (delay <= 0) {
				delay = 0;
				unit = TimeUnit.MILLISECONDS;
			}

			if (EmergencyConfig.ENABLE_METRICS) {
				final String metricName = name != null ? name : task.getClass().getName();
				SYSTEM_SCHEDULER_METRICS.record(Metrics.happened(metricName));
				SYSTEM_SCHEDULER_METRICS.record(Metrics.value(metricName + "[schedule delay]", unit.convert(delay, TimeUnit.MILLISECONDS)));
			}
			return SYSTEM_SCHEDULER.schedule(new RunnableWrapper(task, name), delay, unit);
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Scheduler: Failed schedule task!", e);
				Thread.dumpStack();
			}
			return null;
		}
	}

	/**
	 * Schedules a general task to be executed at fixed rate (in milliseconds)
	 *
	 * @param task         the task to execute
	 * @param initialDelay the initial delay in the given time unit
	 * @param period       the period between executions in the given time unit
	 * @return a ScheduledFuture representing pending completion of the task, and whose get() method will throw an exception upon cancellation
	 */
	public ScheduledFuture<?> scheduleAtFixedRate(Runnable task, long initialDelay, long period) {
		return scheduleAtFixedRate(task, initialDelay, period, TimeUnit.MILLISECONDS);
	}

	/**
	 * Schedules a general task to be executed at fixed rate.
	 *
	 * @param task         the task to execute
	 * @param initialDelay the initial delay in the given time unit
	 * @param period       the period between executions in the given time unit
	 * @param unit         the time unit of the initialDelay and period parameters
	 * @return a ScheduledFuture representing pending completion of the task, and whose get() method will throw an exception upon cancellation
	 */
	public @Nullable ScheduledFuture<?> scheduleAtFixedRate(Runnable task, long initialDelay, long period, TimeUnit unit) {
		return scheduleAtFixedRate(task, initialDelay, period, unit, null);
	}

	/**
	 * Schedules a general task to be executed at fixed rate.
	 *
	 * @param task         the task to execute
	 * @param initialDelay the initial delay in the given time unit
	 * @param period       the period between executions in the given time unit
	 * @param unit         the time unit of the initialDelay and period parameters
	 * @param name         optional task name for metrics
	 * @return a ScheduledFuture representing pending completion of the task, and whose get() method will throw an exception upon cancellation
	 */
	public @Nullable ScheduledFuture<?> scheduleAtFixedRate(Runnable task, long initialDelay, long period, TimeUnit unit, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				final String metricName = name != null ? name : task.getClass().getName();
				SYSTEM_SCHEDULER_METRICS.record(Metrics.happened(metricName));
				SYSTEM_SCHEDULER_METRICS.record(Metrics.value(metricName + "[schedule fixed rate]", unit.convert(period, TimeUnit.MILLISECONDS)));
			}
			return SYSTEM_SCHEDULER.scheduleAtFixedRate(new RunnableWrapper(task, name), Math.max(1, initialDelay), Math.max(1, period), unit);
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Scheduler: Failed schedule task!", e);
				Thread.dumpStack();
			}
			return null;
		}
	}

	/**
	 * Returns a new CompletableFuture that is asynchronously completed
	 * by a task running in ForkJoinPool with the value obtained
	 * by calling the given Supplier.
	 *
	 * @param supplier a function returning the value to be used
	 * to complete the returned CompletableFuture
	 * @param <U> the function's return type
	 * @return the new CompletableFuture
	 */
	public <U> CompletableFuture<U> supplyAsync(Supplier<U> supplier) {
		return CompletableFuture.supplyAsync(supplier, FORK_JOIN_POOL);
	}

	/**
	 * Returns a new CompletableFuture that is asynchronously completed
	 * by a task running in ForkJoinPool after it runs the given
	 * action.
	 *
	 * @param task the action to run before completing the returned CompletableFuture
	 * @return the new CompletableFuture
	 */
	public CompletableFuture<Void> runAsync(Runnable task) {
		return CompletableFuture.runAsync(task, FORK_JOIN_POOL);
	}

	/**
	 * Arranges for (asynchronous) execution of the given task in ForkJoinPool
	 * @param task the task
	 */
	public void executeForkJoin(Runnable task) {
		executeForkJoin(task, null);
	}

	/**
	 * Arranges for (asynchronous) execution of the given task in ForkJoinPool
	 * @param task the task
	 * @param name optional task name for metrics
	 */
	public void executeForkJoin(Runnable task, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				FORK_JOIN_POOL_METRICS.record(Metrics.happened(name != null ? name : task.getClass().getName()));
			}
			FORK_JOIN_POOL.execute(new RunnableWrapper(task, name));
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed execute task!", e);
				Thread.dumpStack();
			}
		}
	}

	/**
	 * Submits a task to ForkJoin pool for execution.
	 * @param task the task to submit
	 * @param name optional task name for metrics
	 * @return the task
	 */
	public @Nullable ForkJoinTask<?> submitForkJoin(Runnable task, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				FORK_JOIN_POOL_METRICS.record(Metrics.happened(name));
			}
			return FORK_JOIN_POOL.submit(new RunnableWrapper(task, name));
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed submit task!", e);
				Thread.dumpStack();
			}
			return null;
		}
	}

	/**
	 * Submits a specified task to ForkJoin pool for execution and wait until a task completed
	 * @param task the task to submit
	 */
	public void submitForkJoinGet(Runnable task) {
		submitForkJoinGet(task, null);
	}

	/**
	 * Submits a specified task to ForkJoin pool for execution and wait until a task completed
	 * @param task the task to submit
	 * @param name optional task name for metrics
	 */
	public void submitForkJoinGet(Runnable task, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				FORK_JOIN_POOL_METRICS.record(Metrics.happened(name));
			}
			FORK_JOIN_POOL.submit(new RunnableWrapper(task, name)).get();
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed submit task!", e);
				Thread.dumpStack();
			}
		} catch (Exception e) {
			log.error("Executor: Failed submit task!", e);
		}
	}

	/**
	 * Convert a specified runnable task to virtual thread and start this thread
	 * @param task task to convert
	 * @return virtual thread or {@code null} if failed to create virtual thread
	 */
	public @Nullable Thread toVT(Runnable task) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				VIRTUAL_EXECUTOR_METRICS.record(Metrics.happened(task.getClass().getName()));
			}
			return Thread.ofVirtual().start(task);
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed virtualize task!", e);
				Thread.dumpStack();
			}
			return null;
		}
	}

	/**
	 * Convert a specified runnable task to virtual thread, start this thread and waits for this thread to terminate.
	 * @param task task to convert
	 */
	public void toVTJoin(Runnable task) {
		try {
			final Thread virtualThread = toVT(task);
			if (virtualThread != null) {
				virtualThread.join();
			}
		} catch (InterruptedException e) {
			log.error("Executor: Failed virtualize task!", e);
			Thread.currentThread().interrupt();
		}
	}

	/**
	 * Print thread pool stats to log.
	 */
	public void printStats() {
		log.info("ThreadPoolManagerStats:");
		log.info("- SystemScheduler: {}", SYSTEM_SCHEDULER);
		log.info("- SystemExecutor: {}", SYSTEM_EXECUTOR);
		log.info("- ForkJoinPool: {}", FORK_JOIN_POOL);
	}

	@Override
	public @NotNull List<MetricResult> getMetric() {
		final List<MetricResult> metricResults = new ArrayList<>();

		final MetricResult scheduleResult = new MetricResult();
		scheduleResult.setMetricGroupType(MetricGroupType.THREADPOOL);
		scheduleResult.setName("SystemScheduler");
		scheduleResult.setData(SYSTEM_SCHEDULER_METRICS.render().get());
		metricResults.add(scheduleResult);

		final MetricResult systemExecutorResult = new MetricResult();
		systemExecutorResult.setMetricGroupType(MetricGroupType.THREADPOOL);
		systemExecutorResult.setName("SystemExecutor");
		systemExecutorResult.setData(SYSTEM_EXECUTOR_METRICS.render().get());
		metricResults.add(systemExecutorResult);

		final MetricResult virtualExecutorResult = new MetricResult();
		virtualExecutorResult.setMetricGroupType(MetricGroupType.THREADPOOL);
		virtualExecutorResult.setName("VirtualExecutor");
		virtualExecutorResult.setData(VIRTUAL_EXECUTOR_METRICS.render().get());
		metricResults.add(virtualExecutorResult);

		final MetricResult forkJoinResult = new MetricResult();
		forkJoinResult.setMetricGroupType(MetricGroupType.THREADPOOL);
		forkJoinResult.setName("ForkJoinPool");
		forkJoinResult.setData(FORK_JOIN_POOL_METRICS.render().get());
		metricResults.add(forkJoinResult);
		return metricResults;
	}

	/**
	 * Tries to remove from the work queue all
	 * {@link Future} tasks that have been cancelled. This
	 * method can be useful as a storage reclamation operation, that has no
	 * other impact on functionality. Cancelled tasks are never executed, but
	 * may accumulate in work queues until worker threads can actively remove
	 * them. Invoking this method instead tries to remove them now. However,
	 * this method may fail to remove tasks in the presence of interference by
	 * other threads.
	 */
	@SuppressWarnings("unused")
	@Scheduled(period = 10, timeUnit = TimeUnit.MINUTES, runAfterServerStart = true)
	public void purge() {
		SYSTEM_SCHEDULER.purge();
		SYSTEM_EXECUTOR.purge();
		printStats();
	}

	/**
	 * @return {@code true} if thread pool's in shutdown mode, {@code false} otherwise
	 */
	@SuppressWarnings("BooleanMethodIsAlwaysInverted")
	public boolean isShutdown() {
		return _shutdown.get();
	}

	@Override
	@SuppressWarnings("ResultOfMethodCallIgnored")
	public void onShutdown() {
		if (_shutdown.compareAndSet(false, true)) {
			try {
				SYSTEM_SCHEDULER.awaitTermination(10, TimeUnit.SECONDS);
				SYSTEM_EXECUTOR.awaitTermination(10, TimeUnit.SECONDS);

				VIRTUAL_EXECUTOR.awaitTermination(10, TimeUnit.SECONDS);
				FORK_JOIN_POOL.awaitTermination(10, TimeUnit.SECONDS);

				SYSTEM_SCHEDULER.shutdown();
				SYSTEM_EXECUTOR.shutdown();
				VIRTUAL_EXECUTOR.shutdown();
				FORK_JOIN_POOL.shutdown();

				log.info("All ThreadPools are now stopped.");

			} catch (InterruptedException e) {
				log.error("Error while shutdown()", e);
			}
		}
	}
}