package host.anzo.commons.utils;

import host.anzo.core.service.HttpService;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.net.*;
import java.util.regex.Pattern;

/**
 * @author rage, ANZO
 * @since 03.03.12 16:01
 */
@Slf4j
public class NetworkUtils {
    private static final Pattern VALID_IPV4_PATTERN = Pattern.compile("(([01]?\\d\\d?|2[0-4]\\d|25[0-5])\\.){3}([01]?\\d\\d?|2[0-4]\\d|25[0-5])", Pattern.CASE_INSENSITIVE);
    private static String EXTERNAL_IP = null;

    /**
     * @return external IP address, {@code null} if can't detect external IP
     */
    public static @Nullable String getExternalIP() {
        if (isValidIp(EXTERNAL_IP)) {
            return EXTERNAL_IP;
        }

        String ip = HttpService.getInstance().httpGet("https://api.ipify.org/");
        if (!isValidIp(ip)) {
            ip = HttpService.getInstance().httpGet("https://ipinfo.io/ip");
        }
        if (!isValidIp(ip)) {
            ip = HttpService.getInstance().httpGet("https://www.trackip.net/ip");
        }
        if (!isValidIp(ip)) {
            ip = HttpService.getInstance().httpGet("https://checkip.amazonaws.com/");
        }
        if (isValidIp(ip)) {
            EXTERNAL_IP = ip;
            return ip;
        }
        return null;
    }

    /**
     * @param address address for check
     * @return {@code true} if specified address is local, {@code false} otherwise
     */
    public static boolean isLocalAddress(@NotNull InetAddress address) {
        // Check if the address is a valid special local or loop back
        if (address.isAnyLocalAddress() || address.isLoopbackAddress()) {
            return true;
        }

        // Check if the address is defined on any interface
        try {
            return NetworkInterface.getByInetAddress(address) != null;
        } catch (SocketException e) {
            return false;
        }
    }

    /**
     * @param address address for check
     * @return {@code true} if specified address is local, {@code false} otherwise
     */
    public static boolean isLocalAddress(@NotNull String address) {
	    try {
		    return isLocalAddress(InetAddress.getByName(address));
	    } catch (UnknownHostException e) {
		    return false;
	    }
    }

    /**
     * @param ip ip to check
     * @return {@code true} if specified IP is valid, {@code false} otherwise
     */
    public static boolean isValidIp(String ip) {
        if (ip == null || ip.isEmpty()) {
            return false;
        }
        return VALID_IPV4_PATTERN.matcher(ip).matches();
    }

    /**
     * @param port port to check
     * @return {@code true} if port is free, {@code false} otherwise
     */
    public static boolean isPortFree(int port) {
        try (ServerSocket socket = new ServerSocket(port, 0)) {
            return true;
        }
        catch (Exception e) {
            log.error("Port {} is already bind. Please free it and restart server.", port);
            return false;
        }
    }
}