package host.anzo.core.service;

import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.model.enums.EHttpRequestContentType;
import host.anzo.core.config.WebServerConfig;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.conn.HttpHostConnectException;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.util.EntityUtils;
import org.jetbrains.annotations.NotNull;
import org.owasp.validator.html.AntiSamy;
import org.owasp.validator.html.Policy;
import org.owasp.validator.html.PolicyException;
import org.owasp.validator.html.ScanException;

import javax.net.ssl.*;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author ANZO
 * @since 04.07.2016
 */
@Slf4j
@StartupComponent("Service")
public class HttpService {
	@Getter(lazy = true)
	private final static HttpService instance = new HttpService();

	private static AntiSamy antiSamy;

	private HttpService() {
		try (InputStream ruleInputStream = getClass().getClassLoader().getResourceAsStream("antisamy-slashdot.xml")) {
			if (ruleInputStream != null) {
				antiSamy = new AntiSamy(Policy.getInstance(ruleInputStream));
			}
		}
		catch (Exception e) {
			throw new RuntimeException("Error while loading antisamy rules", e);
		}

		if (WebServerConfig.IGNORE_SSL_CERTIFICATE_ERRORS) {
			final TrustManager[] trustAllCerts = new TrustManager[] {
					new X509TrustManager() {
						public X509Certificate[] getAcceptedIssuers() {
							return null;
						}

						public void checkServerTrusted(X509Certificate[] certs, String authType) {
						}

						public void checkClientTrusted(X509Certificate[] certs, String authType) {
						}
					}
			};
			try {
				final SSLContext sslContext = SSLContext.getInstance("SSL");
				sslContext.init(null, trustAllCerts, new SecureRandom());
				HttpsURLConnection.setDefaultSSLSocketFactory(sslContext.getSocketFactory());
				final HostnameVerifier hv = (urlHostName, session) -> {
					if (!urlHostName.equalsIgnoreCase(session.getPeerHost())) {
						log.warn("URL host '{}' is different to SSLSession host '{}'.", urlHostName, session.getPeerHost());
					}
					return true;
				};
				HttpsURLConnection.setDefaultHostnameVerifier(hv);
			}
			catch (Exception e) {
				log.error("Can't override default hostname verifier due error", e);
			}
		}
	}

	public String stripXSS(String input) {
		try {
			return antiSamy.scan(input).getCleanHTML();
		}
		catch (PolicyException | ScanException e) {
			return "";
		}
	}

	public String httpGet(String urlString) {
		return httpGet(urlString, null);
	}

	public String httpGet(@NotNull String urlString, Map<String, String> headers) {
		if (urlString.startsWith("https")) {
			try {
				final SSLContext sslContext = new SSLContextBuilder().loadTrustMaterial(null, (certificate, authType) -> true).build();
				try (CloseableHttpClient client = HttpClients.custom().setSSLContext(sslContext).setSSLHostnameVerifier(new NoopHostnameVerifier()).build()) {
					final HttpGet httpGet = new HttpGet(urlString);
					if (headers != null) {
						for(Map.Entry<String, String> entry : headers.entrySet()) {
							httpGet.addHeader(entry.getKey(), entry.getValue());
						}
					}
					final HttpResponse response = client.execute(httpGet);
					final HttpEntity entity = response.getEntity();
					return EntityUtils.toString(entity, "UTF-8");
				}
			}
			catch (SocketTimeoutException | SSLHandshakeException | HttpHostConnectException ignore) {
			}
			catch (Exception e) {
				log.error("Fatal transport error", e);
			}
		}
		else {
			try (CloseableHttpClient client = HttpClients.createDefault()) {
				final HttpGet httpGet = new HttpGet(urlString);
				final HttpResponse response = client.execute(httpGet);
				final HttpEntity entity = response.getEntity();
				return EntityUtils.toString(entity, "UTF-8");
			}
			catch (Exception e) {
				log.error("Fatal transport error", e);
			}
		}
		return null;
	}

	public String httpPost(String urlString, @NotNull String postData) {
		return httpPost(urlString, postData, null, EHttpRequestContentType.FormUrlEncoded);
	}

	public String httpPost(String urlString, String postData, Map<String, String> headers, @NotNull EHttpRequestContentType requestContentType) {
		try {
			final URL url = new URL(urlString);
			final HttpURLConnection connection = (HttpURLConnection) url.openConnection();
			connection.setRequestMethod("POST");
			connection.setRequestProperty("charset", "UTF-8");
			connection.setRequestProperty("Content-Type", requestContentType.getTextValue());
			connection.setDoOutput(true);
			connection.setDoInput(true);

			if (headers != null) {
				for(Map.Entry<String, String> entry : headers.entrySet()) {
					connection.setRequestProperty(entry.getKey(), entry.getValue());
				}
			}

			if (postData != null) {
				try (final OutputStream out = connection.getOutputStream()) {
					out.write(postData.getBytes());
				}
				catch (Exception e) {
					log.error("Error while writing postData=[{}] to url=[{}]", postData, urlString, e);
				}
			}

			try(BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8))) {
				return reader.lines().collect(Collectors.joining(System.lineSeparator()));
			}
		}
		catch (IOException ignored) {
			return null;
		}
	}
}