package host.anzo.core.service;

import host.anzo.classindex.ClassIndex;
import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.utils.ClassUtils;
import host.anzo.core.extensions.cachedenum.CachedEnum;
import host.anzo.core.extensions.cachedenum.ICachedEnum;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The CachedEnumService class is responsible for managing cached enum values
 * that are annotated with the CachedEnum annotation. It initializes the cache
 * by scanning for classes that are annotated and implements the ICachedEnum interface.
 *
 * @author ANZO
 * @since 12/18/2024
 */
@Slf4j
@StartupComponent("Service")
public class CachedEnumService {
	@Getter(lazy = true)
	private final static CachedEnumService instance = new CachedEnumService();

	private static final Map<String, List<Enum<?>>> cache = new HashMap<>();

	private CachedEnumService() {
		ClassIndex.getAnnotated(CachedEnum.class).forEach(enumClass -> {
			final CachedEnum annotation = enumClass.getAnnotation(CachedEnum.class);
			if (annotation != null) {
				if (ICachedEnum.class.isAssignableFrom(enumClass)) {
					cache.put(enumClass.getName(), ClassUtils.getEnumValues(enumClass));
				}
				else {
					throw new RuntimeException("Found marked with CachedEnum annotation class, but without ICachedEnum implementation: " + enumClass.getSimpleName());
				}
			}
		});
		log.info("Initialized [{}] cached enums.", cache.size());
	}

	/**
	 * Retrieves the cached enum values for the specified enum class name.
	 *
	 * @param enumClassName the fully qualified name of the enum class
	 * @return a List of enum values associated with the specified class name,
	 *         or null if no values are cached for that class name.
	 */
	public List<Enum<?>> getEnumValues(String enumClassName) {
		return cache.get(enumClassName);
	}
}