package host.anzo.core.service;

import de.mxro.metrics.jre.Metrics;
import delight.async.properties.PropertyNode;
import host.anzo.commons.annotations.startup.Scheduled;
import host.anzo.commons.annotations.startup.StartupComponent;
import host.anzo.commons.emergency.metric.IMetric;
import host.anzo.commons.emergency.metric.Metric;
import host.anzo.commons.emergency.metric.MetricGroupType;
import host.anzo.commons.emergency.metric.MetricResult;
import host.anzo.commons.enums.startup.EShutdownPriority;
import host.anzo.commons.interfaces.startup.IShutdownable;
import host.anzo.commons.threading.CallableWrapper;
import host.anzo.commons.threading.RunnableWrapper;
import host.anzo.commons.threading.ThreadPoolPriorityFactory;
import host.anzo.commons.utils.ConsoleUtils;
import host.anzo.core.config.EmergencyConfig;
import host.anzo.core.config.ThreadPoolConfig;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.text.TextStringBuilder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

/**
 * @author Aristo, ANZO
 * @since 8/19/2021
 */
@Slf4j
@Metric
@SuppressWarnings("unused")
@StartupComponent(value = "Threading", shutdownPriority = EShutdownPriority.MINOR)
public class ThreadPoolService implements IShutdownable, IMetric {
	@Getter(lazy = true)
	private static final ThreadPoolService instance = new ThreadPoolService();

	private final AtomicInteger threadNumber = new AtomicInteger(1);

	// System thread executors
	private final ScheduledThreadPoolExecutor SYSTEM_SCHEDULER;
	private final ThreadPoolExecutor SYSTEM_EXECUTOR;

	// Virtual thread executor
	private final ExecutorService VIRTUAL_EXECUTOR = Executors.newVirtualThreadPerTaskExecutor();

	// ForkJoin
	private final ForkJoinPool FORK_JOIN_POOL;

	// Metrics
	private final PropertyNode SYSTEM_SCHEDULER_METRICS;
	private final PropertyNode SYSTEM_EXECUTOR_METRICS;
	private final PropertyNode VIRTUAL_EXECUTOR_METRICS;
	private final PropertyNode FORK_JOIN_POOL_METRICS;

	private final AtomicBoolean _shutdown = new AtomicBoolean();

	/**
	 * Initializes the ThreadPoolService.
	 * Creates and configures the thread pools and metrics objects.
	 */
	ThreadPoolService() {
		ConsoleUtils.printSection("ThreadPoolService Loading");

		SYSTEM_SCHEDULER = new ScheduledThreadPoolExecutor(ThreadPoolConfig.SCHEDULED_THREAD_POOL_CONFIG[0] > -1 ? ThreadPoolConfig.SCHEDULED_THREAD_POOL_CONFIG[0] : Runtime.getRuntime().availableProcessors(), new ThreadPoolPriorityFactory("[POOL]Scheduled", ThreadPoolConfig.SCHEDULED_THREAD_POOL_CONFIG[1]));
		SYSTEM_SCHEDULER.setRemoveOnCancelPolicy(true);
		SYSTEM_SCHEDULER.prestartAllCoreThreads();
		SYSTEM_SCHEDULER.setExecuteExistingDelayedTasksAfterShutdownPolicy(false);

		SYSTEM_EXECUTOR = new ThreadPoolExecutor(ThreadPoolConfig.THREAD_POOL_EXECUTOR_CONFIG[0] > -1 ? ThreadPoolConfig.THREAD_POOL_EXECUTOR_CONFIG[0] : Runtime.getRuntime().availableProcessors(), Integer.MAX_VALUE, 1, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(), new ThreadPoolPriorityFactory("[POOL]Execute", ThreadPoolConfig.THREAD_POOL_EXECUTOR_CONFIG[1]));
		SYSTEM_EXECUTOR.prestartAllCoreThreads();

		FORK_JOIN_POOL = new ForkJoinPool(ThreadPoolConfig.FORK_JOIN_POOL_CONFIG[0] > -1 ? ThreadPoolConfig.FORK_JOIN_POOL_CONFIG[0] : Runtime.getRuntime().availableProcessors(), pool -> {
			final ForkJoinWorkerThread worker = ForkJoinPool.defaultForkJoinWorkerThreadFactory.newThread(pool);
			worker.setName("[POOL]ForkJoin-" + threadNumber.getAndIncrement());
			worker.setPriority(ThreadPoolConfig.FORK_JOIN_POOL_CONFIG[1]);
			return worker;
		}, null, true);

		SYSTEM_SCHEDULER_METRICS = Metrics.create();
		SYSTEM_EXECUTOR_METRICS = Metrics.create();
		VIRTUAL_EXECUTOR_METRICS = Metrics.create();
		FORK_JOIN_POOL_METRICS = Metrics.create();
	}

	/**
	 * Executes an general task in another thread using the {@link #SYSTEM_EXECUTOR}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 *
	 * @param task the task to execute
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public void execute(@NotNull Runnable task) {
		execute(task, false);
	}

	/**
	 * Executes an general task in another thread using the {@link #SYSTEM_EXECUTOR}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param task the task to execute
	 * @param name optional task name for metrics
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public void execute(@NotNull Runnable task, String name) {
		execute(task, false, name);
	}

	/**
	 * Executes an general task in another thread.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * Uses the {@link #VIRTUAL_EXECUTOR} if `isVirtual` is true, otherwise uses the {@link #SYSTEM_EXECUTOR}.
	 *
	 * @param task      the task to execute
	 * @param isVirtual {@code true} if it's must be virtual thread, {@code false} otherwise
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public void execute(Runnable task, boolean isVirtual) {
		execute(task, isVirtual, null);
	}

	/**
	 * Executes an general task in another thread.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * Uses the {@link #VIRTUAL_EXECUTOR} if `isVirtual` is true, otherwise uses the {@link #SYSTEM_EXECUTOR}.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param task      the task to execute
	 * @param isVirtual {@code true} if it's must be virtual thread, {@code false} otherwise
	 * @param name      optional task name for metrics
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public void execute(@NotNull Runnable task, boolean isVirtual, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				(isVirtual ? VIRTUAL_EXECUTOR_METRICS : SYSTEM_EXECUTOR_METRICS).record(Metrics.happened(name != null ? name : task.getClass().getName()));
			}
			(isVirtual ? VIRTUAL_EXECUTOR : SYSTEM_EXECUTOR).execute(new RunnableWrapper(task, isVirtual, name));
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed execute task!", e);
				Thread.dumpStack();
			}
		}
	}

	/**
	 * Submits a general task to the {@link #SYSTEM_EXECUTOR} for execution.
	 * The task is wrapped in a {@link CallableWrapper}.
	 *
	 * @param <V>          the type of the task's result
	 * @param c            the task to submit
	 * @param defaultValue the default value to return if the task is rejected
	 * @return a Future representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will immediately return the default value on get().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public <V> @NotNull Future<V> submit(@NotNull Callable<V> c, V defaultValue) {
		return submit(c, defaultValue, false);
	}

	/**
	 * Submits a general task to the {@link #SYSTEM_EXECUTOR} for execution.
	 * The task is wrapped in a {@link CallableWrapper}.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param <V>          the type of the task's result
	 * @param c            the task to submit
	 * @param defaultValue the default value to return if the task is rejected
	 * @param name         optional task name for metrics
	 * @return a Future representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will immediately return the default value on get().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public <V> @NotNull Future<V> submit(@NotNull Callable<V> c, V defaultValue, String name) {
		return submit(c, defaultValue, false, name);
	}

	/**
	 * Submits a general task to the thread pool for execution.
	 * The task is wrapped in a {@link CallableWrapper}.
	 * Uses the {@link #VIRTUAL_EXECUTOR} if `virtual` is true, otherwise uses the {@link #SYSTEM_EXECUTOR}.
	 *
	 * @param <V>          the type of the task's result
	 * @param c            the task to submit
	 * @param defaultValue the default value to return if the task is rejected
	 * @param virtual      {@code true} if it's must be virtual thread, {@code false} otherwise
	 * @return a Future representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will immediately return the default value on get().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull <V> Future<V> submit(@NotNull Callable<V> c, V defaultValue, boolean virtual) {
		return submit(c, defaultValue, virtual, null);
	}

	/**
	 * Submits a general task to the thread pool for execution.
	 * The task is wrapped in a {@link CallableWrapper}.
	 * Uses the {@link #VIRTUAL_EXECUTOR} if `virtual` is true, otherwise uses the {@link #SYSTEM_EXECUTOR}.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param <V>          the type of the task's result
	 * @param c            the task to submit
	 * @param defaultValue the default value to return if the task is rejected
	 * @param virtual      {@code true} if it's must be virtual thread, {@code false} otherwise
	 * @param name         optional task name for metrics
	 * @return a Future representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will immediately return the default value on get().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull <V> Future<V> submit(@NotNull Callable<V> c, V defaultValue, boolean virtual, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				(virtual ? VIRTUAL_EXECUTOR_METRICS : SYSTEM_EXECUTOR_METRICS).record(Metrics.happened(name != null ? name : c.getClass().getSimpleName()));
			}
			return (virtual ? VIRTUAL_EXECUTOR : SYSTEM_EXECUTOR).submit(new CallableWrapper<>(c, defaultValue, virtual, name));
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed submit task!", e);
				Thread.dumpStack();
			}
			return new RejectedFuture<>(defaultValue, e);
		}
	}

	/**
	 * Submits a one-shot task that becomes enabled after the given delay to the {@link #SYSTEM_SCHEDULER}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * Uses {@link TimeUnit#MILLISECONDS} as the default time unit if the provided unit is null.
	 *
	 * @param task  the task to execute
	 * @param delay the time from now to delay execution
	 * @return a ScheduledFuture representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will immediately return {@code null} on get().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull ScheduledFuture<?> schedule(@NotNull Runnable task, long delay) {
		return schedule(task, delay, TimeUnit.MILLISECONDS);
	}

	/**
	 * Submits a one-shot task that becomes enabled after the given delay to the {@link #SYSTEM_SCHEDULER}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 *
	 * @param task  the task to execute
	 * @param delay the time from now to delay execution
	 * @param unit  the time unit of the delay parameter
	 * @return a ScheduledFuture representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will immediately return {@code null} on get().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull ScheduledFuture<?> schedule(@NotNull Runnable task, long delay, TimeUnit unit) {
		return schedule(task, delay, unit, null);
	}

	/**
	 * Submits a one-shot task that becomes enabled after the given delay to the {@link #SYSTEM_SCHEDULER}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * If metrics are enabled, records the task execution and the scheduling delay.
	 *
	 * @param task  the task to execute
	 * @param delay the time from now to delay execution
	 * @param unit  the time unit of the delay parameter
	 * @param name  optional task name for metrics
	 * @return a ScheduledFuture representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will immediately return {@code null} on get().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull ScheduledFuture<?> schedule(@NotNull Runnable task, long delay, TimeUnit unit, String name) {
		try {
			if (unit == null) {
				unit = TimeUnit.MILLISECONDS;
			}

			if (delay <= 0) {
				delay = 0;
				unit = TimeUnit.MILLISECONDS;
			}

			if (EmergencyConfig.ENABLE_METRICS) {
				final String metricName = name != null ? name : task.getClass().getName();
				SYSTEM_SCHEDULER_METRICS.record(Metrics.happened(metricName));
				SYSTEM_SCHEDULER_METRICS.record(Metrics.value(metricName + "[schedule delay]", TimeUnit.MILLISECONDS.convert(delay, unit)));
			}
			return SYSTEM_SCHEDULER.schedule(new RunnableWrapper(task, name), delay, unit);
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Scheduler: Failed schedule task!", e);
				Thread.dumpStack();
			}
			return new RejectedFuture<>(null, e);
		}
	}

	/**
	 * Schedules a general task to be executed at fixed rate using the {@link #SYSTEM_SCHEDULER}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * Uses {@link TimeUnit#MILLISECONDS} as the default time unit.
	 *
	 * @param task         the task to execute
	 * @param initialDelay the initial delay in the given time unit
	 * @param period       the period between executions in the given time unit
	 * @return a ScheduledFuture representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will always return {@code true} on cancel().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull ScheduledFuture<?> scheduleAtFixedRate(@NotNull Runnable task, long initialDelay, long period) {
		return scheduleAtFixedRate(task, initialDelay, period, TimeUnit.MILLISECONDS);
	}

	/**
	 * Schedules a general task to be executed at fixed rate using the {@link #SYSTEM_SCHEDULER}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 *
	 * @param task         the task to execute
	 * @param initialDelay the initial delay in the given time unit
	 * @param period       the period between executions in the given time unit
	 * @param unit         the time unit of the initialDelay and period parameters
	 * @return a ScheduledFuture representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will always return {@code true} on cancel().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull ScheduledFuture<?> scheduleAtFixedRate(@NotNull Runnable task, long initialDelay, long period, TimeUnit unit) {
		return scheduleAtFixedRate(task, initialDelay, period, unit, null);
	}

	/**
	 * Schedules a general task to be executed at fixed rate using the {@link #SYSTEM_SCHEDULER}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * If metrics are enabled, records the task execution and the fixed rate period.
	 *
	 * @param task         the task to execute
	 * @param initialDelay the initial delay in the given time unit
	 * @param period       the period between executions in the given time unit
	 * @param unit         the time unit of the initialDelay and period parameters
	 * @param name         optional task name for metrics
	 * @return a ScheduledFuture representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will always return {@code true} on cancel().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull ScheduledFuture<?> scheduleAtFixedRate(@NotNull Runnable task, long initialDelay, long period, TimeUnit unit, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				final String metricName = name != null ? name : task.getClass().getName();
				SYSTEM_SCHEDULER_METRICS.record(Metrics.happened(metricName));
				SYSTEM_SCHEDULER_METRICS.record(Metrics.value(metricName + "[schedule fixed rate]", TimeUnit.MILLISECONDS.convert(period, unit)));
			}
			return SYSTEM_SCHEDULER.scheduleAtFixedRate(new RunnableWrapper(task, name), Math.max(1, initialDelay), Math.max(1, period), unit);
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Scheduler: Failed schedule task!", e);
				Thread.dumpStack();
			}
			return new RejectedFuture<>(null, e);
		}
	}

	/**
	 * Schedules a general task to be executed periodically with a fixed delay between the end of one execution and the start of the next using the {@link #SYSTEM_SCHEDULER}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * Uses {@link TimeUnit#MILLISECONDS} as the default time unit.
	 *
	 * @param task       the task to execute
	 * @param initialDelay the initial delay in milliseconds before the first execution
	 * @param delay   the delay in milliseconds between the end of one execution and the start of the next
	 * @return a ScheduledFuture representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will always return {@code true} on cancel().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public ScheduledFuture<?> scheduleWithFixedDelay(@NotNull Runnable task, long initialDelay, long delay) {
		return scheduleWithFixedDelay(task, initialDelay, delay, TimeUnit.MILLISECONDS);
	}

	/**
	 * Schedules a general task to be executed periodically with a fixed delay between the end of one execution and the start of the next using the {@link #SYSTEM_SCHEDULER}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 *
	 * @param task       the task to execute
	 * @param initialDelay the initial delay before the first execution
	 * @param delay   the delay between the end of one execution and the start of the next
	 * @param unit    the time unit of the initial and delay parameters
	 * @return a ScheduledFuture representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will always return {@code true} on cancel().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull ScheduledFuture<?> scheduleWithFixedDelay(@NotNull Runnable task, long initialDelay, long delay, TimeUnit unit) {
		return scheduleWithFixedDelay(task, initialDelay, delay, unit, null);
	}

	/**
	 * Schedules a general task to be executed periodically with a fixed delay between the end of one execution and the start of the next using the {@link #SYSTEM_SCHEDULER}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * If metrics are enabled, records the task execution and the fixed delay period.
	 *
	 * @param task       the task to execute
	 * @param initialDelay the initial delay before the first execution
	 * @param delay   the delay between the end of one execution and the start of the next
	 * @param unit    the time unit of the initial and delay parameters
	 * @param name    optional task name for metrics
	 * @return a ScheduledFuture representing pending completion of the task.
	 * If the task is rejected, returns a {@link RejectedFuture} that will always return {@code true} on cancel().
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @NotNull ScheduledFuture<?> scheduleWithFixedDelay(@NotNull Runnable task, long initialDelay, long delay, TimeUnit unit, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				final String metricName = name != null ? name : task.getClass().getName();
				SYSTEM_SCHEDULER_METRICS.record(Metrics.happened(metricName));
				SYSTEM_SCHEDULER_METRICS.record(Metrics.value(metricName + "[schedule fixed delay]", TimeUnit.MILLISECONDS.convert(delay, unit)));
			}
			return SYSTEM_SCHEDULER.scheduleWithFixedDelay(new RunnableWrapper(task, name), Math.max(1, initialDelay), Math.max(1, delay), unit);
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Scheduler: Failed schedule task!", e);
				Thread.dumpStack();
			}
			return new RejectedFuture<>(null, e);
		}
	}

	/**
	 * Returns a new CompletableFuture that is asynchronously completed
	 * by a task running in {@link #FORK_JOIN_POOL} with the value obtained
	 * by calling the given Supplier.
	 *
	 * @param supplier a function returning the value to be used
	 *                 to complete the returned CompletableFuture
	 * @param <U>      the function's return type
	 * @return the new CompletableFuture
	 */
	public <U> CompletableFuture<U> supplyAsync(@NotNull Supplier<U> supplier) {
		return CompletableFuture.supplyAsync(supplier, FORK_JOIN_POOL);
	}

	/**
	 * Returns a new CompletableFuture that is asynchronously completed
	 * by a task running in {@link #FORK_JOIN_POOL} after it runs the given
	 * action.
	 *
	 * @param task the action to run before completing the returned CompletableFuture
	 * @return the new CompletableFuture
	 */
	public CompletableFuture<Void> runAsync(@NotNull Runnable task) {
		return CompletableFuture.runAsync(task, FORK_JOIN_POOL);
	}

	/**
	 * Arranges for (asynchronous) execution of the given task in {@link #FORK_JOIN_POOL}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 *
	 * @param task the task
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public void executeForkJoin(@NotNull Runnable task) {
		executeForkJoin(task, null);
	}

	/**
	 * Arranges for (asynchronous) execution of the given task in {@link #FORK_JOIN_POOL}.
	 * The task is wrapped in a {@link RunnableWrapper}.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param task the task
	 * @param name optional task name for metrics
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public void executeForkJoin(@NotNull Runnable task, String name) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				FORK_JOIN_POOL_METRICS.record(Metrics.happened(name != null ? name : task.getClass().getName()));
			}
			FORK_JOIN_POOL.execute(new RunnableWrapper(task, name));
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed execute task!", e);
				Thread.dumpStack();
			}
		}
	}

	/**
	 * Submits a task to the {@link #FORK_JOIN_POOL}, wraps it into a {@link RunnableWrapper} and blocks until it is completed.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param task the task to submit
	 * @param name optional task name for metrics
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public void submitForkJoinGet(@NotNull Runnable task, String name) {
		try {
			submitForkJoin(task, name).get();
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed submit task!", e);
				Thread.dumpStack();
			}
		} catch (Exception e) {
			log.error("Executor: Failed submit task!", e);
		}
	}

	/**
	 * Submits a task to the {@link #FORK_JOIN_POOL} for execution wrapping it into a {@link RunnableWrapper}.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param task the task to submit
	 * @param name optional task name for metrics
	 * @return a ForkJoinTask representing pending completion of the task
	 */
	public ForkJoinTask<?> submitForkJoin(@NotNull Runnable task, String name) {
		if (EmergencyConfig.ENABLE_METRICS) {
			FORK_JOIN_POOL_METRICS.record(Metrics.happened(name));
		}
		return FORK_JOIN_POOL.submit(new RunnableWrapper(task, name));
	}

	/**
	 * Submits a task to the {@link #FORK_JOIN_POOL}, wraps it into a {@link CallableWrapper}, blocks until it is completed, and returns the result.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param <V>          the type of the task's result
	 * @param task         the task to submit
	 * @param defaultValue the default value to return if the task fails
	 * @param name         optional task name for metrics
	 * @return the result of the task, or the default value if the task fails
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public <V> V submitForkJoinGet(@NotNull Callable<V> task, V defaultValue, String name) {
		try {
			return submitForkJoin(task, name).get();
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed submit task!", e);
				Thread.dumpStack();
			}
		} catch (Exception e) {
			log.error("Executor: Failed submit task!", e);
		}
		return defaultValue;
	}

	/**
	 * Submits a task to the {@link #FORK_JOIN_POOL} for execution wrapping it into a {@link CallableWrapper}.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param <V>  the type of the task's result
	 * @param task the task to submit
	 * @param name optional task name for metrics
	 * @return a ForkJoinTask representing pending completion of the task
	 */
	public <V> ForkJoinTask<V> submitForkJoin(@NotNull Callable<V> task, String name) {
		return submitForkJoin(task, null, name);
	}

	/**
	 * Submits a task to the {@link #FORK_JOIN_POOL} for execution wrapping it into a {@link CallableWrapper}.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param <V>          the type of the task's result
	 * @param task         the task to submit
	 * @param defaultValue the default value to return if the task fails
	 * @param name         optional task name for metrics
	 * @return a ForkJoinTask representing pending completion of the task
	 */
	public <V> ForkJoinTask<V> submitForkJoin(@NotNull Callable<V> task, V defaultValue, String name) {
		if (EmergencyConfig.ENABLE_METRICS) {
			FORK_JOIN_POOL_METRICS.record(Metrics.happened(name));
		}
		return FORK_JOIN_POOL.submit(new CallableWrapper<>(task, defaultValue, false, name));
	}

	/**
	 * Submits a task to the {@link #FORK_JOIN_POOL} from an external thread, wraps it into a {@link CallableWrapper}, blocks until it is completed, and returns the result.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param <V>          the type of the task's result
	 * @param task         the task to submit
	 * @param defaultValue the default value to return if the task fails
	 * @param name         optional task name for metrics
	 * @return the result of the task, or the default value if the task fails
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public <V> V submitExternalForkJoinGet(@NotNull Callable<V> task, V defaultValue, String name) {
		try {
			return submitExternalForkJoin(ForkJoinTask.adapt(new CallableWrapper<>(task, defaultValue, false, name)), name).get();
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed submit task!", e);
				Thread.dumpStack();
			}
		} catch (Exception e) {
			log.error("Executor: Failed submit task!", e);
		}
		return defaultValue;
	}

	/**
	 * Submits a task to the {@link #FORK_JOIN_POOL} from an external thread, wraps it into a {@link RunnableWrapper} and blocks until it is completed.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param task the task to submit
	 * @param name optional task name for metrics
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public void submitExternalForkJoinGet(@NotNull Runnable task, String name) {
		try {
			submitExternalForkJoin(ForkJoinTask.adapt(new RunnableWrapper(task, false, name)), name).get();
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed submit task!", e);
				Thread.dumpStack();
			}
		} catch (Exception e) {
			log.error("Executor: Failed submit task!", e);
		}
	}

	/**
	 * Submits a ForkJoinTask to the {@link #FORK_JOIN_POOL} for execution from an external thread.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param <T>  the type of the task's result
	 * @param task the ForkJoinTask to submit
	 * @param name optional task name for metrics
	 * @return the submitted ForkJoinTask
	 */
	public <T> @NotNull ForkJoinTask<T> submitExternalForkJoin(@NotNull ForkJoinTask<T> task, String name) {
		if (EmergencyConfig.ENABLE_METRICS) {
			FORK_JOIN_POOL_METRICS.record(Metrics.happened(name));
		}
		return FORK_JOIN_POOL.externalSubmit(task);
	}

	/**
	 * Convert a specified runnable task to virtual thread and start this thread using {@link Thread#ofVirtual()}.
	 * If metrics are enabled, records the task execution.
	 *
	 * @param task task to convert
	 * @return virtual thread or {@code null} if failed to create virtual thread
	 * @throws RejectedExecutionException if the task cannot be accepted for execution
	 */
	public @Nullable Thread toVT(@NotNull Runnable task) {
		try {
			if (EmergencyConfig.ENABLE_METRICS) {
				VIRTUAL_EXECUTOR_METRICS.record(Metrics.happened(task.getClass().getName()));
			}
			return Thread.ofVirtual().start(task);
		} catch (RejectedExecutionException e) {
			if (!isShutdown()) {
				log.error("Executor: Failed virtualize task!", e);
				Thread.dumpStack();
			}
			return null;
		}
	}

	/**
	 * Convert a specified runnable task to virtual thread, start this thread and waits for this thread to terminate using {@link #toVT} and {@link Thread#join}.
	 *
	 * @param task task to convert
	 */
	public void toVTJoin(Runnable task) {
		try {
			final Thread virtualThread = toVT(task);
			if (virtualThread != null) {
				virtualThread.join();
			}
		} catch (InterruptedException e) {
			log.error("Executor: Failed virtualize task!", e);
			Thread.currentThread().interrupt();
		}
	}

	/**
	 * @return thread scheduler {@link #SYSTEM_SCHEDULER}
	 */
	public ScheduledExecutorService getScheduler() {
		return SYSTEM_SCHEDULER;
	}

	/**
	 * @return thread executor {@link #SYSTEM_EXECUTOR}
	 */
	public ExecutorService getExecutor() {
		return SYSTEM_EXECUTOR;
	}

	/**
	 * @return ForkJoinPool {@link #FORK_JOIN_POOL}
	 */
	public ForkJoinPool getForkJoinPool() {
		return FORK_JOIN_POOL;
	}

	/**
	 * @return all threads executor's stats
	 */
	public @NotNull String getStats() {
		final TextStringBuilder builder = new TextStringBuilder();
		builder.appendln(" | ----------------");
		builder.appendln(" + SystemScheduler:");
		builder.appendln(" |- ActiveThreads:       " + SYSTEM_SCHEDULER.getActiveCount());
		builder.appendln(" |- CorePoolSize:        " + SYSTEM_SCHEDULER.getCorePoolSize());
		builder.appendln(" |- PoolSize:            " + SYSTEM_SCHEDULER.getPoolSize());
		builder.appendln(" |- MaximumPoolSize:     " + SYSTEM_SCHEDULER.getMaximumPoolSize());
		builder.appendln(" |- CompletedTasks:      " + SYSTEM_SCHEDULER.getCompletedTaskCount());
		builder.appendln(" |- ScheduledTasks:      " + SYSTEM_SCHEDULER.getQueue().size());
		builder.appendln(" | ----------------");
		builder.appendln(" + SystemExecutor:");
		builder.appendln(" |- ActiveThreads:       " + SYSTEM_EXECUTOR.getActiveCount());
		builder.appendln(" |- CorePoolSize:        " + SYSTEM_EXECUTOR.getCorePoolSize());
		builder.appendln(" |- PoolSize:            " + SYSTEM_EXECUTOR.getPoolSize());
		builder.appendln(" |- MaximumPoolSize:     " + SYSTEM_EXECUTOR.getMaximumPoolSize());
		builder.appendln(" |- CompletedTasks:      " + SYSTEM_EXECUTOR.getCompletedTaskCount());
		builder.appendln(" |- ScheduledTasks:      " + SYSTEM_EXECUTOR.getQueue().size());
		builder.appendln(" | ----------------");
		builder.appendln(" + ForkJoinPool:");
		builder.appendln(" |- ActiveThreads:       " + FORK_JOIN_POOL.getActiveThreadCount());
		builder.appendln(" |- PoolSize:            " + FORK_JOIN_POOL.getPoolSize());
		builder.appendln(" |- QueuedTasks:         " + FORK_JOIN_POOL.getQueuedTaskCount());
		builder.appendln(" |- RunningThreadCount:  " + FORK_JOIN_POOL.getRunningThreadCount());
		builder.appendln(" | ----------------");
		return builder.toString();
	}

	/**
	 * Print thread pool stats to log.
	 */
	public void printStats() {
		log.info("ThreadPoolService statistic:");
		for (String statsLine : getStats().split("\n")) {
			log.info(statsLine);
		}
	}

	@Override
	public @NotNull List<MetricResult> getMetric() {
		final List<MetricResult> metricResults = new ArrayList<>();

		final MetricResult scheduleResult = new MetricResult();
		scheduleResult.setMetricGroupType(MetricGroupType.THREADPOOL);
		scheduleResult.setName("SystemScheduler");
		scheduleResult.setData(SYSTEM_SCHEDULER_METRICS.render().get());
		metricResults.add(scheduleResult);

		final MetricResult systemExecutorResult = new MetricResult();
		systemExecutorResult.setMetricGroupType(MetricGroupType.THREADPOOL);
		systemExecutorResult.setName("SystemExecutor");
		systemExecutorResult.setData(SYSTEM_EXECUTOR_METRICS.render().get());
		metricResults.add(systemExecutorResult);

		final MetricResult virtualExecutorResult = new MetricResult();
		virtualExecutorResult.setMetricGroupType(MetricGroupType.THREADPOOL);
		virtualExecutorResult.setName("VirtualExecutor");
		virtualExecutorResult.setData(VIRTUAL_EXECUTOR_METRICS.render().get());
		metricResults.add(virtualExecutorResult);

		final MetricResult forkJoinResult = new MetricResult();
		forkJoinResult.setMetricGroupType(MetricGroupType.THREADPOOL);
		forkJoinResult.setName("ForkJoinPool");
		forkJoinResult.setData(FORK_JOIN_POOL_METRICS.render().get());
		metricResults.add(forkJoinResult);
		return metricResults;
	}

	/**
	 * Tries to remove from the work queue all
	 * {@link Future} tasks that have been cancelled using {@link #SYSTEM_SCHEDULER} and {@link #SYSTEM_EXECUTOR}.
	 * This method can be useful as a storage reclamation operation, that has no
	 * other impact on functionality. Cancelled tasks are never executed, but
	 * may accumulate in work queues until worker threads can actively remove
	 * them. Invoking this method instead tries to remove them now. However,
	 * this method may fail to remove tasks in the presence of interference by
	 * other threads.
	 */
	@SuppressWarnings("unused")
	@Scheduled(period = 10, timeUnit = TimeUnit.MINUTES, runAfterServerStart = true)
	public void purge() {
		SYSTEM_SCHEDULER.purge();
		SYSTEM_EXECUTOR.purge();
		printStats();
	}

	/**
	 * @return {@code true} if thread pool's in shutdown mode, {@code false} otherwise
	 */
	@SuppressWarnings("BooleanMethodIsAlwaysInverted")
	public boolean isShutdown() {
		return _shutdown.get();
	}

	@Override
	public void onShutdown() {
		if (_shutdown.compareAndSet(false, true)) {
			try {
				SYSTEM_SCHEDULER.shutdown();
				SYSTEM_EXECUTOR.shutdown();
				VIRTUAL_EXECUTOR.shutdown();
				FORK_JOIN_POOL.shutdown();

				boolean terminated;

				terminated = SYSTEM_SCHEDULER.awaitTermination(10, TimeUnit.SECONDS);
				logTerminationResult(SYSTEM_SCHEDULER, terminated);

				terminated = SYSTEM_EXECUTOR.awaitTermination(10, TimeUnit.SECONDS);
				logTerminationResult(SYSTEM_EXECUTOR, terminated);

				terminated = VIRTUAL_EXECUTOR.awaitTermination(10, TimeUnit.SECONDS);
				logTerminationResult(VIRTUAL_EXECUTOR, terminated);

				terminated = FORK_JOIN_POOL.awaitTermination(10, TimeUnit.SECONDS);
				logTerminationResult(FORK_JOIN_POOL, terminated);

				log.info("All ThreadPools are now stopped.");

			} catch (InterruptedException e) {
				log.error("Error while shutdown()", e);
				Thread.currentThread().interrupt();
			}
		}
	}

	private void logTerminationResult(ExecutorService executor, boolean terminated) {
		if (terminated) {
			log.info("{} terminated successfully.", executor);
		} else {
			log.warn("{} did not terminate within the timeout.", executor);
		}
	}

	/**
	 * A Future that represents a rejected task. It returns immediately the default value on get() and indicates that it is not done and can be cancelled.
	 *
	 * @param <V> the type of the default value
	 */
	private record RejectedFuture<V>(V defaultValue, Throwable cause) implements ScheduledFuture<V> {
		@Override
		public boolean cancel(boolean mayInterruptIfRunning) {
			return true;
		}

		@Override
		public boolean isCancelled() {
			return true;
		}

		@Override
		public boolean isDone() {
			return false;
		}

		@Override
		public V get() {
			if (cause != null) {
				log.warn("Try get from RejectedFuture. Return default value. Cause:", cause);
			} else {
				log.warn("Try get from RejectedFuture. Return default value.");
			}
			return defaultValue;
		}

		@Override
		public V get(long timeout, @NotNull TimeUnit unit) {
			if (cause != null) {
				log.warn("Try get with timeout from RejectedFuture. Return default value immediately. Cause:", cause);
			} else {
				log.warn("Try get with timeout from RejectedFuture. Return default value immediately.");
			}
			return defaultValue;
		}

		@Override
		public long getDelay(@NotNull TimeUnit unit) {
			return 0;
		}

		@Override
		public int compareTo(@NotNull Delayed o) {
			return 0;
		}
	}
}