package host.anzo.commons.collection;

import org.jetbrains.annotations.NotNull;

import java.util.AbstractSet;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * A thread-safe implementation of a Set using a ConcurrentHashMap.
 * This class allows for concurrent access and modification of the set
 * without the need for external synchronization.
 *
 * @param <E> the type of elements in this set
 * @author ANZO
 * @since 9/20/2021
 */
public class ConcurrentHashSet<E> extends AbstractSet<E> implements Set<E> {
    private final ConcurrentMap<E, Object> map;

    private static final Object EMPTY_OBJECT = new Object();

    /**
     * Constructs a new, empty ConcurrentHashSet.
     */
    public ConcurrentHashSet(){
        map = new ConcurrentHashMap<>();
    }

    /**
     * Returns the number of elements in this set.
     *
     * @return the number of elements in this set
     */
    @Override
    public int size() {
        return map.size();
    }

    /**
     * Returns an iterator over the elements in this set.
     *
     * @return an iterator over the elements in this set
     */
    @Override
    public @NotNull Iterator<E> iterator(){
        return map.keySet().iterator();
    }

    /**
     * Returns true if this set contains no elements.
     *
     * @return true if this set is empty, false otherwise
     */
    @Override
    public boolean isEmpty(){
        return map.isEmpty();
    }

    /**
     * Adds the specified element to this set if it is not already present.
     *
     * @param o element to be added to this set
     * @return true if this set did not already contain the specified element
     */
    @Override
    public boolean add(final E o){
        return map.put(o, ConcurrentHashSet.EMPTY_OBJECT) == null;
    }

    /**
     * Returns true if this set contains the specified element.
     *
     * @param o element whose presence in this set is to be tested
     * @return true if this set contains the specified element, false otherwise
     */
    @Override
    public boolean contains(final Object o){
        return map.containsKey(o);
    }

    /**
     * Removes all the elements from this set.
     */
    @Override
    public void clear(){
        map.clear();
    }

    /**
     * Removes the specified element from this set if it is present.
     *
     * @param o element to be removed from this set, if present
     * @return true if this set contained the specified element
     */
    @Override
    public boolean remove(final Object o){
        return map.remove(o) == ConcurrentHashSet.EMPTY_OBJECT;
    }

    /**
     * Adds the specified element to this set if it is not already present.
     * This method is equivalent to {@link #add} method, but may be more
     * efficient in a concurrent context.
     *
     * @param o element to be added to this set
     * @return true if this set did not already contain the specified element
     */
    public boolean addIfAbsent(final E o){
        final Object obj = map.putIfAbsent(o, ConcurrentHashSet.EMPTY_OBJECT);
        return obj == null;
    }
}